package com.els.base.notice.web.controller;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.notice.command.BatchCancelCommand;
import com.els.base.notice.command.BatchPublishCommand;
import com.els.base.notice.command.ImInvoker;
import com.els.base.notice.command.CreateCommand;
import com.els.base.notice.command.DeleteCommand;
import com.els.base.notice.command.ModifyCommand;
import com.els.base.notice.command.PublishCommand;
import com.els.base.notice.entity.Notice;
import com.els.base.notice.entity.NoticeExample;
import com.els.base.notice.entity.NoticeReceiver;
import com.els.base.notice.entity.NoticeReceiverExample;
import com.els.base.notice.service.NoticeReceiverService;
import com.els.base.notice.service.NoticeService;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

@Api(value = "CMS-公告管理")
@Controller("cmsNoticeController")
@RequestMapping("cmsNotice")
public class NoticeController {
	@Resource
	protected NoticeService noticeService;
	
	@Resource
	protected NoticeReceiverService noticeReceiverService;
	
	@Resource
	protected ImInvoker noticeInvoker;
	
	@ApiOperation(httpMethod = "POST", value = "创建公告")
	@RequestMapping("service/create")
	@ResponseBody
	public ResponseResult<String> create(@RequestBody Notice notice) {
		
		Assert.isNotNull(notice, "公告不能为空");
		notice.setCreateUserId(SpringSecurityUtils.getLoginUserId());
		notice.setCreateUserName(SpringSecurityUtils.getLoginUser().getNickName());
		
		CreateCommand createComand = new CreateCommand(notice);
		this.noticeInvoker.invoke(createComand);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod = "POST", value = "编辑公告")
	@RequestMapping("service/edit")
	@ResponseBody
	public ResponseResult<String> edit(@RequestBody Notice notice) {
		Assert.isNotBlank(notice.getId(), "id 为空，保存失败");
		
		ModifyCommand modifyCommand = new ModifyCommand(notice);
		this.noticeInvoker.invoke(modifyCommand);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod = "POST", value = "删除公告")
	@RequestMapping("service/deleteBatch")
	@ResponseBody
	public ResponseResult<String> deleteBatch(@RequestBody List<String> ids) {
		Assert.isNotEmpty(ids, "删除失败,id不能为空");
		
		DeleteCommand deleteCommand = new DeleteCommand(ids);
		this.noticeInvoker.invoke(deleteCommand);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod = "POST", value = "公告展示")
	@ApiImplicitParams({
		@ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
		@ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10")})
	@RequestMapping("service/show")
	@ResponseBody
	public ResponseResult<PageView<Notice>> show(@RequestParam(defaultValue = "0") int pageNo,
			@RequestParam(defaultValue = "10") int pageSize) {
		NoticeExample example = new NoticeExample();
		example.setPageView(new PageView<Notice>(pageNo, pageSize));
		example.setOrderByClause("PUBLISH_TIME DESC");

		NoticeExample.Criteria criteria = example.createCriteria();
		criteria.andStatusEqualTo(Constant.YES_INT);
		
		//是否供应商
		boolean isSupCompany = !CompanyUtils.currentCompanyId().equals(ProjectUtils.getProject().getCompanyId());
		if (isSupCompany) {
			criteria.andIsAllReceivedEqualTo(Constant.YES_INT);
			this.orAsReceiver(example);
		}
		

		PageView<Notice> pageData = this.noticeService.queryObjByPage(example);
		return ResponseResult.success(pageData);
	}
	
	/**
	 * 添加作为指定供应商能看的公告
	 * @param example
	 */
	private void orAsReceiver(NoticeExample example) {
		NoticeReceiverExample noticeReceiverExample = new NoticeReceiverExample();
		noticeReceiverExample.createCriteria().andReceiverCompanyIdEqualTo(CompanyUtils.currentCompanyId());
		
		List<NoticeReceiver> list = this.noticeReceiverService.queryAllObjByExample(noticeReceiverExample);
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		
		List<String> noticeIds = new ArrayList<>(list.size());
		for(NoticeReceiver receiver : list){
			noticeIds.add(receiver.getNoticeId());
		}
		
		example.or().andIdIn(noticeIds);
	}
	
	@ApiOperation(httpMethod = "POST", value = "采购商-公告列表查询")
	@ApiImplicitParams({
			@ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
			@ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
			@ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 Notice", paramType = "body", dataType = "QueryParamWapper") })
	@RequestMapping("service/findByPage")
	@ResponseBody
	public ResponseResult<PageView<Notice>> findByPage(@RequestParam(defaultValue = "0") int pageNo,
			@RequestParam(defaultValue = "10") int pageSize, @RequestBody(required = false) QueryParamWapper wapper) {
		NoticeExample example = new NoticeExample();
		example.setPageView(new PageView<Notice>(pageNo, pageSize));
		example.setOrderByClause("CREATE_TIME DESC");

		NoticeExample.Criteria criteria = example.createCriteria();
		if (wapper != null) {
			CriteriaUtils.addCriterion(criteria, wapper);
		}

		PageView<Notice> pageData = this.noticeService.queryObjByPage(example);
		return ResponseResult.success(pageData);
	}
	
	@ApiOperation(httpMethod = "POST", value = "采购商-发布公告")
	@RequestMapping("service/publish")
	@ResponseBody
	public ResponseResult<String> publish(@RequestBody Notice notice) {
		
		notice.setPublishUserId(SpringSecurityUtils.getLoginUserId());
		notice.setPublishUserName(SpringSecurityUtils.getLoginUser().getNickName());
		
		PublishCommand publishCommand = new PublishCommand(notice);
		this.noticeInvoker.invoke(publishCommand);
		
		return ResponseResult.success();
	}
	
	@ApiOperation(httpMethod = "POST", value = "采购商-批量发布公告")
	@RequestMapping("service/batchPublish")
	@ResponseBody
	public ResponseResult<String> batchPublish(@RequestBody List<String> noticeIdList) {
		Assert.isNotEmpty(noticeIdList, "发布的公告不能为空");
		
		BatchPublishCommand publishCommand = new BatchPublishCommand(noticeIdList, SpringSecurityUtils.getLoginUser());
		this.noticeInvoker.invoke(publishCommand);
		
		return ResponseResult.success();
	}
	
	@ApiOperation(httpMethod = "POST", value = "采购商-批量作废公告")
	@RequestMapping("service/batchCancel")
	@ResponseBody
	public ResponseResult<String> batchCancel(@RequestBody List<String> noticeIdList) {
		Assert.isNotEmpty(noticeIdList, "撤销的公告不能为空");
		
		BatchCancelCommand batchCanCelCommand = new BatchCancelCommand(noticeIdList);
		this.noticeInvoker.invoke(batchCanCelCommand);
		
		return ResponseResult.success();
	}
	
}