package com.els.base.msg;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;

/**
 * 消息
 * @author hzy
 *
 * @param <T>
 */
public class Message<T> implements IMessage<T>, Serializable {

	private static final Logger logger = LoggerFactory.getLogger(Message.class);
	private static final long serialVersionUID = 1L;
	
	private String senderId;
	private List<String> receiverIdList;
	private String businessTypeCode;
	private MessageLevelEnum msgLevel;
	private String companyCode;
	private T data;
	
	/**
	 * 消息构造方法
	 * @param sendCompanyCode 发送者 公司srm code
	 * @param senderId 发送者 userId
	 * @param receiverId 接受者 userId
	 * @param businessTypeCode 业务类型
	 * @param msgLevel 消息等级
	 * @param data 消息的数据
	 */
	public Message(String sendCompanyCode, String senderId, String receiverId, String businessTypeCode, MessageLevelEnum msgLevel, T data) {
		this.data = data;
		this.businessTypeCode = businessTypeCode;
		
		this.senderId = senderId;
		this.msgLevel = msgLevel;
		this.companyCode = sendCompanyCode;
		
		List<String> receiverIds = new ArrayList<>();
		receiverIds.add(receiverId);
		this.receiverIdList = receiverIds;
		
	}

	private Message(T data) {
		this.data = data;
	}

	@Override
	public String getSenderId() {
		return senderId;
	}

	public Message<T> setSenderId(String senderId) {
		this.senderId = senderId;
		return this;
	}

	@Override
	public T getData() {
		return data;
	}

	public void setData(T data) {
		this.data = data;
	}

	@Override
	public List<String> getReceiverIdList() {
		return receiverIdList;
	}

	/**
	 * 添加多个收件人,多个userId
	 * @param receiverIdList
	 * @return
	 */
	public Message<T> addReceiverIdList(List<String> receiverIdList) {
		if (this.receiverIdList == null) {
			this.receiverIdList = new ArrayList<>();
		}
		this.receiverIdList.addAll(receiverIdList);
		return this;
	}
	
	/**
	 * 添加收件人，userId
	 * @param receiverId
	 * @return
	 */
	public Message<T> addReceiverId(String receiverId) {
		if (this.receiverIdList == null) {
			this.receiverIdList = new ArrayList<>();
		}
		this.receiverIdList.add(receiverId);
		return this;
	}

	@Override
	public MessageLevelEnum getMsgLevel() {
		return msgLevel;
	}

	public Message<T> setMsgLevel(MessageLevelEnum msgLevel) {
		this.msgLevel = msgLevel;
		return this;
	}
	
	public String getCompanyCode() {
		return companyCode;
	}

	public Message<T> setCompanyCode(String companyCode) {
		this.companyCode = companyCode;
		return this;
	}
	
	public String getBusinessTypeCode() {
		return businessTypeCode;
	}

	public Message<T> setBusinessTypeCode(String businessTypeCode) {
		this.businessTypeCode = businessTypeCode;
		return this;
	}

	public void checkData(){
		try {
			
			Assert.isNotBlank(this.getCompanyCode(), "companyCode不能为空");
			Assert.isNotNull(data, "消息数据不能为空");
			Assert.isNotEmpty(receiverIdList, "消息接受人不能为空");
//			
		} catch (Exception e) {
			logger.error("消息初始化失败", e);
			throw new CommonException(e);
		}
	}
	
	public static <T> Message<T> init(T data){
		Message<T> message = new Message<T>(data);
		return message;
	}
}
