package com.els.base.inquiry.command.pur;

import static java.util.stream.Collectors.toList;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.inquiry.AbstractInquiryCommand;
import com.els.base.inquiry.IOrderItem;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.entity.InquiryQuoteLadder;
import com.els.base.inquiry.entity.InquiryQuoteLadderExample;
import com.els.base.inquiry.entity.InquirySupOrder;
import com.els.base.inquiry.entity.InquirySupOrderExample;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryLadderBelongType;
import com.els.base.inquiry.enumclass.InquiryQuoteLadderType;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;
import com.els.base.inquiry.enumclass.MessageNotify;
import com.els.base.inquiry.enumclass.OperationTypeEnum;
import com.els.base.inquiry.web.vo.ReQuoteVo;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;

/**
 * 采购方执行重报价操作
 * @author wuwenxin
 *
 */
public class BatchReQuoteCommand extends AbstractInquiryCommand<String> {
	
	private ReQuoteVo vo;
	
	public BatchReQuoteCommand(ReQuoteVo vo) {
		this.vo = vo;
	}

	@SuppressWarnings("unchecked")
	@Override
	public String execute(InquiryCommandInvoker invoker) {
		Assert.isNotNull(vo);
		Assert.isNotEmpty(vo.getIds(), "询价单行不能为空");
		Assert.isNotBlank(vo.getPurOrderId(), "询价单不能为空");
		
		PurOrder purOrder = invoker.getPurOrderService().queryObjById(vo.getPurOrderId());
		TemplateConf templateConf = invoker.getTemplateConfService().queryObjById(purOrder.getTemplateId());
		
		List<IOrderItem> orderItemList = this.vo.getIds()
			.parallelStream()
			.distinct()
			.map(orderItemId->{
				return (IOrderItem) templateConf.getOrderItemService().queryObjById(orderItemId);
			})
			.collect(toList());
		
		//1、数据校验
		orderItemList.forEach(orderItem -> valid(orderItem));
		
		// 2、修改供应商询价单行状态
		orderItemList.stream().forEach(orderItem->{
			this.reset(orderItem, templateConf);
		});
		
		// 3、修改供应商询价单头状态
		List<String> supOrderIdList = orderItemList.stream()
			.map(IOrderItem::getSupOrderId)
			.distinct()
			.collect(toList());
		
		InquirySupOrderExample supOrderExample = new InquirySupOrderExample();
		supOrderExample.createCriteria().andIdIn(supOrderIdList);
		
		InquirySupOrder temp = new InquirySupOrder();
		temp.setQuoteStatus(InquiryQuoteStatus.REQUOTE.getCode());
		this.invoker.getInquirySupOrderService().modifyByExample(temp, supOrderExample);
		
		// 4、将采购方询价单头改成重报价
		PurOrder tempPurOrder = new PurOrder();
		tempPurOrder.setId(vo.getPurOrderId());
		tempPurOrder.setQuoteStatus(InquiryQuoteStatus.REQUOTE.getCode());
		
		this.invoker.getPurOrderService().modifyObj(tempPurOrder);
		
		// 5、发送通知消息
		supOrderIdList.parallelStream().forEach(supordrId->{
			InquirySupOrder supOrder = invoker.getInquirySupOrderService().queryObjById(supordrId);
			this.sendMsg(supordrId, supOrder.getOrderNo(), supOrder.getSupUserId());
		});
		
		return null;
	}

	private void reset(IOrderItem orderItem, TemplateConf templateConf) {
		try {
			IOrderItem temp = templateConf.getOrderItemClass().newInstance();
			temp.setId(orderItem.getId());
			temp.setQuotationStatus(InquiryQuoteStatus.REQUOTE.getCode());
			temp.setOperation(OperationTypeEnum.NO_OPERATION.getCode());
			temp.setUntaxedUnitPrice(BigDecimal.ZERO);
			templateConf.getOrderItemService().modifyStatus(temp);
			
			if (!InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(orderItem.getQuoteType())) {
				return;
			}

			InquiryQuoteLadderExample example = new InquiryQuoteLadderExample();
			example.createCriteria()
				.andItemIdEqualTo(orderItem.getId())
				.andBelongTypeEqualTo(InquiryLadderBelongType.BELONG_TO_ITEM.getCode());
			
			InquiryQuoteLadder tempLadder = new InquiryQuoteLadder();
			tempLadder.setUntaxedUnitPrice(BigDecimal.ZERO);
			this.invoker.getInquiryQuoteLadderService().modifyByExample(tempLadder, example);
			
		} catch (InstantiationException | IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}

	private void sendMsg(String supOrderId, String orderNo, String supUserId) {
		Map<String, String> data = new HashMap<>();
		data.put("inquirySupOrderId", supOrderId);
		data.put("orderNo", orderNo);
		Message<Map<String, String>> msg = Message.init(data).
				setCompanyCode(getPurCompany().getCompanyCode()).
				setBusinessTypeCode(MessageNotify.INQUIRY_ORDER_REQUOTE.getCode()).
				setSenderId(getPurUser().getId()).
				addReceiverId(supUserId).
				setMsgLevel(MessageLevelEnum.HIGH);
				MessageSendUtils.sendMessage(msg);
	}

	private void valid(IOrderItem orderItem) {
		if (!this.vo.getPurOrderId().equals(orderItem.getPurOrderId())) {
			throw new CommonException("只能选择一个重报价的询价单进行重报价");
		}
		
		// 只有处于已报价或已重报状态下的询价单行才能执行重报价操作
		if (!InquiryQuoteStatus.QUOTED.getCode().equals(orderItem.getQuotationStatus())
				&& !InquiryQuoteStatus.REQUOTED.getCode().equals(orderItem.getQuotationStatus())) {
			throw new CommonException("只有处于已报价或已重报状态下的行才能执行重报价操作");
		}
	}
	
}
