package com.els.base.inquiry.command.pur;

import static java.util.stream.Collectors.toList;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.AbstractInquiryCommand;
import com.els.base.inquiry.IMould;
import com.els.base.inquiry.ITarget;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.entity.InquiryBusiCondition;
import com.els.base.inquiry.entity.InquiryPurFile;
import com.els.base.inquiry.entity.InquiryQuoteLadder;
import com.els.base.inquiry.entity.InquiryWaitMaterial;
import com.els.base.inquiry.entity.InquiryWaitMaterialExample;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryFileType;
import com.els.base.inquiry.enumclass.InquiryLadderBelongType;
import com.els.base.inquiry.enumclass.InquiryOrderStatus;
import com.els.base.inquiry.enumclass.InquiryQuoteLadderType;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;
import com.els.base.inquiry.utils.LaddPriceValider;
import com.els.base.inquiry.utils.PropertyValueUtils;

public class CreateCommand extends AbstractInquiryCommand<String>{

	private PurOrder order;
	private TemplateConf templateConf;

	public CreateCommand(PurOrder inquiryPurOrder) {
		this.order = inquiryPurOrder;
	}

	@Override
	public String execute(InquiryCommandInvoker invoker) {
		templateConf = invoker.getTemplateConfService().queryObjById(order.getTemplateId());
		
		this.valid(this.order);
		
		this.init(order);
		
		//1、保存头
		invoker.getPurOrderService().addObj(order);
		
		//2、保存物料或者其他的询价目标
		this.saveTarget(order);
		
		//3、插入物料清单的阶梯报价
		this.saveLadderPrice(order.getTargetList());
		
		//4、修改物料清单中关联的带询价清单
		this.setWaitMaterialStatus(order);
		
		//5、保存采购方表单文件
		this.savePurFile(order);
		
		//6、保存供应商
		invoker.getInquirySupplierService().addAll(order.getId(), order.getInquirySuppliers());
		
		//7、保存商务条件
		this.saveBusiCondition(order);
		
		
		//5、保存模具清单
		if (Constant.YES_INT.equals(templateConf.getIsMouldEnable())) {
			this.addMould(order);
		}
		
		return order.getId();
	}
	
	private void saveBusiCondition(PurOrder purOrder) {
		if (Constant.NO_INT.equals(this.templateConf.getIsBusiCondEnable())) {
			return;
		}
		List<InquiryBusiCondition> busiConditionList = purOrder.getBusiConditions();
		busiConditionList.forEach(busiCondition -> {
			busiCondition.setPurOrderId(purOrder.getId());
			busiCondition.setTemplateConfId(purOrder.getTemplateId());
			busiCondition.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(busiCondition));
		});
		/*List<InquiryBusiCondition> busiConditionList = purOrder.getInquirySuppliers().stream().map(supplier->{
			InquiryBusiCondition inquiryBusiCondition = new InquiryBusiCondition();
			inquiryBusiCondition.setPurOrderId(purOrder.getId());
			inquiryBusiCondition.setSupCompanyId(supplier.getSupCompanyId());
			inquiryBusiCondition.setSupCompanyName(supplier.getSupCompanyName());
			inquiryBusiCondition.setSupCompanySrmCode(supplier.getSupCompanySrmCode());
			inquiryBusiCondition.setSupCompanySapCode(supplier.getSupCompanySapCode());
			inquiryBusiCondition.setTemplateConfId(purOrder.getTemplateId());
			inquiryBusiCondition.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(inquiryBusiCondition));
			return inquiryBusiCondition;
		}).collect(toList());*/
		
		invoker.getInquiryBusiConditionService().addAll(busiConditionList);
	}
	
	public void savePurFile(PurOrder order) {
		if (CollectionUtils.isEmpty(order.getPurFileList())) {
			return;
		}
		
		for (InquiryPurFile inquiryPurFile : order.getPurFileList()) {
			inquiryPurFile.setPurOrderId(order.getId());
			
			// 已规定前端上传的询价单行号为-10，则说明是公共附件
			if ("-10".equals(inquiryPurFile.getPurOrderItemNo().trim())) {
				inquiryPurFile.setType(InquiryFileType.PUBLIC.getCode());
			} else {
				inquiryPurFile.setType(InquiryFileType.PRIVATE.getCode());
			}
			
			invoker.getInquiryPurFileService().addObj(inquiryPurFile);
		}
		
	}

	@SuppressWarnings("unchecked")
	protected void saveTarget(PurOrder order) {
		if (CollectionUtils.isEmpty(order.getTargetList())) {
			return; 
		}
		
		order.getTargetList().forEach(target->{
			target.setPurOrderId(order.getId());
			target.setOrderNo(order.getOrderNo());
			target.setTemplateId(templateConf.getId());
			target.setUnableToQuote(null);
			target.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(target));
		});
		
		templateConf.getTargetService().addAll(order.getTargetList());
	}

	@SuppressWarnings("unchecked")
	protected void addMould(PurOrder order) {
		if (CollectionUtils.isEmpty(order.getMouldList())) {
			return;
		}
		
		for (int i = 0; i < order.getMouldList().size(); i++) {
			IMould mould = order.getMouldList().get(i);
			mould.setPurOrderId(order.getId());
			mould.setSortNo(i+1);
			mould.setTemplateId(templateConf.getId());
			mould.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(mould));
		}
		
		templateConf.getMouldService().addAll(order.getMouldList());
	}

	protected void init(PurOrder order) {
		//生成询价单头信息
		this.generatePurOrder(order);
		
		this.generateTarget(order);
		
	}

	private void generateTarget(PurOrder order) {
		
		order.getTargetList().sort((t1, t2)->{
			if (StringUtils.isBlank(t1.getMaterialCode())) {
				return -1;
			}
			if (StringUtils.isBlank(t2.getMaterialCode())) {
				return 1;
			}
			
			return t1.getMaterialCode().compareTo(t2.getMaterialCode());
		});
		
		for (int i=0; i<order.getTargetList().size(); i++) {
			ITarget target = order.getTargetList().get(i);
			target.setSortNo(i + 1);
			
			target.setTemplateId(order.getTemplateId());
			target.setOrderNo(order.getOrderNo());
		}
	}

	protected void valid(PurOrder order) {
		Assert.isNotNull(order, "询价单头不能为空");
		
		Assert.isNotBlank(order.getOrderNo(), "询价单号不能为空");
		Assert.isNotBlank(order.getTemplateId(), "询价单的模板id不能为空");
		Assert.isNotEmpty(order.getTargetList(), "询价单的物料清单不能为空");
		Assert.isNotEmpty(order.getInquirySuppliers(), "询价单的供应商列表不能为空");
		Assert.isNotEmpty(order.getBusiConditions(), "询价单的商务条件列表不能为空");
		
		//校验表单文件
		if (CollectionUtils.isNotEmpty(order.getPurFileList())) {
			order.getPurFileList().forEach(purFile->{
				Assert.isNotNull(purFile.getPurOrderItemNo(), "表单文件，行项目不能为空");
			});
		}
		
		//校验供应商
		if (CollectionUtils.isNotEmpty(order.getInquirySuppliers())) {
			order.getInquirySuppliers().forEach(supCompany->{
				Assert.isNotBlank(supCompany.getSupCompanyId(), "供应商id不能为空");
				Assert.isNotBlank(supCompany.getSupCompanyName(), "供应商名称不能为空");
				Assert.isNotBlank(supCompany.getSupCompanySapCode(), "供应商SAP编码不能为空");
				Assert.isNotBlank(supCompany.getSupCompanySrmCode(), "供应商SRM编码不能为空");
			});
		}
		
		//校验商务条件
		order.getBusiConditions().forEach(busiCondition -> {
			Assert.isNotBlank(busiCondition.getPayMethod(), "支付方式不能为空");
			Assert.isNotBlank(busiCondition.getHandOverAddress(), "交付地点不能为空");
			Assert.isNotBlank(busiCondition.getTaxCode(), "税码不能为空");
		});
		
		//校验物料清单
		for (int i=0; CollectionUtils.isNotEmpty(order.getTargetList()) && i< order.getTargetList().size(); i++) {
			ITarget target = order.getTargetList().get(i);
//			Assert.isNotBlank(target.getMaterialCode(), "物料编码不能为空");
			//Assert.isNotBlank(target.getMaterialDesc(), "物料描述不能为空");
			Assert.isNotBlank(target.getInfoCategory(), "信息类别不能为空");
			Assert.isNotBlank(target.getMaterialCode(), "物料编码不能为空");
			Assert.isNotBlank(target.getMapNo(), "图号不能为空");
			Assert.isNotBlank(target.getName(), "名称不能为空");
			Assert.isNotNull(target.getQuantity(), "数量不能为空");
			Assert.isNotNull(target.getRecentUntaxedUnitPrice(), "最近一次采购不含税单价不能为空");
			Assert.isNotBlank(target.getDecreasingAmplitude(), "降幅不能为空");
			Assert.isNotBlank(target.getPurchaseApply(), "采购申请单号不能为空");
			Assert.isNotBlank(target.getProject(), "行项目号不能为空");
			Assert.isNotBlank(target.getPurchaseGroup(), "采购组不能为空");
			Assert.isNotNull(target.getPriceUnit(), "价格单位不能为空");
			Assert.isNotNull(target.getValidDateFrom(), "有效期从不能为空");
			Assert.isNotNull(target.getValidDateTo(), "有效期到不能为空");
			
			if (StringUtils.isNotBlank(target.getMaterialCode())
					&& !invoker.getMaterialService().isExists(target.getMaterialCode()) ) {
				throw new CommonException(String.format("物料编码[%s],系统中不存在!", target.getMaterialCode()));
			}

			if (!Constant.YES_INT.equals(templateConf.getIsOrderItemDetailEnable())
					&& !InquiryQuoteLadderType.CONVENTIONAL_QUOTE.getCode().equals(target.getQuoteType())
					&& !InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(target.getQuoteType()) ) {
				throw new CommonException(String.format("报价类型不能为空,请检查物料编码[%s]", target.getMaterialCode()));
			}
			
			if (StringUtils.isBlank(target.getWaitMaterialId())) {
				continue;
			}
			
			InquiryWaitMaterial temp = invoker.getInquiryWaitMaterialService().queryObjById(target.getWaitMaterialId());
			if (temp != null && Constant.YES_INT.equals(temp.getIsCreateInquiryOrder())) {
				throw new CommonException("待询价物料已经创建了询价单，请不要重复创建");
			}
			
			LaddPriceValider.valid(target.getInquiryQuoteLadders());

		}
		
		//4、校验物料清单中的阶梯报价
		order.getTargetList().stream()
		.filter(target -> InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(target.getQuoteType()))
		.forEach(target->{
			
			if (CollectionUtils.isEmpty(target.getInquiryQuoteLadders())) {
				throw new CommonException(String.format("物料编码[%s] 选择了报价方式为“阶梯报价”，但未设置报价阶梯！", target.getMaterialCode()));
			}
			
			target.getInquiryQuoteLadders().forEach(quoteLadder->{
				
//						Assert.isNotNull(quoteLadder.getMaterialCode(), "阶梯报价缺少物料编码");
				//Assert.isNotNull(quoteLadder.getMaterialDesc(), "阶梯报价缺少物料描述");
				Assert.isNotNull(quoteLadder.getNumberFrom(), String.format("物料[%s]阶梯报价缺少\"数量从\"", quoteLadder.getMaterialDesc()));
				Assert.isNotNull(quoteLadder.getNumberTo(), String.format("物料[%s]阶梯报价缺少\"数量到\"", quoteLadder.getMaterialDesc()));
				
				if (quoteLadder.getNumberTo().compareTo(quoteLadder.getNumberFrom()) < 0) {
					throw new CommonException(String.format("物料[%s]阶梯报价异常， \"数量从\"大于\"数量到\"", quoteLadder.getMaterialDesc()));
				}
			});
		});
		
	}

	/**
	 * 构造询价单头信息，包括询价单ID，创建人ID，公司等信息
	 * @param order
	 */
	private void generatePurOrder(PurOrder order) {
		if (StringUtils.isBlank(order.getPurCompanyId())) {
			order.setPurCompanyId(getPurCompany().getId());	//创建者公司ID
		}
		if (StringUtils.isBlank(order.getPurCompanyName())) {
			order.setPurCompanyName(getPurCompany().getCompanyName());	//创建者公司名称
		}
		if (StringUtils.isBlank(order.getPurUserId())) {
			order.setPurUserId(getPurUser().getId());	//创建者ID
		}
		if (StringUtils.isBlank(order.getPurUserName())) {
			order.setPurUserName(getPurUser().getLoginName());	//创建者名称
		}
		if (order.getCreateDate() == null) {
			order.setCreateDate(new Date());
		}
		if (order.getInquiryOrderStatus() == null) {
			order.setInquiryOrderStatus(InquiryOrderStatus.UNPUBLISHED.getCode());
		}
		if (order.getQuoteStatus() == null) {
			order.setQuoteStatus(InquiryQuoteStatus.UNQUOTED.getCode());
		}
	}
	
	protected void setWaitMaterialStatus(PurOrder order){
		if (CollectionUtils.isEmpty(order.getTargetList())) {
			return;
		}
		
		List<String> waitMaterialIdList = order.getTargetList().stream()
				.filter(target-> StringUtils.isNotBlank(target.getWaitMaterialId()))
				.map(ITarget::getWaitMaterialId).collect(Collectors.toList());
		
		if (CollectionUtils.isNotEmpty(waitMaterialIdList)) {
			InquiryWaitMaterialExample example = new InquiryWaitMaterialExample();
			example.createCriteria().andIdIn(waitMaterialIdList);
			InquiryWaitMaterial tmp = new InquiryWaitMaterial();
			tmp.setIsCreateInquiryOrder(Constant.YES_INT);
			tmp.setInquiryOrderNo(order.getOrderNo());
			invoker.getInquiryWaitMaterialService().modifyByExample(tmp, example);
		}
	}
	
	/**
	 * 批量插入阶梯报价
	 * @param list
	 */
	protected void saveLadderPrice(List<ITarget> list){
		if (CollectionUtils.isEmpty(list)){
			return;
		}
		
		list.forEach(target->{
			Assert.isNotBlank(target.getId(), "targetId不能为空");
		});
		
		List<InquiryQuoteLadder> totalinquiryQuoteLadder = list.stream()
		
		.filter(target->{
			
			//过滤出 有阶梯报价的物料
			return InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(target.getQuoteType())
					&& CollectionUtils.isNotEmpty(target.getInquiryQuoteLadders()); 
			
		}).flatMap(target->{
			
			//把物料中的阶梯报价列出来
			List<InquiryQuoteLadder> inquiryQuoteLadders =  target.getInquiryQuoteLadders();
			inquiryQuoteLadders.forEach(inquiryQuoteLadder->{
				inquiryQuoteLadder.setMaterialCode(target.getMaterialCode());
				//inquiryQuoteLadder.setMaterialDesc(target.getMaterialDesc());
				inquiryQuoteLadder.setBelongType(InquiryLadderBelongType.BELONG_TO_TARGET.getCode());
				inquiryQuoteLadder.setTargetId(target.getId());
			});
			return inquiryQuoteLadders.stream();
			
		}).collect(toList());
		
		//批量插入阶梯报价
		this.invoker.getInquiryQuoteLadderService().addAll(totalinquiryQuoteLadder);
	}
	
}
