package com.els.base.inquiry.command.pur;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.AbstractInquiryCommand;
import com.els.base.inquiry.ITarget;
import com.els.base.inquiry.ITargetService;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.entity.InquiryBusiConditionExample;
import com.els.base.inquiry.entity.InquiryPurFileExample;
import com.els.base.inquiry.entity.InquiryQuoteLadderExample;
import com.els.base.inquiry.entity.InquirySupplierExample;
import com.els.base.inquiry.entity.InquiryWaitMaterial;
import com.els.base.inquiry.entity.InquiryWaitMaterialExample;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryLadderBelongType;
import com.els.base.inquiry.enumclass.InquiryOrderStatus;

/**
 * 采购方删除操作，只做逻辑删除，不做物理删除
 * @author wuwenxin
 *
 */
public class DeleteCommand extends AbstractInquiryCommand<String> {
	
	private String id;

	public DeleteCommand(String id) {
		this.id = id;
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public String execute(InquiryCommandInvoker invoker) {
		// 校验所选的询价单是否符合删除的条件
		Assert.isNotBlank(id, "数据不能为空");
		PurOrder purOrder = this.invoker.getPurOrderService().queryObjById(id);
		TemplateConf templateConf = invoker.getTemplateConfService().queryObjById(purOrder.getTemplateId());
		this.valid(purOrder);
		
		//1、删除询价单头
		invoker.getPurOrderService().deleteObjById(id);
		
		//2、删除询价单物料行
		ITargetService targetService = templateConf.getTargetService();
		List<ITarget> targetList = targetService.queryByPurOrderId(id);
		
		targetService.deleteObjByPurOrderId(id);
		
		//3、删除物料行关联的阶梯报价
		this.deleteLadderPrice(targetList);
		
		//4、修改带询价物料清单的状态
		this.clearWaitMaterial(targetList);
		
		//5、删除表单文件
		InquiryPurFileExample example = new InquiryPurFileExample();
		example.createCriteria().andPurOrderIdEqualTo(id);
		this.invoker.getInquiryPurFileService().deleteByExample(example);	
		
		//6、删除供应商
		InquirySupplierExample inquirySupplierExample = new InquirySupplierExample();
		inquirySupplierExample.createCriteria().andPurOrderIdEqualTo(id);
		this.invoker.getInquirySupplierService().deleteByExample(inquirySupplierExample);
		
		InquiryBusiConditionExample busiConditionExample = new InquiryBusiConditionExample();
		busiConditionExample.createCriteria().andPurOrderIdEqualTo(id);
		this.invoker.getInquiryBusiConditionService().deleteByExample(busiConditionExample);
		
		//7、删除模具清单
		templateConf.getMouldService().deleteByPurOrder(id);
		return null;
	}

	private void clearWaitMaterial(List<ITarget> targetList) {
		if (CollectionUtils.isEmpty(targetList)) {
			return;
		}
		
		List<String> waitMaterialIdList = targetList.stream()
				.filter(target-> StringUtils.isNotBlank(target.getWaitMaterialId()))
				.map(ITarget::getWaitMaterialId).collect(Collectors.toList());
		
		if (CollectionUtils.isEmpty(waitMaterialIdList)) {
			return;
		}

		InquiryWaitMaterialExample example = new InquiryWaitMaterialExample();
		example.createCriteria().andIdIn(waitMaterialIdList);
		InquiryWaitMaterial tmp = new InquiryWaitMaterial();
		tmp.setIsCreateInquiryOrder(Constant.NO_INT);
		tmp.setInquiryOrderNo("");
		invoker.getInquiryWaitMaterialService().modifyByExample(tmp, example);
	}

	private void deleteLadderPrice(List<ITarget> targetList) {
		if(CollectionUtils.isEmpty(targetList)) {
			return;
		}
		
		List<String> targetIdList = targetList.stream().map(ITarget::getId).collect(Collectors.toList());
		
		InquiryQuoteLadderExample example = new InquiryQuoteLadderExample();
		example.createCriteria()
			.andTargetIdIn(targetIdList)
			.andBelongTypeEqualTo(InquiryLadderBelongType.BELONG_TO_TARGET.getCode());
		
		this.invoker.getInquiryQuoteLadderService().deleteByExample(example);
	}

	private void valid(PurOrder purOrder) {
		
		Assert.isNotNull(purOrder, "询价单不存在，请刷新后重试");

		// 只有当询价单为未发布时才能删除
		if ( !InquiryOrderStatus.UNPUBLISHED.getCode().equals(purOrder.getInquiryOrderStatus())) {
			throw new CommonException(String.format("只能删除未发布的询价单，请检查单号为[%s]的询价单是否符合条件", purOrder.getOrderNo()));
		}
	}

}
