package com.els.base.inquiry.command.pur;

import java.util.List;
import java.util.stream.Collectors;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.inquiry.AbstractInquiryCommand;
import com.els.base.inquiry.IOrderItem;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryAuditStatus;
import com.els.base.inquiry.enumclass.InquiryOrderStatus;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;
import com.els.base.inquiry.enumclass.OperationTypeEnum;

/**
 * 采购方定价
 * @author wuwenxin
 *
 */
public class PricingCommand extends AbstractInquiryCommand<String> {
	
	private PurOrder purOrder;
	
	public PricingCommand(PurOrder purOrder) {
		this.purOrder = purOrder;
	}

	@SuppressWarnings("unchecked")
	@Override
	public String execute(InquiryCommandInvoker invoker) {
		Assert.isNotBlank(purOrder.getTemplateId(), "模板id不能为空");
		// 1、检验各询价单行是否符合定价条件
		this.valid(purOrder);
		TemplateConf templateConf = invoker.getTemplateConfService().queryObjById(purOrder.getTemplateId());
		
		//2、批量更新询价单行
		List<IOrderItem> orderItemList = purOrder.getItemList().parallelStream()
			.filter(orderItem->{
				//筛选出已报价或者已重报价的询价单行
				IOrderItem temp = (IOrderItem) templateConf.getOrderItemService().queryObjById(orderItem.getId());
				return InquiryQuoteStatus.QUOTED.getCode().equals(temp.getQuotationStatus()) 
						|| InquiryQuoteStatus.REQUOTED.getCode().equals(temp.getQuotationStatus());
				
			}).filter(orderItem -> OperationTypeEnum.isPricingOperation(orderItem.getOperation()))
			.collect(Collectors.toList());
		
		//3、更新状态
		orderItemList.forEach(orderItem -> {
			this.modifyStatus(templateConf, orderItem);
		});
		
		return null;
	}
	
	public void modifyStatus(TemplateConf templateConf, IOrderItem orderItem){
		
		try {
			IOrderItem tempItem = templateConf.getOrderItemClass().newInstance();
			tempItem.setId(orderItem.getId());
			tempItem.setOperation(orderItem.getOperation());
			tempItem.setMaterialCode(orderItem.getMaterialCode());
			templateConf.getOrderItemService().modifyStatus(tempItem);
			
		} catch (InstantiationException | IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}

	private void valid(PurOrder purOrder) {
		if (!InquiryOrderStatus.PUBLISHED.getCode().equals(purOrder.getInquiryOrderStatus())) {
			throw new CommonException("询价单发布之后才能进行定价");
		}
		
		if (InquiryAuditStatus.AUDITING.getCode().equals(purOrder.getAuditStatus())) {
			throw new CommonException("询价单审核中，无法定价");
		}
		if (InquiryAuditStatus.AUDIT_PASS.getCode().equals(purOrder.getAuditStatus())) {
			throw new CommonException("询价单审核已通过，无法定价");
		}
		
		// 只有处于已报价或已重报状态下的询价单行才能执行定价操作
		purOrder.getItemList().forEach(item -> {
			if (OperationTypeEnum.isPricingOperation(item.getOperation())
					&& !InquiryQuoteStatus.QUOTED.getCode().equals(item.getQuotationStatus())
					&& !InquiryQuoteStatus.REQUOTED.getCode().equals(item.getQuotationStatus() )) {
				throw new CommonException("只有已报价或已重报的物料才能执行定价操作");
			}
		});
		
	}

}
