package com.els.base.inquiry.command.pur;

import static java.util.stream.Collectors.toList;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;

import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.AbstractInquiryCommand;
import com.els.base.inquiry.IMould;
import com.els.base.inquiry.IMouldDetail;
import com.els.base.inquiry.IOrderItem;
import com.els.base.inquiry.IOrderItemDetail;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.entity.InquiryBusiCondition;
import com.els.base.inquiry.entity.InquiryBusiConditionExample;
import com.els.base.inquiry.entity.InquiryQuoteLadder;
import com.els.base.inquiry.entity.InquirySupOrder;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryLadderBelongType;
import com.els.base.inquiry.enumclass.InquiryOrderStatus;
import com.els.base.inquiry.enumclass.InquiryQuoteLadderType;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;
import com.els.base.inquiry.enumclass.MessageNotify;
import com.els.base.inquiry.utils.PropertyValueUtils;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;

/**
 * 采购方发布询价单
 * @author wuwenxin
 *
 */
public class PublishCommand extends AbstractInquiryCommand<String> {
	
	private final String defaultCountry = "CN"; // 默认国家是中国
	
	private PurOrder purOrder;
	
	private TemplateConf templateConf;
	
	public PublishCommand(PurOrder purOrder) {
		this.purOrder = purOrder;
	}

	@Override
	public String execute(InquiryCommandInvoker invoker) {
		this.templateConf = invoker.getTemplateConfService().queryObjById(purOrder.getTemplateId());
		
		// 1、校验是否符合发布条件
		this.valid(purOrder);
		
		//2、初始化数据
		this.init(purOrder);
		
		// 2、如果是填写信息后直接执行发布操作，则新增，如果是保存后执行发布操作，则更新
		this.createOrModifyPurOrder(purOrder);
		
		// 3、开始发布
		List<InquirySupOrder> inquirySupOrders = this.publish(purOrder);
		
		// 4、发送消息通知供应商
		this.sendMsg(inquirySupOrders);
		return null;
	}

	private void sendMsg(List<InquirySupOrder> inquirySupOrders) {
		inquirySupOrders.forEach(inquirySupOrder -> {
			User user = invoker.getCompanyUserRefService().
			queryMainUserOfCompany(inquirySupOrder.getSupCompanyId());
			if (user != null) {
				String supUserId = user.getId();
				Map<String, String> data = new HashMap<>();
				data.put("inquirySupOrderId", inquirySupOrder.getId());
				data.put("orderNo", inquirySupOrder.getOrderNo());
				
				Message<Map<String, String>> msg = Message.init(data).
					setCompanyCode(getPurCompany().getCompanyCode()).
					setBusinessTypeCode(MessageNotify.INQUIRY_ORDER_PUBLISH.getCode()).
					setSenderId(purOrder.getPurUserId()).
					addReceiverId(supUserId).
					setMsgLevel(MessageLevelEnum.HIGH);
				MessageSendUtils.sendMessage(msg);
			}
		});
	}

	private List<InquirySupOrder> publish(PurOrder purOrder) {
		// 1、生成供应商的单头
		List<InquirySupOrder> supOrderList = this.addSupOrderList(purOrder);
		
		// 2、根据物料清单生成询价单行
		this.addOrderItemList(purOrder, supOrderList);
		
		// 3、生成商务条件
		this.fillBusiConditionList(purOrder, supOrderList);
		
		// 3、生成模具清单
		this.addMouldList(purOrder, supOrderList);
		
		// 5、将询价单状态定义为已发布状态
		this.transformToPublishStatus(purOrder);
		
		// 6、将待物料清单标记为已生成询价单状态
		return supOrderList;
	}

	@SuppressWarnings("unchecked")
	private void addMouldList(PurOrder purOrder, List<InquirySupOrder> supOrderList) {
		if (CollectionUtils.isEmpty(purOrder.getMouldList())) {
			return;
		}
		List<IMould> allMouldList = purOrder.getMouldList().parallelStream().flatMap(mould->{
			//根据采购方的询价单头的模具清单， 给每个供应商订单生成相同的模具清单
			return supOrderList.stream().map(supOrder->{
				IMould newMould = mould.build(purOrder, supOrder);
				newMould.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(newMould));
				return newMould;
			});
		}).collect(toList());
		
		templateConf.getMouldService().deleteByPurOrder(purOrder.getId());
		templateConf.getMouldService().addAll(allMouldList);
		
		if (Constant.YES_INT.equals(templateConf.getIsMouldDetailEnable())) {
			this.addMouldDetail(allMouldList);
		}
	}

	@SuppressWarnings("unchecked")
	private void addMouldDetail(List<IMould> allMouldList) {
		if (CollectionUtils.isEmpty(allMouldList)) {
			return;
		}
		
		List<IMouldDetail> allMouldDetailList = allMouldList.parallelStream().map(mould ->{
			IMouldDetail mouldDetail;
			try {
				mouldDetail = templateConf.getTplMouLdDetail().getMouldDetailClass().newInstance();
			} catch (InstantiationException | IllegalAccessException e) {
				throw new RuntimeException(e);
			}
			mouldDetail.build(mould);
			mouldDetail.setTemplateId(templateConf.getTplMouLdDetail().getId());
			mouldDetail.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(mouldDetail));
			return mouldDetail;
		}).collect(toList());
		
		templateConf.getTplMouLdDetail().getMouldDetailService().addAll(allMouldDetailList);
	}

	private void fillBusiConditionList(PurOrder purOrder, List<InquirySupOrder> supOrderList) {
		if (Constant.NO_INT.equals(this.templateConf.getIsBusiCondEnable())) {
			return;
		}
		List<InquiryBusiCondition> busiConditions = purOrder.getBusiConditions();
		Assert.isNotEmpty(busiConditions, String.format("根据询价单id：%s无法获取商务条件，发布失败", purOrder.getId()));
		Map<String, String> map = new HashMap<>(supOrderList.size());
		
		supOrderList.forEach(supOrder -> {
			map.put(supOrder.getSupCompanyId(), supOrder.getId());
		});
		busiConditions.forEach(busiCondition -> {
			busiCondition.setSupOrderId(map.get(busiCondition.getSupCompanyId()));
			busiCondition.setCountry(defaultCountry);
			busiCondition.setTemplateConfId(purOrder.getTemplateId());
			busiCondition.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(busiCondition));
		});
		
		InquiryBusiConditionExample example = new InquiryBusiConditionExample();
		example.createCriteria().andPurOrderIdEqualTo(purOrder.getId());
		invoker.getInquiryBusiConditionService().deleteByExample(example);
		
		/*List<InquiryBusiCondition> busiConditionList = supOrderList.stream().map(supOrder->{
			InquiryBusiCondition inquiryBusiCondition = new InquiryBusiCondition();
			inquiryBusiCondition.setSupOrderId(supOrder.getId());
			inquiryBusiCondition.setPurOrderId(purOrder.getId());
			inquiryBusiCondition.setSupCompanyId(supOrder.getSupCompanyId());
			inquiryBusiCondition.setSupCompanyName(supOrder.getSupCompanyName());
			inquiryBusiCondition.setSupCompanySrmCode(supOrder.getSupCompanySrmCode());
			inquiryBusiCondition.setSupCompanySapCode(supOrder.getSupCompanySapCode());
			inquiryBusiCondition.setTemplateConfId(purOrder.getTemplateId());
			inquiryBusiCondition.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(inquiryBusiCondition));
			return inquiryBusiCondition;
		}).collect(toList());*/
		
		invoker.getInquiryBusiConditionService().addAll(busiConditions);
	}

	@SuppressWarnings("unchecked")
	private void addOrderItemList(PurOrder order, List<InquirySupOrder> supOrderList) {
		
		//1、根据供应商询价单，物料清单，转化询价单行
		List<IOrderItem> allOrderItemList = supOrderList.parallelStream().flatMap(supOrder->{
			
			return order.getTargetList().stream().map(target->{
				Class<? extends IOrderItem> orderItemCls = this.templateConf.getOrderItemClass();
				IOrderItem orderItem = null;
				try {
					orderItem = orderItemCls.newInstance();
				} catch (InstantiationException | IllegalAccessException e) {
					throw new RuntimeException(e);
				}
				
				orderItem.build(order, supOrder, target);
				orderItem.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(orderItem));
				return orderItem;
			});
			
		}).collect(toList());
		
		//2、设置行号
		int purOrderIndex= 1;
		for (IOrderItem orderItem : allOrderItemList) {
			orderItem.setOrderItemNo(purOrderIndex ++);
		}
		
		this.templateConf.getOrderItemService().addAll(allOrderItemList);
		
		//3、批量添加询价单的阶梯报价
		this.addOrderItemQuoteLadderList(order, allOrderItemList);
		
		//4、添加对应的物料成本明细
		if (Constant.YES_INT.equals(templateConf.getIsOrderItemDetailEnable())) {
			this.addOrderItemDetail(allOrderItemList);
		}
	}
	
	@SuppressWarnings("unchecked")
	private void addOrderItemDetail(List<IOrderItem> allOrderItemList) {
		if (CollectionUtils.isEmpty(allOrderItemList)) {
			return;
		}
		
		List<IOrderItemDetail> detailList = allOrderItemList.stream().map(item->{
			IOrderItemDetail detail = null;
			try {
				detail = templateConf.getTplOrderItemDetail().getOrderItemDetailClass().newInstance();
			} catch (InstantiationException | IllegalAccessException e) {
				throw new RuntimeException(e);
			}
			detail.build(item);
			detail.setTemplateId(templateConf.getTplOrderItemDetail().getId());
			detail.setPropertyValueList(PropertyValueUtils.getDefaultPropertyValue(detail));
			return detail;
			
		}).collect(toList());
		
		templateConf.getTplOrderItemDetail().getOrderItemDetailService().addAll(detailList);
	}

	private void addOrderItemQuoteLadderList(PurOrder order, List<IOrderItem> allOrderItemList) {
		
		Stream<InquiryQuoteLadder> stream = allOrderItemList.stream().flatMap(orderItem->{
			
			//1、找出询价单行对应的物料清单,并根据物料清单的阶梯报价，生成行的阶梯报价
			return order.getTargetList().stream()
				.filter(target->{
					//1.1 找出物料清单中有设置阶梯报价的
					return CollectionUtils.isNotEmpty(target.getInquiryQuoteLadders()) 
							&& InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(target.getQuoteType());
				
				}).filter(target->{
					//1.2 找出询价单行对应的物料清单
					return target.getId().equals(orderItem.getTargetId());
					
				}).flatMap(target->{
					//1.3 把物料清单的阶梯报价找出来，并设置询价单行的数据
					return target.getInquiryQuoteLadders().stream().map(ladder->{
						InquiryQuoteLadder inquiryQuoteLadder = new InquiryQuoteLadder();
						BeanUtils.copyProperties(ladder, inquiryQuoteLadder);
						inquiryQuoteLadder.setId(null);
						inquiryQuoteLadder.setBelongType(InquiryLadderBelongType.BELONG_TO_ITEM.getCode());
						inquiryQuoteLadder.setItemId(orderItem.getId());
						inquiryQuoteLadder.setCreateCompanyType("PUR");
						return inquiryQuoteLadder;
					});
				});
			
		});
		
		this.invoker.getInquiryQuoteLadderService().addAll(stream.collect(toList()));
	}

	private List<InquirySupOrder> addSupOrderList(PurOrder order) {
		List<InquirySupOrder> supOrderList = order.getInquirySuppliers().stream().map(inquirySupplier ->{
			
			InquirySupOrder inquirySupOrder = new InquirySupOrder();
			inquirySupOrder.setSupCompanyId(inquirySupplier.getSupCompanyId());
			inquirySupOrder.setSupCompanyName(inquirySupplier.getSupCompanyName());
			inquirySupOrder.setSupCompanySrmCode(inquirySupplier.getSupCompanySrmCode());
			inquirySupOrder.setSupCompanySapCode(inquirySupplier.getSupCompanySapCode());
			inquirySupOrder.setType(order.getType());
			inquirySupOrder.setTemplateId(order.getTemplateId());
			inquirySupOrder.setPurOrderId(order.getId());
			inquirySupOrder.setOrderNo(order.getOrderNo());
			inquirySupOrder.setDigestExplain(order.getDigestExplain());
			inquirySupOrder.setPublishDate(order.getPublishDate());
			inquirySupOrder.setQuoteStartDate(order.getQuoteStartDate());
			inquirySupOrder.setQuoteEndDate(order.getQuoteEndDate());
			inquirySupOrder.setRemark(order.getRemark());
			inquirySupOrder.setPurCompanyId(order.getPurCompanyId());
			inquirySupOrder.setPurCompanyName(order.getPurCompanyName());
			inquirySupOrder.setPurUserId(order.getPurUserId());
			inquirySupOrder.setPurUserName(order.getPurUserName());
			inquirySupOrder.setCreateDate(order.getCreateDate());
			inquirySupOrder.setInquiryOrderStatus(InquiryOrderStatus.PUBLISHED.getCode());
			inquirySupOrder.setQuoteStatus(InquiryQuoteStatus.UNQUOTED.getCode());
			inquirySupOrder.setTemplateId(order.getTemplateId());
			return inquirySupOrder;
			
		}).collect(toList());
		
		this.invoker.getInquirySupOrderService().addAll(supOrderList);
		return supOrderList;
	}

	/**
	 * 创建或修改询价单，针对带询价单详情的发布
	 * @param order
	 */
	private void createOrModifyPurOrder(PurOrder purOrder) {
		AbstractInquiryCommand<String> command;
		// 如果询价单ID为空，说明发布之前没有进行过任何保存操作
		if (StringUtils.isBlank(purOrder.getId())) {
			command = new CreateCommand(purOrder);
		}else {
			command = new ModifyCommand(purOrder);
		}
		command.copyProperties(this);
		invoker.invoke(command);
	}

	/**
	 * 验证询价单是否满足发布的条件
	 * @param purOrder
	 */
	private void valid(PurOrder order) {
		Assert.isNotBlank(order.getTemplateId(), "订单的模板id不能为空");
		Assert.isNotNull(this.templateConf, "订单的模板不能为空");
		
		//1、如果询价单状态不为空或不处于未报价状态，则报错
		Assert.isNotNull(order.getInquiryOrderStatus() , "询价单状态不能为空");
		if ( !InquiryOrderStatus.UNPUBLISHED.getCode().equals(order.getInquiryOrderStatus())) {
			throw new CommonException(String.format("只有处于未发布状态询价单才能执行发布操作，请检查询价单[%s]", order.getOrderNo()));
		}
		
		//2、校验供应商
		// 如果供应商列表为空，则报错
		Assert.isNotEmpty(order.getInquirySuppliers(), String.format("发布询价单时必须选择供应商，请检查询价单[%s]", order.getOrderNo()));
		order.getInquirySuppliers().forEach(supCompany->{
			Assert.isNotBlank(supCompany.getId(), "供应商id不能为空");
			Assert.isNotBlank(supCompany.getSupCompanyName(), "供应商名称不能为空");
			Assert.isNotBlank(supCompany.getSupCompanySapCode(), "供应商SAP编码不能为空");
			Assert.isNotBlank(supCompany.getSupCompanySrmCode(), "供应商SRM编码不能为空");
		});
		
		//3、校验物料清单
		Assert.isNotEmpty(order.getTargetList(), String.format("物料清单不能为空，请检查询价单[%s]", order.getOrderNo()));
		order.getTargetList().forEach(target->{
			Assert.isNotBlank(target.getInfoCategory(), "信息类别不能为空");
			Assert.isNotBlank(target.getMaterialCode(), "物料编码不能为空");
			Assert.isNotBlank(target.getMapNo(), "图号不能为空");
			Assert.isNotBlank(target.getName(), "名称不能为空");
			Assert.isNotNull(target.getQuantity(), "数量不能为空");
			Assert.isNotNull(target.getRecentUntaxedUnitPrice(), "最近一次采购不含税单价不能为空");
			Assert.isNotBlank(target.getDecreasingAmplitude(), "降幅不能为空");
			Assert.isNotBlank(target.getPurchaseApply(), "采购申请单号不能为空");
			Assert.isNotBlank(target.getProject(), "行项目号不能为空");
			Assert.isNotBlank(target.getPurchaseGroup(), "采购组不能为空");
			Assert.isNotNull(target.getPriceUnit(), "价格单位不能为空");
			Assert.isNotNull(target.getValidDateFrom(), "有效期从不能为空");
			Assert.isNotNull(target.getValidDateTo(), "有效期到不能为空");

			if (!Constant.YES_INT.equals(templateConf.getIsOrderItemDetailEnable())
					&& !InquiryQuoteLadderType.CONVENTIONAL_QUOTE.getCode().equals(target.getQuoteType())
					&& !InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(target.getQuoteType()) ) {
				throw new CommonException(String.format("报价类型不能为空,请检查物料编码[%s]", target.getMaterialCode()));
			}

		});
		
		//4、校验物料清单中的阶梯报价
		
		
		
		order.getTargetList().stream()
		.filter(target -> InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(target.getQuoteType()))
		.forEach(target->{
			
			if (CollectionUtils.isEmpty(target.getInquiryQuoteLadders())) {
				throw new CommonException(String.format("物料编码[%s] 选择了报价方式为“阶梯报价”，但未设置报价阶梯！", target.getMaterialCode()));
			}
			
			target.getInquiryQuoteLadders().forEach(quoteLadder->{
				
//				Assert.isNotNull(quoteLadder.getMaterialCode(), "阶梯报价缺少物料编码");
				//Assert.isNotNull(quoteLadder.getMaterialDesc(), "阶梯报价缺少物料描述");
				Assert.isNotNull(quoteLadder.getNumberFrom(), String.format("物料[%s]阶梯报价缺少\"数量从\"", quoteLadder.getMaterialDesc()));
				Assert.isNotNull(quoteLadder.getNumberTo(), String.format("物料[%s]阶梯报价缺少\"数量到\"", quoteLadder.getMaterialDesc()));
				
				if (quoteLadder.getNumberTo().compareTo(quoteLadder.getNumberFrom()) < 0) {
					throw new CommonException(String.format("物料[%s]阶梯报价异常， \"数量从\"大于\"数量到\"", quoteLadder.getMaterialDesc()));
				}
			});
		});
		
		//5、校验模具清单
		if (Constant.YES_INT.equals(templateConf.getIsMouldEnable())
				&& CollectionUtils.isNotEmpty(order.getMouldList())) {
//			Assert.isNotEmpty(order.getMouldList(), String.format("询价单[%s]的模具清单不能为空", order.getOrderNo()));
			
			order.getMouldList().forEach(mould->{
//				Assert.isNotBlank(mould.getMouldCode(), "模具编码不能为空");
				Assert.isNotBlank(mould.getMouldName(), "模具名称不能为空");
			});
		}
	}

	/**
	 * 将询价单状态改为已发布
	 * @param order
	 */
	private void transformToPublishStatus(PurOrder order) {
		PurOrder temp = new PurOrder();
		temp.setId(order.getId());
		temp.setInquiryOrderStatus(InquiryOrderStatus.PUBLISHED.getCode());
		temp.setPublishDate(new Date());
		invoker.getPurOrderService().modifyObj(temp);
	}
	
	private void init(PurOrder purOrder) {
		purOrder.setPublishDate(new Date());
	}

}
