package com.els.base.inquiry.command.pur;

import static java.util.stream.Collectors.toList;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.AbstractInquiryCommand;
import com.els.base.inquiry.IMould;
import com.els.base.inquiry.IOrderItem;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.entity.InquiryQuoteLadderExample;
import com.els.base.inquiry.entity.InquirySupOrder;
import com.els.base.inquiry.entity.InquirySupOrderExample;
import com.els.base.inquiry.entity.InquirySupplier;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.PurOrderExample;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryLadderBelongType;
import com.els.base.inquiry.enumclass.InquiryOrderStatus;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;

/**
 * 采购方撤回操作
 * @author wuwenxin
 *
 */
public class RecallCommand extends AbstractInquiryCommand<String> {
	
	private List<String> ids;
	
	public RecallCommand(List<String> ids) {
		this.ids = ids;
	}

	@Override
	public String execute(InquiryCommandInvoker invoker) {
		Assert.isNotEmpty(ids, "数据不能为空");
		
		logger.info("根据询价单ID，查询询价单数据");
		List<PurOrder> purOrderList = this.queryPurOrdersByIds(ids);
		
		this.valid(purOrderList);
		
		//1、根据询价单ID,删除所有供应商询价单
		this.deleteAllSupOrders(ids);
		
		//2、根据询价单ID,删除所有行项目
		this.deleteAllItems(purOrderList);
		
		//3、根据询价单ID,删除所有模具清单
		this.deleteAllMoulds(purOrderList);
		
		//4、删除所有商务条件
		//this.deleteAllBusinessCondtion(ids);
		
		//5、将询价单状态改成未发布
		invoker.getPurOrderService().transformToUnPublish(ids);
		
		return null;
	}

	/*private void deleteAllBusinessCondtion(List<String> ids) {
		InquiryBusiConditionExample example = new InquiryBusiConditionExample();
		example.createCriteria().andPurOrderIdIn(ids);
		
		invoker.getInquiryBusiConditionService().deleteByExample(example);
	}*/

	@SuppressWarnings("unchecked")
	private void deleteAllMoulds(List<PurOrder> purOrderList) {
		logger.info("根据询价单ID,删除所有模具清单,与模具清单明细");
		purOrderList.stream()
			.forEach(purOrder->{
				
				TemplateConf templateConf = invoker.getTemplateConfService().queryObjById(purOrder.getTemplateId());
				if (!Constant.YES_INT.equals(templateConf.getIsMouldEnable())) {
					return;
				}
				
				List<? extends IMould> mouldList = templateConf.getMouldService().queryByPurOrderId(purOrder.getId());
				if (CollectionUtils.isEmpty(mouldList)) {
					return;
				}
				
				logger.info("根据询价单ID[{}],删除所有模具清单", purOrder.getId());
				templateConf.getMouldService().deleteByPurOrder(purOrder.getId());
				
				logger.info("根据询价单ID,删除所有模具清单成本清单");
				List<String> mouldIdList = mouldList.stream().map(IMould::getId).collect(Collectors.toList());
				if (Constant.YES_INT.equals(templateConf.getIsMouldDetailEnable())
						&& CollectionUtils.isNotEmpty(mouldIdList)) {
					
					templateConf.getTplMouLdDetail()
						.getMouldDetailService()
						.deleteByMouldIdList(mouldIdList);
				}
				
				logger.info("根据询价单ID[{}],插入未发送的模具清单", purOrder.getId());
				List<InquirySupplier> supplierList = invoker.getInquirySupplierService().queryByPurOrderId(purOrder.getId());
				InquirySupplier supplier = supplierList.get(0);
				
				List<? extends IMould> recallMouldList = mouldList.stream()
					.filter(mould-> supplier.getSupCompanySrmCode().equals(mould.getSupCompanySrmCode()))
					.collect(toList());
				
				recallMouldList.forEach(newMould->{
					newMould.setSupCompanyName(null);
					newMould.setSupCompanySrmCode(null);
					newMould.setSupCompanySapCode(null);
				});
				
				templateConf.getMouldService().addAll(recallMouldList);
			});
	}

	@SuppressWarnings("unchecked")
	private void deleteAllItems(List<PurOrder> purOrderList) {
		
		logger.info("根据所有询价单行id,删除所有询价单行");
		List<String> orderItemIdList = purOrderList.stream()
			.flatMap(purOrder->{
				
				TemplateConf templateConf = invoker.getTemplateConfService().queryObjById(purOrder.getTemplateId());
				
				List<IOrderItem> tempList = templateConf.getOrderItemService().queryByPurOrderId(purOrder.getId());
				List<String> itemIdList = tempList.stream().map(IOrderItem::getId).collect(toList());
				
				if (Constant.YES_INT.equals(templateConf.getIsOrderItemDetailEnable())
						&& CollectionUtils.isNotEmpty(itemIdList)) {
					templateConf.getTplOrderItemDetail().getOrderItemDetailService().deleteByOrderItemIds(itemIdList);
				}
				templateConf.getOrderItemService().deleteByPurOrderId(purOrder.getId());
				
				return itemIdList.stream();
				
			}).collect(toList());
		
		if (CollectionUtils.isEmpty(orderItemIdList)) {
			return;
		}
		
		logger.info("根据所有询价单行,删除阶梯报价");
		InquiryQuoteLadderExample example = new InquiryQuoteLadderExample();
		example.createCriteria()
			.andBelongTypeEqualTo(InquiryLadderBelongType.BELONG_TO_ITEM.getCode())
			.andItemIdIn(orderItemIdList);
		
		invoker.getInquiryQuoteLadderService().deleteByExample(example);
	}

	private void deleteAllSupOrders(List<String> idList) {
		logger.info("根据询价单ID,删除所有供应商询价单,ids[{}]", ids);
		InquirySupOrderExample example = new InquirySupOrderExample();
		InquirySupOrderExample.Criteria criteria = example.createCriteria();
		criteria.andPurOrderIdIn(idList);
		invoker.getInquirySupOrderService().deleteByExample(example);
	}

	private void valid(List<PurOrder> purOrders) {
		
		// 根据ids批量查询询价单
		Assert.isNotEmpty(purOrders, "询价单在系统不存在，请刷新后重试");
		
		for (PurOrder purOrder : purOrders) {
			// 只有当询价单为已发布时才能撤回
			if ( ! InquiryOrderStatus.PUBLISHED.getCode().equals(purOrder.getInquiryOrderStatus())) {
				throw new CommonException(String.format("询价单状态不是已发布，不能撤回。询价单[%s]", purOrder.getOrderNo()));
			}
		}
		
		InquirySupOrderExample example = new InquirySupOrderExample();
		example.createCriteria()
			.andPurOrderIdIn(purOrders.stream().map(PurOrder::getId).collect(toList()))
			.andQuoteStatusNotEqualTo(InquiryQuoteStatus.UNQUOTED.getCode());
		
		List<InquirySupOrder> inquirySupOrders = invoker.getInquirySupOrderService().queryAllObjByExample(example);
		
		// 根据ids批量查询供应商询价单
		if (CollectionUtils.isNotEmpty(inquirySupOrders)) {
			throw new CommonException(String.format("询价单已经有供应商进行报价，无法撤回。询价单[%s]", inquirySupOrders.get(0).getOrderNo()));
		}
		
	}

	private List<PurOrder> queryPurOrdersByIds(List<String> idList) {
		PurOrderExample example = new PurOrderExample();
		PurOrderExample.Criteria criteria = example.createCriteria();
		criteria.andIdIn(idList);
		return invoker.getPurOrderService().queryAllObjByExample(example);
	}
	
}
