package com.els.base.inquiry.command.sup;

import static java.util.stream.Collectors.toList;

import java.math.BigDecimal;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.AbstractInquiryCommand;
import com.els.base.inquiry.IMould;
import com.els.base.inquiry.IMouldDetail;
import com.els.base.inquiry.IOrderItem;
import com.els.base.inquiry.IOrderItemDetail;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.command.pur.FrozenOrderCommand;
import com.els.base.inquiry.command.strategy.LadderPriceStrategy;
import com.els.base.inquiry.entity.InquiryBusiCondition;
import com.els.base.inquiry.entity.InquiryQuoteLadder;
import com.els.base.inquiry.entity.InquiryQuoteLadderExample;
import com.els.base.inquiry.entity.InquirySupFile;
import com.els.base.inquiry.entity.InquirySupFileExample;
import com.els.base.inquiry.entity.InquirySupOrder;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryFileType;
import com.els.base.inquiry.enumclass.InquiryLadderBelongType;
import com.els.base.inquiry.enumclass.InquiryOrderStatus;
import com.els.base.inquiry.enumclass.InquiryQuoteLadderType;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;
import com.els.base.inquiry.utils.LaddPriceValider;
import com.els.base.utils.uuid.UUIDGenerator;

/**
 * 供应商修改报价
 * @author wuwenxin
 *
 */
public class ModifyQuoteCommand extends AbstractInquiryCommand<String> {
	
	private InquirySupOrder supOrder;
	private TemplateConf templateConf;
	
	public ModifyQuoteCommand(InquirySupOrder supOrder) {
		this.supOrder = supOrder;
	}

	@Override
	public String execute(InquiryCommandInvoker invoker) {
		this.valid(this.supOrder);
		
		this.init(this.supOrder);
		
		//1、批量更新询价单行
		this.modifyOrderItemList(supOrder);
		
		//2、批量更新阶梯报价
		this.modifyQuoteLadder(supOrder.getItemList());
		
		//3、批量更新物料成本清单
		if (Constant.YES_INT.equals(templateConf.getIsOrderItemDetailEnable())) {
			this.modifyItemDetail(supOrder.getItemList());
		}
		
		//4、批量更新模具清单
		if (Constant.YES_INT.equals(templateConf.getIsMouldEnable())) {
			this.modifyMould(supOrder);
			
			if (Constant.YES_INT.equals(templateConf.getIsMouldDetailEnable())) {
				this.modifyMouldDetail(supOrder.getMouldList());
			}
		}
		
		//5、修改供应商表单文件
		this.modifyInquirySupFiles(supOrder, supOrder.getSupFileList());
		
		//6、修改商务条件
		this.modifyInquiryBusiCondition(supOrder.getBusiConditions());
		return null;
	}
	
	@SuppressWarnings("unchecked")
	private void modifyOrderItemList(InquirySupOrder supOrder) {
		supOrder.getItemList().parallelStream().forEach(supOrderItem-> {
			supOrderItem.setSupOrderId(supOrder.getId());
			
			supOrderItem.setSupUserId(this.getSupUser().getId());
			supOrderItem.setSupUserName(this.getSupUser().getNickName());
			
			//更新未含税金额
			calculateUnTaxedPrice(supOrderItem, this.supOrder);
			
		});

		templateConf.getOrderItemService().deleteBySupOrderId(supOrder.getId());
		templateConf.getOrderItemService().addAll(supOrder.getItemList());
	}


	/**
	 * 计算未含税金额
	 * @param supOrderItem
	 * @param supOrder2
	 * @return
	 */
	private BigDecimal calculateUnTaxedPrice(IOrderItem supOrderItem, InquirySupOrder supOrder) {
		if (Constant.YES_INT.equals(this.templateConf.getIsOrderItemDetailEnable())
				&& supOrderItem.getOrderItemDetail() != null) {
			return supOrderItem.getOrderItemDetail().calculate();
			
		}
		
		if (Constant.YES_INT.equals(this.templateConf.getIsLadderPriceEnable())
				&& CollectionUtils.isNotEmpty(supOrderItem.getInquiryQuoteLadders())) {
			return new LadderPriceStrategy(supOrderItem.getInquiryQuoteLadders()).calculate();
		}
		
		return supOrderItem.getUntaxedUnitPrice();
	}

	protected void init(InquirySupOrder supOrder) {
		this.templateConf = invoker.getTemplateConfService().queryObjById(supOrder.getTemplateId());
		supOrder.setSupUserId(this.getSupUser().getId());
		supOrder.setSupUserName(this.getSupUser().getNickName());	
	}
	
	@SuppressWarnings("unchecked")
	protected void modifyMouldDetail(List<IMould> mouldList) {
		if (CollectionUtils.isEmpty(mouldList)) {
			return;
		}
		
		List<String> mouldIdList = mouldList.stream().map(IMould::getId).collect(toList());
		templateConf.getTplMouLdDetail().getMouldDetailService().deleteByMouldIdList(mouldIdList);
		
		List<IMouldDetail> detailList = mouldList.stream()
			.filter(mould-> mould.getMouldDetail()!=null)
			.map(mould->{
				IMouldDetail mouldDetail = mould.getMouldDetail();
				mouldDetail.setMouldId(mould.getId());
				mouldDetail.setPurOrderId(mould.getPurOrderId());
				mouldDetail.setSupOrderId(mould.getSupOrderId());
				mouldDetail.setSupCompanyId(mould.getSupCompanyId());
				mouldDetail.setSupCompanyName(mould.getSupCompanyName());
				mouldDetail.setSupCompanySapCode(mould.getSupCompanySapCode());
				mouldDetail.setSupCompanySrmCode(mould.getSupCompanySrmCode());
				mouldDetail.setTemplateId(this.templateConf.getTplMouLdDetail().getId());
				return mouldDetail;
			
			}).collect(toList());
		
		templateConf.getTplMouLdDetail().getMouldDetailService().addAll(detailList);
	}

	@SuppressWarnings("unchecked")
	protected void modifyMould(InquirySupOrder supOrder) {
		
		templateConf.getMouldService().deleteBySupOrder(supOrder.getId());
		
		supOrder.getMouldList().forEach(newMould->{
			newMould.setPurOrderId(supOrder.getPurOrderId());
			newMould.setSupOrderId(supOrder.getId());
			newMould.setSupCompanyName(supOrder.getSupCompanyName());
			newMould.setSupCompanySrmCode(supOrder.getSupCompanySrmCode());
			newMould.setSupCompanySapCode(supOrder.getSupCompanySapCode());
		});
		templateConf.getMouldService().addAll(supOrder.getMouldList());
	}

	@SuppressWarnings("unchecked")
	protected void modifyItemDetail(List<IOrderItem> itemList) {
		List<IOrderItemDetail> detailList = itemList.stream()
			.filter(item-> item.getOrderItemDetail()!=null)
			.map(item->{
			
				IOrderItemDetail detail = item.getOrderItemDetail();
				detail.setOrderItemId(item.getId());
				detail.setTemplateId(this.templateConf.getTplOrderItemDetail().getId());
				detail.setMaterialCode(item.getMaterialCode());
				return detail;
			
			}).collect(toList());
		
		List<String> orderItemIdList = itemList.stream().map(IOrderItem::getId).collect(toList());
		
		templateConf.getTplOrderItemDetail().getOrderItemDetailService().deleteByOrderItemIds(orderItemIdList);
		templateConf.getTplOrderItemDetail().getOrderItemDetailService().addAll(detailList);
		
	}

	protected void modifyQuoteLadder(List<IOrderItem> itemList) {
		
		List<InquiryQuoteLadder> ladderList = itemList.stream()
			.filter(item-> CollectionUtils.isNotEmpty(item.getInquiryQuoteLadders()))
			.flatMap(item->{
			
			item.getInquiryQuoteLadders().forEach(ladder-> {
				ladder.setItemId(item.getId());
				ladder.setBelongType(InquiryLadderBelongType.BELONG_TO_ITEM.getCode());
				ladder.setCreateCompanyType(StringUtils.defaultIfBlank(ladder.getCreateCompanyType(), "SUP"));
			});
			
			return item.getInquiryQuoteLadders().stream();
			
		}).collect(toList());
		
		List<String> orderItemIdList = itemList.stream().map(IOrderItem::getId).collect(toList());
		
		//先批量删除
		InquiryQuoteLadderExample ladderExample = new InquiryQuoteLadderExample();
		ladderExample.createCriteria()
			.andBelongTypeEqualTo(InquiryLadderBelongType.BELONG_TO_ITEM.getCode())
			.andItemIdIn(orderItemIdList);
		
		invoker.getInquiryQuoteLadderService().deleteByExample(ladderExample);
		invoker.getInquiryQuoteLadderService().addAll(ladderList);
	}

	protected void valid(InquirySupOrder supOrder) {
		Assert.isNotBlank(supOrder.getId(), "询价单id不能为空");
		Assert.isNotBlank(supOrder.getPurOrderId(), "询价单id不能为空");
		Assert.isNotEmpty(supOrder.getItemList(), "询价单行不能为空");
		Assert.isNotBlank(supOrder.getTemplateId(), "模板id不能为空");
		
		if (!InquiryOrderStatus.PUBLISHED.getCode().equals(supOrder.getInquiryOrderStatus())) {
			throw new CommonException("询价单不是已发布状态，不能修改");
		}
		
		if (!InquiryQuoteStatus.UNQUOTED.getCode().equals(supOrder.getQuoteStatus())
				&& !InquiryQuoteStatus.REQUOTE.getCode().equals(supOrder.getQuoteStatus())) {
			throw new CommonException("询价单已经报价，不能修改");
		}
		
		supOrder.getItemList().stream()
			.filter(item-> StringUtils.isBlank(item.getId()))
			.findAny()
			.ifPresent(item -> {
				throw new CommonException("询价单行Id不能为空");
			});
		
		supOrder.getItemList().stream()
			.filter(item-> !Constant.YES_INT.equals(item.getUnableToQuote()))
			.forEach(item->{
				if (!InquiryQuoteLadderType.CONVENTIONAL_QUOTE.getCode().equals(item.getQuoteType())
						&& !InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(item.getQuoteType())) {
					throw new CommonException("报价类型无法识别");
				}
				
				LaddPriceValider.valid(item.getInquiryQuoteLadders());
			});
		
		PurOrder purOrder = invoker.getPurOrderService().queryObjById(supOrder.getPurOrderId());
		if (purOrder.getQuoteEndDate()!= null && System.currentTimeMillis() > purOrder.getQuoteEndDate().getTime()) {
			this.invoker.getThreadPoolTaskExecutor().execute(new Runnable() {
				
				@Override
				public void run() {
					invoker.invoke(new FrozenOrderCommand(supOrder.getPurOrderId()));
				}
			});
			throw new CommonException("该询价单已经截止报价");
		}
	}

	/**
	 * 修改供应商商务条件
	 * @param busiConditions
	 */
	protected void modifyInquiryBusiCondition(List<InquiryBusiCondition> busiConditions) {
		if (CollectionUtils.isEmpty(busiConditions)) {
			return;
		}
		
		if(busiConditions.size() != 1) {
			throw new CommonException("商务条件存在多个商务条件");
		}
		
		InquiryBusiCondition busiCondition = busiConditions.get(0);
		busiCondition.setSupUserId(this.getSupUser().getId());
		busiCondition.setSupUserName(this.getSupUser().getLoginName());
		busiCondition.setSupCompanySrmCode(this.getSupCompany().getCompanyCode());
		busiCondition.setSupCompanySapCode(this.getSupCompany().getCompanySapCode());
		invoker.getInquiryBusiConditionService().modifyObj(busiCondition);
	}

	/**
	 * 修改表单文件，先全部删除，然后新增
	 * @param supOrder
	 * @param inquirySupFiles
	 */
	protected void modifyInquirySupFiles(InquirySupOrder supOrder, List<InquirySupFile> inquirySupFiles) {
		
		this.deleteSurFilesBySupOrderId(supOrder.getId());
		
		if (CollectionUtils.isEmpty(inquirySupFiles)) {
			return;
		}
		
		inquirySupFiles.forEach(inquirySupFile -> {
			inquirySupFile.setId(UUIDGenerator.generateUUID());
			inquirySupFile.setType(InquiryFileType.PUBLIC.getCode());//所有的供应商都属于公有文件
			inquirySupFile.setSupOrderId(supOrder.getId());
			inquirySupFile.setPurOrderId(supOrder.getPurOrderId());
			inquirySupFile.setSupCompanyId(this.getSupCompany().getId());
			inquirySupFile.setSupCompanyName(this.getSupCompany().getCompanyName());
			inquirySupFile.setSupUserId(this.getSupUser().getId());
			inquirySupFile.setSupUserName(this.getSupUser().getLoginName());
			invoker.getInquirySupFileService().addObj(inquirySupFile);
		});
	}

	/**
	 * 根据供应商询价单ID删除表单文件
	 * @param supOrderId
	 */
	private void deleteSurFilesBySupOrderId(String supOrderId) {
		InquirySupFileExample example = new InquirySupFileExample();
		InquirySupFileExample.Criteria criteria = example.createCriteria();
		criteria.andSupOrderIdEqualTo(supOrderId);
		invoker.getInquirySupFileService().deleteByExample(example);
	}

}
