package com.els.base.inquiry.command.sup;

import static java.util.stream.Collectors.counting;
import static java.util.stream.Collectors.groupingBy;

import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.AbstractInquiryCommand;
import com.els.base.inquiry.IOrderItem;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.command.pur.FrozenOrderCommand;
import com.els.base.inquiry.entity.InquiryQuoteLadder;
import com.els.base.inquiry.entity.InquirySupOrder;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryQuoteLadderType;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;
import com.els.base.inquiry.enumclass.MessageNotify;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;

/**
 * 供应商发布报价单
 * @author wuwenxin
 *
 */
public class PublishQuoteCommand extends AbstractInquiryCommand<String> {
	
	//修改前的询价单头
	private InquirySupOrder preSupOrder;
	private InquirySupOrder supOrder;
	private TemplateConf templateConf;
	
	public PublishQuoteCommand(InquirySupOrder supOrder) {
		this.supOrder = supOrder;
	}

	@Override
	public String execute(InquiryCommandInvoker invoker) {
		// 1、校验供应商报价是否符合发布条件
		
		this.valid(supOrder);
		
		// 3、开始发布
		this.publish(supOrder);
		
		// 4、发送消息通知采购方
		this.sendMsg(supOrder);
		
		return null;
	}

	private void sendMsg(InquirySupOrder order) {
		Map<String, String> data = new HashMap<>();
		data.put("purOrderId", order.getPurOrderId());
		data.put("orderNo", order.getOrderNo());
		data.put("supCompanyName", order.getSupCompanyName());
		Message<Map<String, String>> msg = Message.init(data).
				setCompanyCode(getSupCompany().getCompanyCode()).
				setBusinessTypeCode(MessageNotify.INQUIRY_ORDER_QUOTE.getCode()).
				setSenderId(getSupUser().getId()).
				addReceiverId(order.getPurUserId()).
				setMsgLevel(MessageLevelEnum.HIGH);
				MessageSendUtils.sendMessage(msg);
	}

	private void publish(InquirySupOrder order) {
		//1、 改变询价单行状态
		this.transformItemQuoteStatus(order);
		
		//2、 改变供应商询价单头状态
		this.transformSupOrderQuoteStatus(order);
		
		//3、 改变采购方询价单头报价状态
		this.transformPurOrderQuoteStatus(order);
		
	}

	private void transformPurOrderQuoteStatus(InquirySupOrder order) {
		PurOrder temp = new PurOrder();
		temp.setId(order.getPurOrderId());
		temp.setQuoteStatus(this.getPurOrderQuoteStatus(order.getPurOrderId()));
		invoker.getPurOrderService().modifyObj(temp);
	}

	@SuppressWarnings("unchecked")
	private Integer getPurOrderQuoteStatus(String purOrderId) {
		
		List<IOrderItem> orderItemList = templateConf.getOrderItemService().queryByPurOrderId(purOrderId);
		if (CollectionUtils.isEmpty(orderItemList)) {
			throw new CommonException("询价单行数据不存在，请刷新后重试");
		}
		
		// 根据各个供应商的报价状态做统计，相当与 select count(*) group by quoteStatus
		Map<Integer, Long> statusCount = orderItemList.stream()
				.filter(orderItem-> orderItem.getQuotationStatus()!=null)
				.collect(groupingBy(IOrderItem::getQuotationStatus, counting()));
		
		//已报价的数量
		Long countByQuoted = statusCount.get(InquiryQuoteStatus.QUOTED.getCode());
		if (countByQuoted == null) {
			countByQuoted = 0L;
		}
		
		//已经重报价的数量
		Long countByRequoted = statusCount.get(InquiryQuoteStatus.REQUOTED.getCode());
		if (countByRequoted == null) {
			countByRequoted = 0L;
		}

		//重报价的数量
		Long countByRequote = statusCount.get(InquiryQuoteStatus.REQUOTE.getCode());
		if (countByRequote == null) {
			countByRequote = 0L;
		}
		
		if ( (countByQuoted + countByRequoted) == orderItemList.size() ) {
			//全部是已报价或已重报价，那么就是已报价
			return InquiryQuoteStatus.QUOTED.getCode();
		}
		
		if (countByRequote > 0) {
			//只要有一个重报价，采购方状态显示重报价
			return InquiryQuoteStatus.REQUOTE.getCode();
			
		}else {
			//部分报价
			return InquiryQuoteStatus.QUOTE_SECTION.getCode();
		}
		
	}

	// 改变询价单行状态
	private void transformItemQuoteStatus(InquirySupOrder order) {
		List<IOrderItem> orderItems = order.getItemList();
		orderItems.stream()
			//不修改 已报价，或者已重报价
			.filter(orderItem -> InquiryQuoteStatus.UNQUOTED.getCode().equals(orderItem.getQuotationStatus()) 
					|| InquiryQuoteStatus.REQUOTE.getCode().equals(orderItem.getQuotationStatus()))
			
			//修改询价单行
			.forEach(orderItem -> {
			
			if ( InquiryQuoteStatus.UNQUOTED.getCode().equals(orderItem.getQuotationStatus())) {
				
				// 报价状态为未报价，则改成已报价
				orderItem.setQuotationStatus(InquiryQuoteStatus.QUOTED.getCode());
			} 
			
			if (InquiryQuoteStatus.REQUOTE.getCode().equals(orderItem.getQuotationStatus())) {
				// 报价状态为重报价，则改成已重报
				orderItem.setQuotationStatus(InquiryQuoteStatus.REQUOTED.getCode());
			}
			
			// 更新询价单行
			orderItem.setQuoteTime(new Date());
			templateConf.getOrderItemService().modifyStatus(orderItem);
		});
		
	}

	private void transformSupOrderQuoteStatus(InquirySupOrder order) {
		InquirySupOrder temp = new InquirySupOrder();
		temp.setId(order.getId());
		
		if (InquiryQuoteStatus.UNQUOTED.getCode().equals(order.getQuoteStatus())) {
			// 未报价 --》 已报价
			temp.setQuoteStatus(InquiryQuoteStatus.QUOTED.getCode());
		}
		
		if (InquiryQuoteStatus.REQUOTE.getCode().equals(order.getQuoteStatus())) {
			boolean isExistRequote = order.getItemList().stream().anyMatch(item->InquiryQuoteStatus.REQUOTE.getCode().equals(item.getQuotationStatus()));
			if(isExistRequote){
				//如果还有重报价的行，保持重报价
				temp.setQuoteStatus(InquiryQuoteStatus.REQUOTE.getCode());
				
			}else {
				// 重报价 --》 已报价
				temp.setQuoteStatus(InquiryQuoteStatus.QUOTED.getCode());
			}
		}
		temp.setSupUserId(this.getSupUser().getId());
		temp.setSupUserName(this.getSupUser().getNickName());
		temp.setQuoteTime(new Date());
		invoker.getInquirySupOrderService().modifyObj(temp);
	}

	/**
	 * 验证供应商填写的信息是否符合发布条件
	 * @param order
	 */
	private void valid(InquirySupOrder order) {
		templateConf = invoker.getTemplateConfService().queryObjById(supOrder.getTemplateId());
		preSupOrder = invoker.getInquirySupOrderService().queryObjById(supOrder.getId());
		
		//1、校验询价单的状态。 只有处于供应商询价单头处于未报价或重报价状态时，才允许发布报价
		if ( !InquiryQuoteStatus.UNQUOTED.getCode().equals(preSupOrder.getQuoteStatus()) 
				&& !InquiryQuoteStatus.REQUOTE.getCode().equals(preSupOrder.getQuoteStatus())) {
			throw new CommonException("只有处于未报价或重报价状态下的询价单才允许发布报价");
		}
		
		//2、校验询价单的截止时间。 如果已经超出截止时间，就不允许报价，并把询价单改为冻结
		PurOrder purOrder = invoker.getPurOrderService().queryObjById(supOrder.getPurOrderId());
		if (purOrder.getQuoteEndDate()!= null && System.currentTimeMillis() > purOrder.getQuoteEndDate().getTime()) {
			this.invoker.getThreadPoolTaskExecutor().execute(new Runnable() {
				
				@Override
				public void run() {
					invoker.invoke(new FrozenOrderCommand(supOrder.getPurOrderId()));
				}
			});
			throw new CommonException("该询价单已经截止报价");
		}
		
		//2、校验报价单的报价， 对于未报价，或者重报价的询价单行，填写不含税价或勾选了无法报价之后才允许发布
		List<IOrderItem> orderItems = order.getItemList();
		orderItems.stream()
			.filter(orderItem -> {
				if (Constant.YES_INT.equals(orderItem.getUnableToQuote())) {
					return false;
				}
				if(InquiryQuoteStatus.QUOTED.getCode().equals(orderItem.getQuotationStatus()) 
						|| InquiryQuoteStatus.REQUOTED.getCode().equals(orderItem.getQuotationStatus())){
					return false;
				}
				return true;
				
			}).forEach(orderItem->{
				if (!InquiryQuoteLadderType.CONVENTIONAL_QUOTE.getCode().equals(orderItem.getQuoteType())
						&& !InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(orderItem.getQuoteType())) {
					throw new CommonException("报价类型无法识别");
				}
				
				if (orderItem.getPlanDeliveryDate() == null) {
					throw new CommonException("计划交货时间必填");
				}
				
				if (InquiryQuoteLadderType.CONVENTIONAL_QUOTE.getCode().equals(orderItem.getQuoteType()) 
					&& (orderItem.getUntaxedUnitPrice() == null || BigDecimal.ZERO.compareTo(orderItem.getUntaxedUnitPrice()) == 0)) {
					throw new CommonException(String.format("询价单尚未报价，无法发布！请检查物料编码[%s]。", orderItem.getMaterialCode()));
				}
		
				if (InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(orderItem.getQuoteType())
						&& CollectionUtils.isEmpty(orderItem.getInquiryQuoteLadders())) {
					throw new CommonException(String.format("询价单尚未报价，无法发布！请检查物料编码[%s]。", orderItem.getMaterialCode()));
				}
				
				if (InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(orderItem.getQuoteType())
						&& !CollectionUtils.isEmpty(orderItem.getInquiryQuoteLadders())) {
					this.validLadderPrice(orderItem);
				}
				
				//校验能否报价
				orderItem.isVaildForQuote();
			});
			
		
		//3、校验商务条件
		if (Constant.YES_INT.equals(this.templateConf.getIsBusiCondEnable())) {
			Assert.isNotEmpty(order.getBusiConditions(), "商务条件不能为空");
			order.getBusiConditions().forEach(busiCondition-> busiCondition.isVaildForQuote());
		}
		
		//4、校验模具清单，和模具成本清单
		if (Constant.YES_INT.equals(templateConf.getIsMouldEnable()) 
				&& CollectionUtils.isNotEmpty(order.getMouldList())) {
//			Assert.isNotEmpty(order.getMouldList(), "模具清单不能为空");
			order.getMouldList().forEach(mould->{
				Assert.isNotBlank(mould.getMouldName(), "模具名称不能为空");
				
				if (Constant.YES_INT.equals(templateConf.getIsMouldDetailEnable())) {
					Assert.isNotNull(mould.getMouldDetail(), String.format("模具[%s]成本清单不能为空", mould.getMouldName()));
				}
			});
			
			
		}
		
	}

	private void validLadderPrice(IOrderItem orderItem) {
		for(InquiryQuoteLadder ladderPrice : orderItem.getInquiryQuoteLadders()){
			Assert.isNotNull(ladderPrice.getUntaxedUnitPrice(), "阶梯报价，价格不能为空");
			
			if (ladderPrice.getUntaxedUnitPrice().compareTo(BigDecimal.ZERO) <=0){
				throw new CommonException(String.format("询价单尚未报价完成，无法发布！请检查物料编码 [%s] 的阶梯价", orderItem.getMaterialCode()));
			}
		}
	}

}
