package com.els.base.inquiry.service.detail;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.utils.Assert;
import com.els.base.inquiry.IOrderItemDetail;
import com.els.base.inquiry.IOrderItemDetailService;
import com.els.base.inquiry.dao.detail.OrderItemDetailMapper;
import com.els.base.inquiry.entity.detail.OrderItemDetail;
import com.els.base.inquiry.entity.detail.OrderItemDetailExample;
import com.els.base.inquiry.enumclass.PropertyDefTplType;
import com.els.base.inquiry.utils.PropertyValueUtils;
import com.els.base.utils.uuid.UUIDGenerator;

@Service("defaultOrderItemDetailService")
public class OrderItemDetailServiceImpl implements IOrderItemDetailService<OrderItemDetail, OrderItemDetailExample, String> {
    @Resource
    protected OrderItemDetailMapper orderItemDetailMapper;

    @Transactional
    @CacheEvict(value={"orderItemDetail"}, allEntries=true)
    @Override
    public void addObj(OrderItemDetail t) {
        this.orderItemDetailMapper.insertSelective(t);
        PropertyValueUtils.addPropertyValue(Arrays.asList(t), PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
    }

    @Transactional
    @CacheEvict(value={"orderItemDetail"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.orderItemDetailMapper.deleteByPrimaryKey(id);
        PropertyValueUtils.deleteByExtenable(Arrays.asList(id), PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
    }

    @Transactional
    @CacheEvict(value={"orderItemDetail"}, allEntries=true)
    @Override
    public void modifyObj(OrderItemDetail t) {
        Assert.isNotBlank(t.getId(), "id 为空，无法修改");
        this.orderItemDetailMapper.updateByPrimaryKeySelective(t);
        PropertyValueUtils.deleteByExtenable(Arrays.asList(t.getId()), PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
        PropertyValueUtils.addPropertyValue(Arrays.asList(t), PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
    }

    @Cacheable(value="orderItemDetail", keyGenerator="redisKeyGenerator")
    @Override
    public OrderItemDetail queryObjById(String id) {
    	OrderItemDetail detail = this.orderItemDetailMapper.selectByPrimaryKey(id);
        PropertyValueUtils.queryAndSetValue(Arrays.asList(detail), PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
        return detail;
    }

    @Cacheable(value="orderItemDetail", keyGenerator="redisKeyGenerator")
    @Override
    public List<OrderItemDetail> queryAllObjByExample(OrderItemDetailExample example) {
    	List<OrderItemDetail> list = this.orderItemDetailMapper.selectByExample(example);
        PropertyValueUtils.queryAndSetValue(list, PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
        return list;
    }

    @Cacheable(value="orderItemDetail", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<OrderItemDetail> queryObjByPage(OrderItemDetailExample example) {
        PageView<OrderItemDetail> pageView = example.getPageView();
        List<OrderItemDetail> list = this.orderItemDetailMapper.selectByExampleByPage(example);
        PropertyValueUtils.queryAndSetValue(list, PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
        pageView.setQueryResult(list);
        return pageView;
    }
    
    @Override
	public IOrderItemDetail queryObjByOrderItemId(String itemId) {
		OrderItemDetailExample example = new OrderItemDetailExample();
		example.createCriteria().andOrderItemIdEqualTo(itemId);
		example.setOrderByClause("ID ASC");
		List<OrderItemDetail> list = this.queryAllObjByExample(example);
		if (CollectionUtils.isEmpty(list)) {
			return null;
		}
		
		PropertyValueUtils.queryAndSetValue(list, PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
		return list.get(0);
	}

    @Transactional
	@Override
	public void deleteByOrderItemIds(List<String> orderItemIdList) {
		OrderItemDetailExample example = new OrderItemDetailExample();
		example.createCriteria().andOrderItemIdIn(orderItemIdList);
		
		List<OrderItemDetail> list = this.orderItemDetailMapper.selectByExample(example);
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		List<String> idList = list.stream().map(OrderItemDetail::getId).collect(Collectors.toList());
		PropertyValueUtils.deleteByExtenable(idList, PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
		
		this.orderItemDetailMapper.deleteByExample(example);
	}

	@Transactional
	@Override
	public void addAll(List<OrderItemDetail> detailList) {
		if (CollectionUtils.isEmpty(detailList)) {
			return;
		}
    	
    	List<OrderItemDetail> detailNewList = detailList.stream().map(detail->{
    		OrderItemDetail detailNew = (OrderItemDetail) detail;
    		if (StringUtils.isBlank(detailNew.getId())) {
    			detailNew.setId(UUIDGenerator.generateUUID());
			}
    		if (detailNew.getCreateTime() == null) {
				detailNew.setCreateTime(new Date());
			}
    		return detailNew;
    		
    	}).collect(Collectors.toList());
    	
		this.orderItemDetailMapper.insertBatch(detailNewList);
		PropertyValueUtils.addPropertyValue(detailNewList, PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
	}

    @CacheEvict(value={"orderItemDetail"}, allEntries=true)
    @Override
    public void deleteByExample(OrderItemDetailExample arg0) {
        Assert.isNotNull(arg0, "参数不能为空");
        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
        this.orderItemDetailMapper.deleteByExample(arg0);
        
    }

}