package com.els.base.inquiry.service.impl;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.IMould;
import com.els.base.inquiry.dao.InquiryMouldMapper;
import com.els.base.inquiry.entity.InquiryMould;
import com.els.base.inquiry.entity.InquiryMouldExample;
import com.els.base.inquiry.enumclass.PropertyDefTplType;
import com.els.base.inquiry.service.InquiryMouldService;
import com.els.base.inquiry.utils.PropertyValueUtils;
import com.els.base.utils.uuid.UUIDGenerator;

@Service("defaultInquiryMouldService")
public class InquiryMouldServiceImpl implements InquiryMouldService {
    @Resource
    protected InquiryMouldMapper inquiryMouldMapper;

    @Transactional
    @CacheEvict(value={"inquiryMould"}, allEntries=true)
    @Override
    public void addObj(InquiryMould t) {
    	this.setDefaultValue(t);
        this.inquiryMouldMapper.insertSelective(t);
        PropertyValueUtils.addPropertyValue(Arrays.asList(t), PropertyDefTplType.MOULD_TYPE.getCode());
    }

    @CacheEvict(value={"inquiryMould"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.inquiryMouldMapper.deleteByPrimaryKey(id);
        PropertyValueUtils.deleteByExtenable(Arrays.asList(id), PropertyDefTplType.MOULD_TYPE.getCode());
    }

    @CacheEvict(value={"inquiryMould"}, allEntries=true)
    @Override
    public void modifyObj(InquiryMould t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.inquiryMouldMapper.updateByPrimaryKeySelective(t);
        PropertyValueUtils.deleteByExtenable(Arrays.asList(t.getId()), PropertyDefTplType.MOULD_TYPE.getCode());
        PropertyValueUtils.addPropertyValue(Arrays.asList(t), PropertyDefTplType.MOULD_TYPE.getCode());
    }

    @Cacheable(value="inquiryMould", keyGenerator="redisKeyGenerator")
    @Override
    public InquiryMould queryObjById(String id) {
        InquiryMould mould = this.inquiryMouldMapper.selectByPrimaryKey(id);
        PropertyValueUtils.queryAndSetValue(Arrays.asList(mould), PropertyDefTplType.MOULD_TYPE.getCode());
        return mould;
    }

    @Cacheable(value="inquiryMould", keyGenerator="redisKeyGenerator")
    @Override
    public List<InquiryMould> queryAllObjByExample(InquiryMouldExample example) {
        List<InquiryMould> list = this.inquiryMouldMapper.selectByExample(example);
        PropertyValueUtils.queryAndSetValue(list, PropertyDefTplType.MOULD_TYPE.getCode());
        return list;
    }

    @Cacheable(value="inquiryMould", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<InquiryMould> queryObjByPage(InquiryMouldExample example) {
        PageView<InquiryMould> pageView = example.getPageView();
        List<InquiryMould> list = this.inquiryMouldMapper.selectByExampleByPage(example);
        PropertyValueUtils.queryAndSetValue(list, PropertyDefTplType.MOULD_TYPE.getCode());
        pageView.setQueryResult(list);
        return pageView;
    }
    
    @Transactional
    @CacheEvict(value={"inquiryMould"}, allEntries=true)
    @Override
	public void deleteBySupOrder(String supOrderId) {
    	Assert.isNotBlank(supOrderId, "订单id不能为空");
    	
		InquiryMouldExample example = new InquiryMouldExample();
		example.createCriteria().andSupOrderIdEqualTo(supOrderId);
		
		List<InquiryMould> mouldList = this.inquiryMouldMapper.selectByExample(example);
		List<String> mouldIdList = mouldList.stream().map(InquiryMould::getId).collect(Collectors.toList());
		
		PropertyValueUtils.deleteByExtenable(mouldIdList, PropertyDefTplType.MOULD_TYPE.getCode());
		
		this.inquiryMouldMapper.deleteByExample(example);
	}
    
    @Transactional
    @CacheEvict(value={"inquiryMould"}, allEntries=true)
	@Override
	public void deleteByPurOrder(String purOrderId) {
    	Assert.isNotBlank(purOrderId, "订单id不能为空");
    	
		InquiryMouldExample example = new InquiryMouldExample();
		example.createCriteria().andPurOrderIdEqualTo(purOrderId);
		
		List<InquiryMould> mouldList = this.inquiryMouldMapper.selectByExample(example);
		List<String> mouldIdList = mouldList.stream().map(InquiryMould::getId).collect(Collectors.toList());
		
		PropertyValueUtils.deleteByExtenable(mouldIdList, PropertyDefTplType.MOULD_TYPE.getCode());
		
		this.inquiryMouldMapper.deleteByExample(example);
	}

    @Transactional
    @CacheEvict(value={"inquiryMould"}, allEntries=true)
	@Override
	public void addAll(List<InquiryMould> mouldList) {
    	if (CollectionUtils.isEmpty(mouldList)) {
			return;
		}
    	
		mouldList.stream().forEach(mould-> this.setDefaultValue(mould));
		
		this.inquiryMouldMapper.insertBatch(mouldList);
		
		PropertyValueUtils.addPropertyValue(mouldList, PropertyDefTplType.MOULD_TYPE.getCode());
	}
    
    private void setDefaultValue(InquiryMould newMould){
    	if (StringUtils.isBlank(newMould.getId())) {
    		newMould.setId(UUIDGenerator.generateUUID());
		}
    	
		if (newMould.getCreateTime() == null) {
			newMould.setCreateTime(new Date());
		}
		
		if (!Constant.YES_INT.equals(newMould.getIsEnable())
				&& !Constant.NO_INT.equals(newMould.getIsEnable())) {
			newMould.setIsEnable(Constant.YES_INT);
		}
		
    }

    @Cacheable(value="inquiryMould", keyGenerator="redisKeyGenerator")
	@Override
	public List<? extends IMould> queryBySupOrderId(String supOrderId) {
    	InquiryMouldExample example = new InquiryMouldExample();
    	example.createCriteria().andSupOrderIdEqualTo(supOrderId);
    	
    	List<InquiryMould> list = this.inquiryMouldMapper.selectByExample(example);
		PropertyValueUtils.queryAndSetValue(list, PropertyDefTplType.MOULD_TYPE.getCode());
		
		return list;
	}

    @Cacheable(value="inquiryMould", keyGenerator="redisKeyGenerator")
	@Override
	public List<? extends IMould> queryByPurOrderId(String purOrderId) {
    	InquiryMouldExample example = new InquiryMouldExample();
    	example.createCriteria().andPurOrderIdEqualTo(purOrderId);
    	
    	List<InquiryMould> list = this.inquiryMouldMapper.selectByExample(example);
		PropertyValueUtils.queryAndSetValue(list, PropertyDefTplType.MOULD_TYPE.getCode());
		
		return list;
	}

    @CacheEvict(value={"inquiryMould"}, allEntries=true)
    @Override
    public void deleteByExample(InquiryMouldExample arg0) {
        Assert.isNotNull(arg0, "参数不能为空");
        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
        this.inquiryMouldMapper.deleteByExample(arg0);
        
    }

}