package com.els.base.inquiry.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.ITarget;
import com.els.base.inquiry.dao.InquiryWaitMaterialMapper;
import com.els.base.inquiry.entity.InquiryWaitMaterial;
import com.els.base.inquiry.entity.InquiryWaitMaterialExample;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.service.InquiryWaitMaterialService;
import com.els.base.inquiry.service.TemplateConfService;
import com.els.base.inquiry.web.vo.WaitMaterialTransformVo;
import com.els.base.utils.uuid.UUIDGenerator;

@Service("defaultInquiryWaitMaterialService")
public class InquiryWaitMaterialServiceImpl implements InquiryWaitMaterialService {
    @Resource
    protected InquiryWaitMaterialMapper inquiryWaitMaterialMapper;
    
    @Resource
    private TemplateConfService templateConfService;

    @CacheEvict(value={"inquiryWaitMaterial"}, allEntries=true)
    @Override
    public void addObj(InquiryWaitMaterial t) {
        this.inquiryWaitMaterialMapper.insertSelective(t);
    }

    @CacheEvict(value={"inquiryWaitMaterial"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.inquiryWaitMaterialMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"inquiryWaitMaterial"}, allEntries=true)
    @Override
    public void modifyObj(InquiryWaitMaterial t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.inquiryWaitMaterialMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="inquiryWaitMaterial", keyGenerator="redisKeyGenerator")
    @Override
    public InquiryWaitMaterial queryObjById(String id) {
        return this.inquiryWaitMaterialMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="inquiryWaitMaterial", keyGenerator="redisKeyGenerator")
    @Override
    public List<InquiryWaitMaterial> queryAllObjByExample(InquiryWaitMaterialExample example) {
        return this.inquiryWaitMaterialMapper.selectByExample(example);
    }

    @Cacheable(value="inquiryWaitMaterial", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<InquiryWaitMaterial> queryObjByPage(InquiryWaitMaterialExample example) {
        PageView<InquiryWaitMaterial> pageView = example.getPageView();
        pageView.setQueryResult(this.inquiryWaitMaterialMapper.selectByExampleByPage(example));
        return pageView;
    }

	@Override
	public List<ITarget> transformToPurOrderMaterial(WaitMaterialTransformVo waitMaterialTransformVo) {
		List<ITarget> targets = null;
		if (waitMaterialTransformVo != null && StringUtils.isNotBlank(waitMaterialTransformVo.getTemplateId())
				&& CollectionUtils.isNotEmpty(waitMaterialTransformVo.getInquiryWaitMaterials())) {
			TemplateConf templateConf = templateConfService.queryObjById(waitMaterialTransformVo.getTemplateId());
			String className = templateConf.getTargetClassName();
			targets = new ArrayList<>();
			try {
				// 通过反射创建目标对象
				Class<?> clazz = Class.forName(className);
				for (InquiryWaitMaterial inquiryWaitMaterial : waitMaterialTransformVo.getInquiryWaitMaterials()) {
					ITarget target = (ITarget)clazz.newInstance();
					target.buildFromWaitMaterial(waitMaterialTransformVo.getTemplateId(), inquiryWaitMaterial);
					targets.add(target);
				}
			} catch (ClassNotFoundException e) {
				throw new CommonException(String.format("转成询价单物料出错,不存在类[%s]", className), e);
			} catch (Exception e) {
				e.printStackTrace();
				throw new CommonException("待询价清单转成询价单物料失败");
			}
		}
		return targets;
	}
	
	@CacheEvict(value={"inquiryWaitMaterial"}, allEntries=true)
	@Override
	public void modifyByExample(InquiryWaitMaterial waitMaterial, InquiryWaitMaterialExample example) {
		this.inquiryWaitMaterialMapper.updateByExampleSelective(waitMaterial, example);
	}
	
	@CacheEvict(value={"inquiryWaitMaterial"}, allEntries=true)
	@Override
	@Transactional
	public void batchInsertWaitMaterial(List<InquiryWaitMaterial> waitMaterials) {
		if (CollectionUtils.isNotEmpty(waitMaterials)) {
			Date now = new Date();
			waitMaterials.forEach(waitMaterial -> {
				waitMaterial.setId(UUIDGenerator.generateUUID());
				waitMaterial.setIsCreateInquiryOrder(Constant.NO_INT);
				waitMaterial.setCreateTime(now);
			});
			inquiryWaitMaterialMapper.insertBatch(waitMaterials);
		}
	}

	@Transactional
	@CacheEvict(value={"inquiryWaitMaterial"}, allEntries=true)
        @Override
        public void addAll(List<InquiryWaitMaterial> list) {
	    if (CollectionUtils.isEmpty(list)) {
	            return;
	        }
	        list.stream().forEach(item->{
	            if (StringUtils.isBlank(item.getId())) {
	                item.setId(UUIDGenerator.generateUUID());
	                this.inquiryWaitMaterialMapper.insertSelective(item);
	            }
	        });
            
        }

	@CacheEvict(value={"inquiryWaitMaterial"}, allEntries=true)
        @Override
        public void deleteByExample(InquiryWaitMaterialExample arg0) {
	    Assert.isNotNull(arg0, "参数不能为空");
	        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
	        this.inquiryWaitMaterialMapper.deleteByExample(arg0);
            
        }
}