package com.els.base.inquiry.service.impl;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.inquiry.dao.PropertyDefMapper;
import com.els.base.inquiry.entity.PropertyDef;
import com.els.base.inquiry.entity.PropertyDefExample;
import com.els.base.inquiry.service.PropertyDefService;
import com.els.base.inquiry.service.TemplateConfService;
import com.els.base.inquiry.service.TplMouldDetailService;
import com.els.base.inquiry.service.TplOrderItemDetailService;
import com.els.base.utils.uuid.UUIDGenerator;

@Service("defaultPropertyDefService")
public class PropertyDefServiceImpl implements PropertyDefService {
    @Resource
    protected PropertyDefMapper propertyDefMapper;
    
    @Resource
    protected TemplateConfService templateConfService;
    
    @Resource
    protected TplMouldDetailService tplMouLdDetailService;
    @Resource
    protected TplOrderItemDetailService tplOrderItemDetailService;

    @CacheEvict(value={"propertyDef"}, allEntries=true)
    @Override
    public void addObj(PropertyDef t) {
    	Assert.isNotBlank(t.getCode(), "字段的编码不能为空");
    	Assert.isNotBlank(t.getName(), "字段的名称不能为空");
    	
    	if (!t.getCode().matches("[\\w_-]+")) {
			throw new CommonException("项目代码只能使用英文、数字、下划线、横线");
		}
    	
    	if (t.getCode().length() > 32) {
    		throw new CommonException("编码最多只支持32个字符");
			
		}
    	
        this.propertyDefMapper.insertSelective(t);
    }

    @CacheEvict(value={"propertyDef"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.propertyDefMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"propertyDef"}, allEntries=true)
    @Override
    public void modifyObj(PropertyDef t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.propertyDefMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="propertyDef", keyGenerator="redisKeyGenerator")
    @Override
    public PropertyDef queryObjById(String id) {
        return this.propertyDefMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="propertyDef", keyGenerator="redisKeyGenerator")
    @Override
    public List<PropertyDef> queryAllObjByExample(PropertyDefExample example) {
        return this.propertyDefMapper.selectByExample(example);
    }

    @Cacheable(value="propertyDef", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<PropertyDef> queryObjByPage(PropertyDefExample example) {
        PageView<PropertyDef> pageView = example.getPageView();
        pageView.setQueryResult(this.propertyDefMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Cacheable(value="propertyDef", keyGenerator="redisKeyGenerator")
	@Override
	public List<PropertyDef> queryPropertyDefList(String templateId, String tplType) {
		PropertyDefExample example = new PropertyDefExample();
		example.createCriteria()
			.andTplIdEqualTo(templateId)
			.andTplTypeEqualTo(tplType);
		
		example.setOrderByClause("SORT_NO ASC");
		return this.queryAllObjByExample(example);
	}
    
    @Cacheable(value="propertyDef", keyGenerator="redisKeyGenerator")
    @Override
	public List<PropertyDef> queryHisPropertyDefList(String templateId, String tplType) {
    	PropertyDefExample example = new PropertyDefExample();
		example.createCriteria()
			.andTplIdEqualTo(templateId)
			.andTplTypeEqualTo(tplType);
		return this.propertyDefMapper.selectHisByExample(example);
	}

    @CacheEvict(value={"propertyDef"}, allEntries=true)
	@Override
	public void removeAllByTpl(String tplId, String propertyTplTypeCod) {
		Assert.isNotNull(tplId, "模板id不能为空");
		Assert.isNotNull(tplId, "模板类型不能为空");
		
		PropertyDefExample example = new PropertyDefExample();
		example.createCriteria()
			.andTplIdEqualTo(tplId)
			.andTplTypeEqualTo(propertyTplTypeCod);
		
		this.propertyDefMapper.deleteByExample(example);
	}

    @CacheEvict(value={"propertyDef"}, allEntries=true)
	@Override
	public void moveAllToHisByTpl(String tplId, String propertyTplTypeCod) {
		Assert.isNotNull(tplId, "模板id不能为空");
		Assert.isNotNull(tplId, "模板类型不能为空");
		
		PropertyDefExample example = new PropertyDefExample();
		example.createCriteria()
			.andTplIdEqualTo(tplId)
			.andTplTypeEqualTo(propertyTplTypeCod);
		
		this.propertyDefMapper.insertHisByExample(example);
		this.propertyDefMapper.deleteByExample(example);
		
	}

        @CacheEvict(value={"propertyDef"}, allEntries=true)
	@Override
	public void addAll(List<PropertyDef> propertyDefList) {
		Assert.isNotEmpty(propertyDefList, "字段定义不能为空");
		for (PropertyDef t : propertyDefList) {
			Assert.isNotBlank(t.getCode(), "字段的编码不能为空");
	    	Assert.isNotBlank(t.getName(), "字段的名称不能为空");
	    	t.setId(UUIDGenerator.generateUUID());
		}
		
		this.propertyDefMapper.insertBatch(propertyDefList);
	}

    @CacheEvict(value={"propertyDef"}, allEntries=true)
    @Override
    public void deleteByExample(PropertyDefExample arg0) {
        Assert.isNotNull(arg0, "参数不能为空");
        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
        this.propertyDefMapper.deleteByExample(arg0);
        
    }

}