package com.els.base.inquiry.service.impl;

import java.util.List;

import javax.annotation.Resource;

import org.activiti.engine.runtime.ProcessInstance;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.inquiry.dao.PurOrderMapper;
import com.els.base.inquiry.entity.PurOrder;
import com.els.base.inquiry.entity.PurOrderExample;
import com.els.base.inquiry.enumclass.InquiryOrderStatus;
import com.els.base.inquiry.event.ApprovePassListener;
import com.els.base.inquiry.service.PurOrderService;
import com.els.base.utils.uuid.UUIDGenerator;
import com.els.base.workflow.common.entity.ProcessStartVO;
import com.els.base.workflow.common.service.WorkFlowService;

@Service("defaultPurOrderService")
public class PurOrderServiceImpl implements PurOrderService {
	private static Logger logger = LoggerFactory.getLogger(PurOrderServiceImpl.class);
	
    @Resource
    protected PurOrderMapper purOrderMapper;
    
    @Resource
    private WorkFlowService workFlowService;
    
    @CacheEvict(value={"purOrder"}, allEntries=true)
    @Override
    public void addObj(PurOrder t) {
        this.purOrderMapper.insertSelective(t);
    }

    @CacheEvict(value={"purOrder"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.purOrderMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"purOrder"}, allEntries=true)
    @Override
    public void modifyObj(PurOrder t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.purOrderMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="purOrder", keyGenerator="redisKeyGenerator")
    @Override
    public PurOrder queryObjById(String id) {
        return this.purOrderMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="purOrder", keyGenerator="redisKeyGenerator")
    @Override
    public List<PurOrder> queryAllObjByExample(PurOrderExample example) {
        return this.purOrderMapper.selectByExample(example);
    }

    @Cacheable(value="purOrder", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<PurOrder> queryObjByPage(PurOrderExample example) {
        PageView<PurOrder> pageView = example.getPageView();
        pageView.setQueryResult(this.purOrderMapper.selectByExampleByPage(example));
        return pageView;
    }

	@CacheEvict(value={"purOrder"}, allEntries=true)
	@Override
	public void transformToUnPublish(List<String> ids) {
		logger.info("将询价单状态改成未发布");
		this.modifyOrderStatusByIds(ids, InquiryOrderStatus.UNPUBLISHED);
	}
	
	private void modifyOrderStatusByIds(List<String> ids, InquiryOrderStatus status){
		Assert.isNotEmpty(ids, "订单id列表不能为空");
		
		PurOrderExample purOrderExample = new PurOrderExample();
		purOrderExample.createCriteria().andIdIn(ids);
		
		PurOrder record = new PurOrder();
		record.setInquiryOrderStatus(status.getCode());
		this.purOrderMapper.updateByExampleSelective(record, purOrderExample);
	}

	@CacheEvict(value={"purOrder"}, allEntries=true)
	@Override
	public void cancelAllPurOrders(List<String> ids) {
		this.modifyOrderStatusByIds(ids, InquiryOrderStatus.INVALID);
		
	}

	@CacheEvict(value={"purOrder"}, allEntries=true)
	@Override
	public void deleteAllPurOrders(List<String> ids) {
		this.modifyOrderStatusByIds(ids, InquiryOrderStatus.DELETE);
	}

	@Override
	public void sendToApprove(PurOrder purOrder) {
		ProcessStartVO vo = ProcessStartVO.newInstance("xjdgl", purOrder.getOrderNo(), purOrder.getId(),
				null);
		vo.setListenerClass(ApprovePassListener.class);
		ProcessInstance startProcess=this.workFlowService.startProcess(vo);
		if (startProcess == null) {
			throw new CommonException("启动审批流程失败");
		}
	}

	@Transactional
	@CacheEvict(value={"purOrder"}, allEntries=true)
        @Override
        public void addAll(List<PurOrder> list) {
	    if (CollectionUtils.isEmpty(list)) {
	            return;
	        }
	        list.stream().forEach(item->{
	            if (StringUtils.isBlank(item.getId())) {
	                item.setId(UUIDGenerator.generateUUID());
	                this.purOrderMapper.insertSelective(item);
	            }
	        });
            
        }

	@CacheEvict(value={"purOrder"}, allEntries=true)
        @Override
        public void deleteByExample(PurOrderExample arg0) {
	    Assert.isNotNull(arg0, "参数不能为空");
	        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
	        this.purOrderMapper.deleteByExample(arg0);
            
        }

}