package com.els.base.inquiry.utils;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.inquiry.IBusiCondition;
import com.els.base.inquiry.IMould;
import com.els.base.inquiry.IOrderItem;
import com.els.base.inquiry.entity.PropertyDef;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.entity.TplMouldDetail;
import com.els.base.inquiry.entity.TplOrderItemDetail;
import com.els.base.inquiry.enumclass.PropertyDefTplType;
import com.els.base.inquiry.enumclass.ReadWriteTypeEnum;
import com.els.base.inquiry.service.PropertyDefService;
import com.els.base.utils.SpringContextHolder;

import io.swagger.annotations.ApiModelProperty;

/**
 * 属性定义的工具类，获取Targe或者ItemTemplate的属性
 * @author hzy
 *
 */
public abstract class PropertyDefUtils {
	
	/**
	 * 以类名作为key，属性定义list作为value
	 */
	private static Map<String, List<PropertyDef>> map = new HashMap<>();

	@SuppressWarnings("rawtypes")
	public static  List<PropertyDef> getProperyDefByClass(Class clazz){
		String name = clazz.getSimpleName();
		if (map.get(name) != null) {
			return map.get(name);
		}
		
		List<Field> fieldsList = new ArrayList<>();
		fieldsList.addAll( Arrays.asList(clazz.getDeclaredFields()));
		
		List<Field> supFieldList = new ArrayList<>();
		supFieldList.addAll(Arrays.asList(clazz.getSuperclass().getDeclaredFields()));
		
		List<Field> removedList = new ArrayList<>();
		for(Field supField : supFieldList){
			for(Field field: fieldsList){
				if (field.getName().equals(supField.getName())
						&& field.getType().equals(supField.getType())) {
					removedList.add(supField);
				}
			}
		}
		
		supFieldList.removeAll(removedList);
		fieldsList.addAll(supFieldList);
		
		List<PropertyDef> itemPropertyDefs = new ArrayList<>();
		
		for (Field field : fieldsList) {
			// 获取属性上的指定类型的注解
			ApiModelProperty annotation = field.getAnnotation(ApiModelProperty.class);
			
			if (annotation == null) {
				continue;
			}
			
			if (annotation.hidden()) {
				continue;
			}
			
			PropertyDef propertyDef = new PropertyDef();
			propertyDef.setName(StringUtils.defaultIfBlank(annotation.value(), annotation.name()));
			propertyDef.setCode(field.getName());
			propertyDef.setWriteType(ReadWriteTypeEnum.BOTH_TYPE.getCode());
			propertyDef.setReadType(ReadWriteTypeEnum.BOTH_TYPE.getCode());
			propertyDef.setIsPrimitive(Constant.YES_INT);
			
			propertyDef.setSortNo(annotation.position());
			
			if (StringUtils.isNotBlank(annotation.dataType())) {
				propertyDef.setType(annotation.dataType());
			}else{
				propertyDef.setType(getType(field.getType()));
			}
			
			if(StringUtils.isNotBlank(annotation.notes())){
				propertyDef.setDicGroupCode(annotation.notes());
			}
			
			if (propertyDef.getName().contains("%") && "number".equals(propertyDef.getType())) {
				propertyDef.setLength(2);
			}
			
			itemPropertyDefs.add(propertyDef);
		}
		
		itemPropertyDefs.sort(Comparator.comparing(PropertyDef::getSortNo));
		for (int index = 0; index < itemPropertyDefs.size(); index++) {
			itemPropertyDefs.get(index).setSortNo((index + 1));
		}
		
		map.put(name, itemPropertyDefs);
		return itemPropertyDefs;
	}
	
	public static String getType(Class<?> clazz){
		String typeName = clazz.getSimpleName().toLowerCase();
		switch (typeName) {
		case "integer":
		case "long":
		case "short":
			return "int";
		case "double":
		case "float":
		case "bigdecimal":
			return "number";
			
		case "string":
		case "boolean":
		case "date":
		default:
			return typeName;
		}
	}

	@SuppressWarnings("rawtypes")
	public static List<PropertyDef> merge(Class clazz, List<PropertyDef> propertyDefList) {
		
		//找出 propertyDefList里面没有的，class的字段
		List<PropertyDef> newPropertyDef = getProperyDefByClass(clazz).stream()
			.filter(classPropertyDef->{
				return propertyDefList.stream().noneMatch(propertyDef ->{
					return classPropertyDef.getCode().equals(propertyDef.getCode());
				});
			})
			.collect(Collectors.toList());
		
		propertyDefList.addAll(newPropertyDef);
		
		return propertyDefList;
	}
	
	/**
	 * 根据模板，查询询价单行的自定义字段
	 * @param templateConf
	 * @return
	 */
	public static List<PropertyDef> queryOrderItemProDef(TemplateConf templateConf){
		Assert.isNotBlank(templateConf.getId(), "templateConf id 不能为空");
		Assert.isNotNull(templateConf.getOrderItemClass(), "getOrderItemClass 不能为空");
		
		PropertyDefService propertyDefService = SpringContextHolder.getOneBean(PropertyDefService.class);
		List<PropertyDef> orderItemPropertyDefList = propertyDefService.queryPropertyDefList(templateConf.getId(), PropertyDefTplType.ORDER_ITEM.getCode());
		
		Class<? extends IOrderItem > orderItemClass = templateConf.getOrderItemClass();
    	return PropertyDefUtils.merge(orderItemClass, orderItemPropertyDefList);
	}
	
	/**
	 * 根据模板，查询的自定义字段
	 * @param templateConf
	 * @return
	 */
	public static List<PropertyDef> queryMouldProDef(TemplateConf templateConf){
		Assert.isNotBlank(templateConf.getId(), "templateConf id 不能为空");
		Assert.isNotNull(templateConf.getMouldClass(), "getMouldClass 不能为空");
		
		PropertyDefService propertyDefService = SpringContextHolder.getOneBean(PropertyDefService.class);
		List<PropertyDef> orderItemPropertyDefList = propertyDefService.queryPropertyDefList(templateConf.getId(), PropertyDefTplType.MOULD_TYPE.getCode());
		
		Class<? extends IMould> orderItemClass = templateConf.getMouldClass();
		return PropertyDefUtils.merge(orderItemClass, orderItemPropertyDefList);
	}
	
	/**
	 * 根据模板，查询配置的商务条件的自定义字段
	 * @param templateConf
	 * @return
	 */
	public static List<PropertyDef> queryBusiConditionProDef(TemplateConf templateConf){
		Assert.isNotBlank(templateConf.getId(), "templateConf id 不能为空");
		Assert.isNotNull(templateConf.getBusiConditionClass(), "getBusiConditionClass 不能为空");
		
		PropertyDefService propertyDefService = SpringContextHolder.getOneBean(PropertyDefService.class);
		List<PropertyDef> orderItemPropertyDefList = propertyDefService.queryPropertyDefList(templateConf.getId(), PropertyDefTplType.BUSI_TYPE.getCode());
		
		Class<? extends IBusiCondition> busiCoditionCls = templateConf.getBusiConditionClass();
		return PropertyDefUtils.merge(busiCoditionCls, orderItemPropertyDefList);
	}
	
	/**
	 * 根据物料成本清单模板，查询自定义字段
	 * @param tplOrderItemDetail
	 * @return
	 */
	public static List<PropertyDef> queryOrderItemDetailProDef(TplOrderItemDetail tplOrderItemDetail){
		Assert.isNotBlank(tplOrderItemDetail.getId(), "templateConf id 不能为空");
		Assert.isNotNull(tplOrderItemDetail.getOrderItemDetailClass(), "getOrderItemDetailClass 不能为空");
		
		PropertyDefService propertyDefService = SpringContextHolder.getOneBean(PropertyDefService.class);
		List<PropertyDef> orderItemPropertyDefList = propertyDefService.queryPropertyDefList(tplOrderItemDetail.getId(), PropertyDefTplType.ORDER_DETAIL_TYPE.getCode());
		
		Class<?> orderItemClass = tplOrderItemDetail.getOrderItemDetailClass();
		return PropertyDefUtils.merge(orderItemClass, orderItemPropertyDefList);
	}
	
	/**
	 * 根据模具成本清单模板，查询自定义字段
	 * @param tplMouldDetail
	 * @return
	 */
	public static List<PropertyDef> queryMouldDetailProDef(TplMouldDetail tplMouldDetail){
		Assert.isNotBlank(tplMouldDetail.getId(), "templateConf id 不能为空");
		Assert.isNotNull(tplMouldDetail.getMouldDetailClass(), " getMouldDetailClass 不能为空");
		
		PropertyDefService propertyDefService = SpringContextHolder.getOneBean(PropertyDefService.class);
		List<PropertyDef> orderItemPropertyDefList = propertyDefService.queryPropertyDefList(tplMouldDetail.getId(), PropertyDefTplType.MOULD_DETAIL_TYPE.getCode());
		
		Class<?> orderItemClass = tplMouldDetail.getMouldDetailClass();
		return PropertyDefUtils.merge(orderItemClass, orderItemPropertyDefList);
	}
	
	public static void valid(List<PropertyDef> propertyDefList) {
		if (CollectionUtils.isEmpty(propertyDefList)) {
			return;
		}
		
		propertyDefList.stream().forEach(t->{
			Assert.isNotBlank(t.getCode(), "字段的编码不能为空");
	    	Assert.isNotBlank(t.getName(), "字段的名称不能为空");
	    	
	    	if (!t.getCode().matches("[\\w_-]+")) {
				throw new CommonException("项目代码只能使用英文、数字、下划线、横线");
			}
	    	
	    	if (t.getCode().length() > 32) {
	    		throw new CommonException("项目代码最多只支持32个字符");
			}
		});
		
		//统计每个code的值
    	Map<String, Long> codeAndProDefList = propertyDefList.stream()
    		.collect(Collectors.groupingBy(PropertyDef::getCode, Collectors.counting()));
		//如果有重复就抛异常
    	codeAndProDefList.entrySet().stream()
    		.filter(entry-> entry.getValue() > 1)
    		.findAny()
    		.ifPresent(entry->{
    			String errMsg = String.format("项目代码[%s]重复了", entry.getKey());
    			throw new CommonException(errMsg);
    		});
	}
}
