package com.els.base.inquiry.utils;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.time.DateUtils;

import com.els.base.core.exception.CommonException;
import com.els.base.inquiry.IExtendable;
import com.els.base.inquiry.entity.PropertyDef;
import com.els.base.inquiry.entity.PropertyValue;
import com.els.base.inquiry.entity.PropertyValueExample;
import com.els.base.inquiry.service.PropertyValueService;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.excel.ConverterFactory;
import com.els.base.utils.excel.StrToObjConverter;
import com.els.base.utils.json.JsonUtils;
import com.els.base.utils.reflect.ReflectUtils;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;

public abstract class PropertyValueUtils {
	
	public static String validStr(String valueStr, String typeName, String proCode){
		switch (typeName) {
		case "string":
		case "dic_group":
			return valueStr;
		case "boolean":
			if (StringUtils.isBlank(valueStr)) {
				return null;
			}
			
			valueStr = valueStr.trim().toLowerCase();
			if (StringUtils.isNotBlank(valueStr)
					&& !"true".equals(valueStr) 
					&& !"false".equals(valueStr)) {
				throw new CommonException(String.format("数据的类型是boolean,输入的格式有误。属性名[%s],属性值[%s]", proCode, valueStr));
			}
			return valueStr;
			
		case "number":
			if(StringUtils.isBlank(valueStr)){
				return null;
			}
			
			valueStr = valueStr.trim();
			if (!valueStr.matches("^\\d+(\\.\\d+)?$")) {
				throw new CommonException(String.format("数据的类型是number,输入的格式有误。属性名[%s],属性值[%s]", proCode, valueStr));
			}
			return valueStr;
			
		case "date":
			if(StringUtils.isBlank(valueStr)){
				return null;
			}
			valueStr = valueStr.trim();
			if (!valueStr.matches("\\d+")
					&& !valueStr.matches("\\d+-\\d+-\\d+\\s\\d+:\\d+:\\d+")
					&& !valueStr.matches("\\d+-\\d+-\\d+")) {
				throw new CommonException(String.format("数据的类型是date,输入的格式有误。属性名[%s],属性值[%s]", proCode, valueStr));
			}
			return valueStr;
				
		default:
			return valueStr;
		}
	}
	
	/**
	 * 解析字符串的值为对象值
	 * @param valueStr
	 * @param typeName
	 * @return
	 */
	public static Object parseStr(String valueStr, String typeName, String proCode){
		if (valueStr == null) {
			return valueStr;
		}
		
		switch (typeName) {
		case "string":
		case "dic_group":
			return valueStr;
			
		case "boolean":
			valueStr = valueStr.trim().toLowerCase();
			if("true".equals(valueStr) 
					|| "false".equals(valueStr)){
				return Boolean.valueOf(valueStr);
			}
			return Boolean.FALSE;
			
		case "number":
			valueStr = valueStr.trim();
			if(StringUtils.isBlank(valueStr)){
				return null;
			}
			return new BigDecimal(valueStr);
			
		case "date":
			if(StringUtils.isBlank(valueStr)){
				return null;
			}
			
			valueStr = valueStr.trim();
			try {
				if (valueStr.matches("\\d+")) {
					return new Date(Long.valueOf(valueStr));
				}
				if (valueStr.matches("\\d+-\\d+-\\d+\\s\\d+:\\d+:\\d+")) {
						return DateUtils.parseDate(valueStr, "yyyy-mm-dd hh:MM:ss");
				}
				if (valueStr.matches("\\d+-\\d+-\\d+")) {
					return DateUtils.parseDate(valueStr, "yyyy-mm-dd");
				}
			} catch (ParseException e) {
				throw new RuntimeException(String.format("属性解析失败, 属性[%s],值[%s]", proCode, valueStr), e);
			}
				
		default:
			return null;
		}
	}
	
	/**
	 * 给物料清单添加自定义的值
	 * @param extendableList
	 */
	public static void queryAndSetValue(List<? extends IExtendable> extendableList, String tplType){
		if (CollectionUtils.isEmpty(extendableList)) {
			return;
		}
		
		//1、查询出所有的自定义字段的值
    	List<String> targetIdList = extendableList.stream().map(IExtendable::getId).collect(Collectors.toList());
        PropertyValueExample propertyValueExample = new PropertyValueExample();
        propertyValueExample.createCriteria()
        	.andRefIdIn(targetIdList)
        	.andTplTypeEqualTo(tplType);
        
        PropertyValueService propertyValueService = SpringContextHolder.getOneBean(PropertyValueService.class);
        List<PropertyValue> propertyValueList = propertyValueService.queryAllObjByExample(propertyValueExample);
        if (CollectionUtils.isEmpty(propertyValueList)) {
			return;
		}	
        
        //2、根据refId进行分组
        Map<String, List<PropertyValue>> refIdAndValueMap = propertyValueList.stream().collect(Collectors.groupingBy(PropertyValue::getRefId));
        
        //3、每个实体，根据refId找到值，并设置到实体里面
        extendableList.forEach(target->{
        	target.setPropertyValueList(refIdAndValueMap.get(target.getId()));
        });
    }
	
	/**
	 * 批量删除自定义字段的值
	 * @param extendableList
	 * @param tplType
	 */
	public static void deleteByExtenable(List<String> idList, String tplType){
		if (CollectionUtils.isEmpty(idList)) {
			return;
		}
		
        PropertyValueExample propertyValueExample = new PropertyValueExample();
        propertyValueExample.createCriteria()
        	.andRefIdIn(idList)
        	.andTplTypeEqualTo(tplType);
        
        PropertyValueService propertyValueService = SpringContextHolder.getOneBean(PropertyValueService.class);
        propertyValueService.deleteByExample(propertyValueExample);
	}
	
	/**
	 * 批量插入自定义字段的值
	 * @param extendables
	 * @param tplType
	 */
	public static void addPropertyValue(List<? extends IExtendable> extendables, String tplType){
		if (CollectionUtils.isEmpty(extendables)) {
			return;
		}
		
		List<PropertyValue> totalPropertyValueList = extendables.stream()
			.filter(target->{
				return CollectionUtils.isNotEmpty(target.getPropertyValueList());
				
			}).flatMap(target->{
				target.getPropertyValueList().forEach(propertyValue->{
					propertyValue.setRefId(target.getId());
		        	propertyValue.setTplType(tplType);
				});
				
				return target.getPropertyValueList().stream();
			}).collect(Collectors.toList());
		
		//TODO 暂时无法查明的问题，totalPropertyValueList自动复制出多个PropertyValue
    	totalPropertyValueList = totalPropertyValueList.stream().distinct().collect(Collectors.toList());
		
		PropertyValueService propertyValueService = SpringContextHolder.getOneBean(PropertyValueService.class);
        propertyValueService.addAll(totalPropertyValueList);
	}
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public static void setValueToInstance(Object instance, Field field, String valueStr) throws Exception {
		if (valueStr == null
				|| "null".equals(valueStr.toLowerCase())) {
			return;
		}
		
		StrToObjConverter converter = ConverterFactory.getDefaultToObjConverter(field.getType());
		if (converter != null) {
			ReflectUtils.setValue(instance, field.getName(), converter.convert(valueStr, null, null));
			return;
		}
		
		if (field.getType().equals(String.class)) {
			ReflectUtils.setValue(instance, field.getName(), valueStr);
			return;
		}
		
		if (StringUtils.isBlank(valueStr)) {
			return;
		}
		
		if (field.getType().equals(List.class)) {
			
			Type genericType = field.getGenericType();   
			if(genericType == null) {
				ReflectUtils.setValue(instance, field.getName(), JsonUtils.convertValue(valueStr, field.getType()));
				
			}else if(!(genericType instanceof ParameterizedType)){
				ReflectUtils.setValue(instance, field.getName(), JsonUtils.convertValue(valueStr, field.getType()));
				
		    }else{
				//得到泛型里的class类型对象    
				Class<?> genericClazz = (Class<?>)((ParameterizedType) genericType).getActualTypeArguments()[0];   
				JavaType javaType = JsonUtils.getObjectMapper().getTypeFactory().constructParametrizedType(List.class, null, genericClazz);
				Object valueObj = JsonUtils.getObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).readValue(valueStr.toString(), javaType);
				ReflectUtils.setValue(instance, field.getName(), valueObj);
			}
			
		}else {
			ReflectUtils.setValue(instance, field.getName(), JsonUtils.convertValue(valueStr, field.getType()));
		}
	}
	
	public static <T extends IExtendable> List<PropertyValue> getDefaultPropertyValue(T value){
		List<Field> fieldList = ClassReflectionUtils.getPrimitiveField(value.getClass());
		List<PropertyDef> defList = value.getPropertyDefList();
		
		defList = defList.stream()
			.filter(def -> fieldList.stream().noneMatch(field-> field.getName().equals(def.getCode())))
			.collect(Collectors.toList());
		
		List<PropertyValue> temp2 = defList.stream()
				.map(proDef->{
					PropertyValue proValue = proDef.buildValue();
					proValue.setRefId(value.getId());
					return proValue;
				})
				.collect(Collectors.toList());
		
		return temp2;
	}

}
