package com.els.base.inquiry.utils.excel;

import java.math.BigDecimal;
import java.text.DateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;

import com.els.base.company.entity.Company;
import com.els.base.core.entity.dictionary.DicGroupItem;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.dictonary.DicUtils;
import com.els.base.inquiry.IMould;
import com.els.base.inquiry.IMouldDetail;
import com.els.base.inquiry.IOrderItem;
import com.els.base.inquiry.IOrderItemDetail;
import com.els.base.inquiry.entity.InquirySupOrder;
import com.els.base.inquiry.entity.PropertyDef;
import com.els.base.inquiry.entity.PropertyValue;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryQuoteLadderType;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;
import com.els.base.inquiry.enumclass.PropertyDefType;
import com.els.base.inquiry.enumclass.ReadWriteTypeEnum;
import com.els.base.inquiry.utils.json.DetailSerialzer;
import com.els.base.utils.SpringContextHolder;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.entity.TemplateExportParams;

public class OrderExcelUtils {
	
	private static final Pattern NUM_PATTERN = Pattern.compile("\\d+");
	private static final Pattern DATE_PATTERN = Pattern.compile("\\d+-\\d+-\\d+");
	
	private ExportParam param;
	private TemplateExportParams tplParams;
	
	private static List<String> ignoreList = Arrays.asList("historicalPrice","materiel","materialMouldList","inquiryQuoteLadders","orderItemDetail","mouldDetail");

	public static Workbook exportToExcel(ExportParam param) {
		String path = SpringContextHolder.getProperty("inquiry.print.template.path");
		Assert.isNotBlank(path, "导出模板路径不存在，无法导出");
		
		TemplateExportParams tplParams = new TemplateExportParams(path, 0, 1);
		OrderExcelUtils utils = new OrderExcelUtils(param, tplParams);
		Workbook workbook = utils.exportExcel();
		
		return workbook;
	}
	
	private Workbook exportExcel() {
		Map<String, Object> tplMap = new HashMap<>();
		tplMap.put("supOrder", param.getSupOrder());
		tplMap.put("templateConf", param.getTemplateConf());
		if (Constant.YES_INT.equals(param.getTemplateConf().getIsBusiCondEnable())) {
			tplMap.put("busiCondition", param.getSupOrder().getBusiConditions().get(0));
		}
		tplMap.put("supCompany", param.getSupCompany());
		tplMap.put("purUser", param.getPurUser());
		tplMap.put("other", param.getOther());
		
		Workbook workbook = ExcelExportUtil.exportExcel(this.tplParams, tplMap);
		if (workbook == null) {
			throw new CommonException("导出异常:excel生成失败");
		}
		exportOrderItem(workbook, param.getSupOrder(), param.getTemplateConf());
		//exportOrderItemDetail(workbook, param.getSupOrder(), param.getTemplateConf());
		exportMould(workbook, param.getSupOrder(), param.getTemplateConf());
		//exportMouldDetail(workbook, param.getSupOrder(), param.getTemplateConf());
		return workbook;
	}

	private CellStyle buildDefaultStyle(Workbook workbook) {
		CellStyle style = workbook.createCellStyle(); 
		style.setBorderBottom(BorderStyle.THIN);
		style.setBorderLeft(BorderStyle.THIN);
		style.setBorderRight(BorderStyle.THIN);
		style.setBorderTop(BorderStyle.THIN);
		return style;
	}

	/**
	 * 导出模具成本清单
	 * @param workbook
	 * @param supOrder
	 * @param templateConf
	 */
	private void exportMouldDetail(Workbook workbook, InquirySupOrder supOrder, TemplateConf templateConf) {
		if (!Constant.YES_INT.equals(templateConf.getIsMouldEnable())
				|| !Constant.YES_INT.equals(templateConf.getIsMouldDetailEnable())) {
			return;
		}
		
		int startColumnIndex = 0;
		int startRowIndex = 1;
		int sheetIndex = 3;
		
		List<PropertyDef> defList = templateConf.getTplMouLdDetail().getPropertyDefList();
		
		Sheet sheet = workbook.getSheetAt(sheetIndex);
		Row startRow = sheet.getRow(startRowIndex);
		CellStyle style = buildDefaultStyle(workbook);
		
		defList = this.filteProDef(defList);
		this.writeHeader(defList, startRow, startColumnIndex, style);
		
		List<IMouldDetail> detailList = supOrder.getMouldList().stream()
			.map(mould->mould.getMouldDetail())
			.collect(Collectors.toList());
		
		if (CollectionUtils.isEmpty(detailList)) {
			return;
		}
		
		for(int i=0; i<detailList.size(); i++){
			IMouldDetail detail = detailList.get(i);
			int currentRowIndex = startRowIndex + i +1;
			
			Row newRow = this.createRow(sheet, currentRowIndex);
			
			List<PropertyValue> valueList = new DetailSerialzer<>().getProperyValueClass(detail);
			this.writeItem(defList, valueList, newRow, startColumnIndex,  i + 1, style);
		}
	}

	/**
	 * 导出物料成本组成清单
	 * @param workbook
	 * @param supOrder
	 * @param templateConf
	 */
	private void exportOrderItemDetail(Workbook workbook, InquirySupOrder supOrder, TemplateConf templateConf) {
		if (!Constant.YES_INT.equals(templateConf.getIsOrderItemDetailEnable())) {
			return;
		}
		
		int startColumnIndex = 0;
		int startRowIndex = 1;
		int sheetIndex = 1;
		
		List<PropertyDef> defList = templateConf.getTplOrderItemDetail().getPropertyDefList();
		
		Sheet sheet = workbook.getSheetAt(sheetIndex);
		Row startRow = sheet.getRow(startRowIndex);
		CellStyle style = buildDefaultStyle(workbook);
		
		defList = this.filteProDef(defList);
		this.writeHeader(defList, startRow, startColumnIndex, style);
		
		List<IOrderItemDetail> detailList = supOrder.getItemList().stream()
			.map(item-> item.getOrderItemDetail())
			.collect(Collectors.toList());
		
		if (CollectionUtils.isEmpty(detailList)) {
			return;
		}
		
		for(int i=0; i<detailList.size(); i++){
			IOrderItemDetail detail = detailList.get(i);
			int currentRowIndex = startRowIndex + i +1;
			
			Row newRow = this.createRow(sheet, currentRowIndex);
			
			List<PropertyValue> valueList = new DetailSerialzer<>().getProperyValueClass(detail);
			this.writeItem(defList, valueList, newRow, startColumnIndex,  i + 1, style);
		}
		
		
	}

	/**
	 * 导出模具清单
	 * @param workbook
	 * @param supOrder
	 * @param templateConf
	 */
	private void exportMould(Workbook workbook, InquirySupOrder supOrder, TemplateConf templateConf) {
		if (!Constant.YES_INT.equals(templateConf.getIsMouldEnable())) {
			return;
		}
		
		int startColumnIndex = 0;
		int startRowIndex = 15;
		int sheetIndex = 1;
		int conditionIndex = 20;
		
		List<PropertyDef> defList = templateConf.getMouldPropertyDefList();
		
		Sheet sheet = workbook.getSheetAt(sheetIndex);
		Row startRow = sheet.getRow(startRowIndex);
		CellStyle style = buildDefaultStyle(workbook);
		
		defList = this.filteProDef(defList);
		this.writeHeader(defList, startRow, startColumnIndex, style);
		
		List<IMould> mouldList = supOrder.getMouldList();
		if (CollectionUtils.isEmpty(mouldList)) {
			return;
		}
		// 动态生成商务条件
		List<PropertyDef> conditionDefList = templateConf.getBusiConditionPropertyDefList();
		List<PropertyValue> conditionDefValueList = new DetailSerialzer<>().getProperyValueClass(supOrder.getBusiConditions().get(0)); 
		this.writeBusiCondition(sheet, conditionDefList, conditionDefValueList, conditionIndex);
		
		BigDecimal sum = BigDecimal.ZERO;
		for(int i=0; i<mouldList.size(); i++){
			IMould mould = mouldList.get(i);
			int currentRowIndex = startRowIndex + i +1;
			
			Row newRow = this.createRow(sheet, currentRowIndex);
			
			List<PropertyValue> valueList = new DetailSerialzer<>().getProperyValueClass(mould);
			this.writeItem(defList, valueList, newRow, startColumnIndex,  i + 1, style);
			
			if (mould.getUntaxedUnitPrice() != null) {
				sum = sum.add(mould.getUntaxedUnitPrice());
			}
		}

		int priceIndex = this.getUntaxedUnitPrice(defList);
		int rowIndex = startRowIndex + mouldList.size() + 1;
		this.writeSum(sheet, sum, rowIndex, priceIndex + 1, defList.size(), style);

	}

	private List<PropertyValue> filteProValue(List<PropertyValue> valueList, List<PropertyDef> defList) {
		if (CollectionUtils.isEmpty(valueList) || CollectionUtils.isEmpty(defList)) {
			return valueList;
		}
		
		valueList = valueList.stream()
			.filter(value-> defList.stream().anyMatch(def-> def.getCode().equals(value.getCode())))
			.collect(Collectors.toList());
		
		valueList.stream().sorted(Comparator.comparing(PropertyValue::getSortNo));
		return valueList;
	}

	/**
	 * 导出物料清单
	 * @param workbook
	 * @param inquirySupOrder
	 * @param templateConf
	 */
	private void exportOrderItem(Workbook workbook, InquirySupOrder inquirySupOrder, TemplateConf templateConf){
		int startColumnIndex = 0;
		int startRowIndex = 13;
		int sheetIndex = 0;
		int conditionIndex = 17;
		
		List<PropertyDef> defList = templateConf.getOrderItemPropertyDefList();
		
		Sheet sheet = workbook.getSheetAt(sheetIndex);
		Row startRow = sheet.getRow(startRowIndex);
		CellStyle style = buildDefaultStyle(workbook);
		
		defList = this.filteProDef(defList);
		this.writeHeader(defList, startRow, startColumnIndex, style);
		
		List<IOrderItem> itemList = inquirySupOrder.getItemList();
		if (CollectionUtils.isEmpty(itemList)) {
			return;
		}
		// 动态生成商务条件
		List<PropertyDef> conditionDefList = templateConf.getBusiConditionPropertyDefList();
		List<PropertyValue> conditionDefValueList = new DetailSerialzer<>().getProperyValueClass(inquirySupOrder.getBusiConditions().get(0)); 
		this.writeBusiCondition(sheet, conditionDefList, conditionDefValueList, conditionIndex);
		
		BigDecimal sum = BigDecimal.ZERO;
		for(int i=0; i<itemList.size(); i++){
			IOrderItem item = itemList.get(i);
			int currentRowIndex = startRowIndex + i +1;
			
			Row newRow = this.createRow(sheet, currentRowIndex);
			
			List<PropertyValue> valueList = new DetailSerialzer<>().getProperyValueClass(item);
			this.writeItem(defList, valueList, newRow, startColumnIndex,  i + 1, style);
			
			if (item.getUntaxedUnitPrice() != null) {
				sum = sum.add(item.getUntaxedUnitPrice());
			}
		}

		int priceIndex = this.getUntaxedUnitPrice(defList);
		int rowIndex = startRowIndex + itemList.size() + 1;
		this.writeSum(sheet, sum, rowIndex, priceIndex + 1, defList.size(), style);
	}
	
	
	private void valueTransfer(PropertyValue proValue) {
		String value = proValue.getValueStr();
		if (value == null) {
			value = "";
		}
		
		if (PropertyDefType.DIC_GROUP.getCode().equals(proValue.getType())
				&& StringUtils.isNotBlank(proValue.getDicGroupCode())
				&& StringUtils.isNotBlank(proValue.getValueStr())) {
			DicGroupItem dicItem = DicUtils.getDicGroupItemByCode(proValue.getDicGroupCode(), proValue.getValueStr());
			value = dicItem == null ? value : dicItem.getName();
		}
		
		if (PropertyDefType.DATE.getCode().equals(proValue.getType()) ) {
			if (NUM_PATTERN.matcher(value).matches()) {
				Date date = new Date(Long.parseLong(value));
				DateFormat dateFormat = DateFormat.getDateInstance();
				value=dateFormat.format(date);
			}else if (!DATE_PATTERN.matcher(value).find()) {
				value = value.replaceAll("\\s*\\d+:\\d+:\\d+\\s*", "");
				value = value.contains("1970-01-01") ? "" : value;
				
			} else {
				Matcher matcher = DATE_PATTERN.matcher(value);
				if (matcher.find()) {
					value = matcher.group(0);
				}
			}
 		}
		
		if ("quotationStatus".equals(proValue.getCode())) {
			value = InquiryQuoteStatus.getNameByCode(proValue.getValueStr());
		}
		if ("quoteType".equals(proValue.getCode()) && StringUtils.isNumeric(proValue.getValueStr())) {
			value = InquiryQuoteLadderType.getNameByCode(Integer.valueOf(proValue.getValueStr()));
		}
		
		if ("operation".equals(proValue.getCode())
				|| "unableToQuote".equals(proValue.getCode())) {
			value = Constant.YES_INT.toString().equals(value) ? "是" : "否";
		}
		
		if ("erpSystemPriceComparison".equals(proValue.getCode())
				|| "targetPriceComparison".equals(proValue.getCode())) {
			value = StringUtils.isBlank(proValue.getValueStr()) ? "" : proValue.getValueStr() + "%";
		}
		
		proValue.setValueStr(value);
		
	}

	private void writeBusiCondition(Sheet sheet, List<PropertyDef> conditionDefList, List<PropertyValue> conditionValueList, int conditionIndex) {
		conditionDefList = this.filteProDef(conditionDefList);
		conditionValueList = this.filteProValue(conditionValueList, conditionDefList);
		
		conditionValueList = conditionValueList.stream()
				.filter(proValue -> ReadWriteTypeEnum.BOTH_TYPE.getCode().equals(proValue.getReadType())
						&& !ignoreList.contains(proValue.getCode())
						&& !PropertyDefType.BUTTON.getCode().equals(proValue.getType()))
				.collect(Collectors.toList());
		
		/**
		 * 待优化成lamda表达式
		 */
		for (int i=0; i<conditionDefList.size(); i++) {
			Row newRow = this.createRow(sheet, conditionIndex);
			conditionIndex++;
			
			PropertyDef propertyDef = conditionDefList.get(i);
			
			Cell cell1 = newRow.createCell(0);
			cell1.setCellValue(propertyDef.getDisplayName());
			
			Cell cell2 = newRow.createCell(1);
			PropertyValue proValue = this.filteProValue(conditionValueList, propertyDef);
			if (proValue != null) {
				this.valueTransfer(proValue);
				cell2.setCellValue(proValue.getValueStr());
			}
		}
		
	}

	private int getUntaxedUnitPrice(List<PropertyDef> defList) {
		for(int i=0; i<defList.size(); i++){
			if (defList.get(i).getCode().equals("untaxedUnitPrice")) {
				return i;
			}
		}
		return 0;
	}

	private void writeSum(Sheet sheet, BigDecimal sum, int rowIndex, int priceIndex, int rowMaxSize, CellStyle style) {
		Row row = sheet.getRow(rowIndex);
		// 去掉首列序号列
		for(int i=0; i<rowMaxSize+1; i++){
			Cell cellTmp = row.createCell(i);
			cellTmp.setCellStyle(style);
			cellTmp.setCellValue("");
			
			if (i == priceIndex) {
				cellTmp.setCellValue(sum.toString());
			}
			if (priceIndex - 1 >=0 && i==priceIndex-1) {
				cellTmp.setCellValue("小计：");
			}
		}
	}


	private Row createRow(Sheet sheet, int rowIndex) {
		int lastRowNo = sheet.getLastRowNum();  
        sheet.shiftRows(rowIndex, lastRowNo, 1);
        
		Row row = sheet.createRow(rowIndex);
		return row;
	}

	private void writeItem(List<PropertyDef> defList, List<PropertyValue> valueList, Row newRow, int startColunm, int indexNo, CellStyle style) {
		Cell indexCell = newRow.createCell(startColunm);
		indexCell.setCellValue(indexNo);
		indexCell.setCellStyle(style);
		
//		valueList = this.filteProValue(valueList, defList);
		
		for(int j=0; j<defList.size(); j++){
			Cell cellTmp = newRow.createCell(startColunm + j + 1);
			cellTmp.setCellStyle(style);
			
			PropertyValue proValue = this.filteProValue(valueList, defList.get(j));
			if (proValue != null) {
				this.valueTransfer(proValue);
				cellTmp.setCellValue(proValue.getValueStr());
			}
		}
		
		this.completeRow(newRow, style, valueList.size()+1);
	}
	
	private PropertyValue filteProValue(List<PropertyValue> valueList, PropertyDef propertyDef) {
		return valueList.stream()
			.filter(value-> propertyDef.getCode().equals(value.getCode()))
			.findAny()
			.orElse(null);
	}

	private void completeRow(Row newRow, CellStyle style, int dataSize){
		if (newRow.getLastCellNum() <= dataSize) {
			return ;
		}
		int lastCellIndex = newRow.getLastCellNum() -1;
		
		for (int i = dataSize; i < lastCellIndex; i++) {
			Cell cellTmp = newRow.createCell(i);
			cellTmp.setCellValue("");
			cellTmp.setCellStyle(style);
		}
	}

	private void writeHeader(List<PropertyDef> defList, Row row, int startColunm, CellStyle style){
		Cell cell = row.createCell(startColunm);
		cell.setCellValue("序号");
		cell.setCellStyle(style);
		
		for (int i = 0; i < defList.size(); i++) {
			Cell cellTmp = row.createCell(startColunm + i + 1);
			cellTmp.setCellValue(defList.get(i).getDisplayName());
			cellTmp.setCellStyle(style);
		}
	}
	
	private List<PropertyDef> filteProDef(List<PropertyDef> defList){
		if (CollectionUtils.isEmpty(defList)) {
			return defList;
		}
		
		defList = defList.stream()
				.filter(proDef-> ReadWriteTypeEnum.BOTH_TYPE.getCode().equals(proDef.getReadType())
						&& !ignoreList.contains(proDef.getCode()))
				.collect(Collectors.toList());
		
		defList.sort((d1, d2)->{
			
			if (d1 == null || d1.getSortNo() == null) {
				//d1 < d2
				return -1;
			}
			if (d2 == null || d2.getSortNo() == null) {
				//d1 > d2
				return 1;
			}
			
			return d1.getSortNo().compareTo(d2.getSortNo());
		});
		return defList;
	}
	
	private OrderExcelUtils (ExportParam param, TemplateExportParams tplParams){
		this.param = param;
		this.tplParams = tplParams;
	}
	
	public static ExportParam createParam() {
		return new ExportParam();
	}

	public static class ExportParam {
		private InquirySupOrder supOrder;
		private TemplateConf templateConf;
		private Company supCompany;
		private User purUser;
		private Map<String, Object> other;

		public InquirySupOrder getSupOrder() {
			return supOrder;
		}

		public ExportParam setSupOrder(InquirySupOrder supOrder) {
			this.supOrder = supOrder;
			return this;
		}

		public TemplateConf getTemplateConf() {
			return templateConf;
		}

		public ExportParam setTemplateConf(TemplateConf templateConf) {
			this.templateConf = templateConf;
			return this;
		}

		public Company getSupCompany() {
			return supCompany;
		}

		public ExportParam setSupCompany(Company supCompany) {
			this.supCompany = supCompany;
			return this;
		}

		public User getPurUser() {
			return purUser;
		}

		public ExportParam setPurUser(User purUser) {
			this.purUser = purUser;
			return this;
		}
		
		public ExportParam setOther(Map<String, Object> other) {
			this.other = other;
			return this;
		}
		
		public Map<String, Object> getOther() {
			return other;
		}
	}
	
}
