package com.els.base.inquiry.utils.excel;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.servlet.ServletOutputStream;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang.time.DateUtils;

import com.els.base.core.entity.dictionary.DicGroup;
import com.els.base.core.entity.dictionary.DicGroupExample;
import com.els.base.core.entity.dictionary.DicGroupItem;
import com.els.base.core.entity.dictionary.DicGroupItemExample;
import com.els.base.core.exception.CommonException;
import com.els.base.core.service.dictionary.DicGroupItemService;
import com.els.base.core.service.dictionary.DicGroupService;
import com.els.base.core.utils.Assert;
import com.els.base.inquiry.ITarget;
import com.els.base.inquiry.entity.PropertyDef;
import com.els.base.inquiry.entity.TargetM001;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryQuoteLadderType;
import com.els.base.inquiry.service.TemplateConfService;
import com.els.base.material.dao.MaterialMapper;
import com.els.base.material.entity.MaterialExample;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.excel.ExcelUtils;
import com.els.base.utils.excel.TitleAndModelKey;

import jxl.write.WritableWorkbook;
import jxl.write.WriteException;
import jxl.write.biff.RowsExceededException;

public class OrderItemExcelUtils {

	private static final String purchaseGroupCode = "purchasing_group";
	private static final String infoCategoryCode = "info_category";
	private static final String[] parsePatterns = new String[] {"yyyyMMdd"};
	private static SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMdd");
	
	public static void export(String templateId, ServletOutputStream os) throws RowsExceededException, WriteException, IOException, ParseException {
		TemplateConf templateConf = SpringContextHolder.getOneBean(TemplateConfService.class).queryObjById(templateId);
		Assert.isNotNull(templateConf, "模板不能为空");
		
		List<PropertyDef> propertyDefList = templateConf.getOrderItemPropertyDefList();
		Assert.isNotEmpty(templateConf.getOrderItemPropertyDefList(), "模板的字段定义不能为空");
		
		List<TitleAndModelKey> titleAndModelKeys = transfromProDef(propertyDefList);
		WritableWorkbook writableWorkbook = ExcelUtils.exportDataToExcel(os, titleAndModelKeys, null, "导入物料", null, 0);
		
		try {
			writableWorkbook.write();
			os.flush();
		} finally {
			writableWorkbook.close();
			os.close();
		}
	}

	/*private static List<TitleAndModelKey> transfromProDef(List<PropertyDef> propertyDefList) {
		Assert.isNotEmpty(propertyDefList, "模板的字段定义不能为空");
		
		propertyDefList.sort(Comparator.comparing(PropertyDef::getSortNo));
		
		return propertyDefList.parallelStream().map(propertyDef->{
			TitleAndModelKey titleAndModelKey = new TitleAndModelKey();
			titleAndModelKey.setModelKey(propertyDef.getCode());
			titleAndModelKey.setTitle(propertyDef.getDisplayName());
			
			if ("date".equals(propertyDef.getType())) {
				DateConverter dateConverter = new DateConverter("yyyymmdd");
				titleAndModelKey.setToObjConverter(dateConverter);
				titleAndModelKey.setToStrConverter(dateConverter);
				
				titleAndModelKey.setTitle(String.format("%s(yyyymmdd)", titleAndModelKey.getTitle()));
			}
			
			return titleAndModelKey;
		}).collect(Collectors.toList());
	}*/
	private static List<TitleAndModelKey> transfromProDef(List<PropertyDef> propertyDefList) {
		Assert.isNotEmpty(propertyDefList, "模板的字段定义不能为空");
		
		List<TitleAndModelKey> titleAndModelKeys = new ArrayList<>();
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("物料编码[必填]", "materialCode"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("信息类别[必填]", "infoCategory"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购申请单号[必填]", "purchaseApply"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("行项目号[必填]", "project"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("图号[必填]", "mapNo"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("名称[必填]", "name"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("数量[必填]", "quantity"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("最近一次采购不含税单价[必填]", "recentUntaxedUnitPrice"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("降幅[必填]", "decreasingAmplitude"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("报价方式，填1常规报价，填2阶梯报价[必填]", "quoteType"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购组[必填]", "purchaseGroup"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("标准采购订单数量", "purOrderQuantity"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("最小订单数量", "minOrderQuantity"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("价格单位[必填]", "priceUnit"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("有效期从(yyyyMMdd)[必填]", "validDateFrom"));
		titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("有效期到(yyyyMMdd)[必填]", "validDateTo"));
		
		return titleAndModelKeys;
	}

	public static List<? extends ITarget> importFromExcel(String templateId, InputStream in) throws Exception {
		TemplateConf templateConf = SpringContextHolder.getOneBean(TemplateConfService.class).queryObjById(templateId);
		Assert.isNotNull(templateConf, "模板不能为空");
		
		List<PropertyDef> propertyDefList = templateConf.getOrderItemPropertyDefList();
		Assert.isNotEmpty(templateConf.getOrderItemPropertyDefList(), "模板的字段定义不能为空");
		
		List<TitleAndModelKey> titleAndModelKeys = transfromProDef(propertyDefList);
		
		List<Map<String, String>> results = ExcelUtil.importExcelDataToMap(in, 0, 1, 0, titleAndModelKeys);
		List<? extends ITarget> targetList = convertMapListToTargetList(results, templateId);
		return targetList;
	}
	
	private static List<? extends ITarget> convertMapListToTargetList(List<Map<String, String>> results, String templateId) {
		if (CollectionUtils.isEmpty(results)) {
			return null;
		}
		List<TargetM001> targetM001s = new ArrayList<>();
		// 查询数据字典信息类别，采购组
		List<String> validCodes = Arrays.asList(infoCategoryCode,purchaseGroupCode);
		DicGroupExample example = new DicGroupExample();
		example.createCriteria().andCodeIn(validCodes);
		List<DicGroup> dicGroups = SpringContextHolder.getOneBean(DicGroupService.class).queryAllObjByExample(example);
		
		if (CollectionUtils.isEmpty(dicGroups) || dicGroups.size() != validCodes.size()) {
			throw new CommonException("查询数据字典出错，请联系管理员");
		}
		Map<String, String> dicGroupMap = new HashMap<>(dicGroups.size());
		for (DicGroup dicGroup : dicGroups) {
			dicGroupMap.put(dicGroup.getCode(), dicGroup.getId());
		}
		DicGroupItemExample itemExample = new DicGroupItemExample();
		itemExample.createCriteria().andGroupIdEqualTo(dicGroupMap.get(infoCategoryCode));
		List<DicGroupItem> infoCategoryCodes = SpringContextHolder.getOneBean(DicGroupItemService.class).queryAllObjByExample(itemExample);
		Assert.isNotEmpty(infoCategoryCodes, "未找到信息类别的数据字典");
		Set<String> infoCategoryCodeSet = infoCategoryCodes.stream().map(DicGroupItem::getValue).collect(Collectors.toSet());
		
		itemExample.clear();
		itemExample.createCriteria().andGroupIdEqualTo(dicGroupMap.get(purchaseGroupCode));
		List<DicGroupItem> purchaseGroupCodes = SpringContextHolder.getOneBean(DicGroupItemService.class).queryAllObjByExample(itemExample);
		Assert.isNotEmpty(purchaseGroupCodes, "未找到采购组的数据字典");
		Set<String> purchaseGroupCodeSet = purchaseGroupCodes.stream().map(DicGroupItem::getValue).collect(Collectors.toSet());
		int i = 2;
		for (Map<String, String> result : results) {
			valid(result, i, infoCategoryCodeSet, purchaseGroupCodeSet);
			TargetM001 targetM001 = convertMapToTarget(result, templateId);
			targetM001s.add(targetM001);
			i++;
		}
		
		return targetM001s;
	}

	private static TargetM001 convertMapToTarget(Map<String, String> result, String templateId) {
		TargetM001 targetM001 = new TargetM001();
		targetM001.setTemplateId(templateId);
		targetM001.setMaterialCode(result.get("materialCode"));//物料编码
		targetM001.setInfoCategory(result.get("infoCategory"));//信息类别
		
		targetM001.setPurchaseApply(result.get("purchaseApply"));//采购申请单号
		targetM001.setProject(result.get("project"));//行项目号
		targetM001.setMapNo(result.get("mapNo"));//图号
		targetM001.setName(result.get("name"));//名称
		targetM001.setQuantity(Long.valueOf(result.get("quantity")));//数量
		targetM001.setRecentUntaxedUnitPrice(new BigDecimal(result.get("recentUntaxedUnitPrice")));//最近一次采购不含税单价
		targetM001.setDecreasingAmplitude(result.get("decreasingAmplitude"));//降幅
		targetM001.setQuoteType(Integer.valueOf(result.get("quoteType")));//报价方式
		targetM001.setPurchaseGroup(result.get("purchaseGroup"));//采购组
		if (StringUtils.isNotBlank(result.get("purOrderQuantity"))) {
			targetM001.setPurOrderQuantity(new BigDecimal(result.get("purOrderQuantity")));//标准采购订单数量
		}
		if (StringUtils.isNotBlank(result.get("minOrderQuantity"))) {
			targetM001.setMinOrderQuantity(new BigDecimal(result.get("minOrderQuantity")));//最小订单数量
		}
		targetM001.setPriceUnit(Integer.valueOf(result.get("priceUnit")));//价格单位
		try {
			targetM001.setValidDateFrom(formatter.parse(result.get("validDateFrom")));//有效期从
		}catch (Exception e) {
			targetM001.setValidDateFrom(null);
		}
		try {
			targetM001.setValidDateTo(formatter.parse(result.get("validDateTo")));//有效期到
		}catch (Exception e) {
			targetM001.setValidDateTo(null);
		}
		return targetM001;
	}

	private static void valid(Map<String, String> result, int i, Set<String> infoCategoryCodeSet, Set<String> purchaseGroupCodeSet) {
		Assert.isNotBlank(result.get("materialCode"), String.format("第%s行中的物料编码不能为空", i));
		// 物料编码必须存在物料主数据中
		MaterialExample materialExample = new MaterialExample();
		materialExample.createCriteria().andMaterialCodeEqualTo(result.get("materialCode"));
		if (SpringContextHolder.getOneBean(MaterialMapper.class).countByExample(materialExample) == 0) {
			throw new CommonException(String.format("第%s行中的物料编码不存在于物料主数据中", i));
		}
		Assert.isNotBlank(result.get("infoCategory"), String.format("第%s行中的信息类别不能为空", i));
		if (!infoCategoryCodeSet.contains(result.get("infoCategory"))) {
			throw new CommonException(String.format("第%s行中的信息类别不存在于数据字典中", i));
		}
		Assert.isNotBlank(result.get("purchaseApply"), String.format("第%s行中的采购申请单号不能为空", i));
		Assert.isNotBlank(result.get("project"), String.format("第%s行中的行项目号不能为空", i));
		Assert.isNotBlank(result.get("mapNo"), String.format("第%s行中的图号不能为空", i));
		Assert.isNotBlank(result.get("name"), String.format("第%s行中的名称不能为空", i));
		Assert.isNotBlank(result.get("quantity"), String.format("第%s行中的数量不能为空", i));
		if (!NumberUtils.isNumber(result.get("quantity"))) {
			throw new CommonException(String.format("第%s行中的数量格式不正确", i));
		}
		
		Assert.isNotBlank(result.get("recentUntaxedUnitPrice"), String.format("第%s行中的最近一次采购不含税单价不能为空", i));
		if (!NumberUtils.isNumber(result.get("recentUntaxedUnitPrice"))) {
			throw new CommonException(String.format("第%s行中的最近一次采购不含税单价格式不正确", i));
		}
		
		Assert.isNotBlank(result.get("decreasingAmplitude"), String.format("第%s行中的降幅不能为空", i));
		if (!NumberUtils.isNumber(result.get("decreasingAmplitude"))) {
			throw new CommonException(String.format("第%s行中的降幅格式不正确", i));
		}
		Assert.isNotBlank(result.get("quoteType"), String.format("第%s行中的报价方式不能为空", i));
		if (!InquiryQuoteLadderType.CONVENTIONAL_QUOTE.getCode().equals(Integer.valueOf(result.get("quoteType")))
				&& !InquiryQuoteLadderType.LADDER_QUOTE.getCode().equals(Integer.valueOf(result.get("quoteType")))) {
			throw new CommonException(String.format("第%s行中的报价方式格式不正确，填1表示常规报价，填2表示阶梯报价", i));
		}
		Assert.isNotBlank(result.get("purchaseGroup"), String.format("第%s行中的采购组不能为空", i));
		if (!purchaseGroupCodeSet.contains(result.get("purchaseGroup"))) {
			throw new CommonException(String.format("第%s行中的采购组不存在于数据字典中", i));
		}
		if (StringUtils.isNotBlank(result.get("purOrderQuantity")) && !NumberUtils.isNumber(result.get("purOrderQuantity"))) {
			throw new CommonException(String.format("第%s行中的标准采购订单数量格式不正确", i));
		}
		if (StringUtils.isNotBlank(result.get("minOrderQuantity")) && !NumberUtils.isNumber(result.get("minOrderQuantity"))) {
			throw new CommonException(String.format("第%s行中的最小订单数量格式不正确", i));
		}
		Assert.isNotBlank(result.get("priceUnit"), String.format("第%s行中的价格单位不能为空", i));
		Assert.isNotBlank(result.get("validDateFrom"), String.format("第%s行中的有效期从不能为空", i));
		if (!dateFormatIsValid(result.get("validDateFrom"))) {
			throw new CommonException(String.format("第%s行中的有效期从格式不正确，请以如“20180101”格式填写", i));
		}
		Assert.isNotBlank(result.get("validDateTo"), String.format("第%s行中的有效期到不能为空", i));
		if (!dateFormatIsValid(result.get("validDateTo"))) {
			throw new CommonException(String.format("第%s行中的有效期到格式不正确，请以如“20180101”格式填写", i));
		}
	}
	
	private static boolean dateFormatIsValid(String dateStr) {
		try {
			DateUtils.parseDate(dateStr, parsePatterns);
		} catch (Exception e) {
			return false;
		}
		return true;
	}

}
