package com.els.base.inquiry.utils.json;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;

import com.els.base.core.utils.Constant;
import com.els.base.inquiry.IExtendable;
import com.els.base.inquiry.entity.PropertyDef;
import com.els.base.inquiry.entity.PropertyValue;
import com.els.base.inquiry.utils.ClassReflectionUtils;
import com.els.base.inquiry.utils.PropertyValueUtils;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 可自定义的对象的序列化
 * @author hzy
 *
 */
public abstract class ExtendableObjectJsonDeSerialzer<T extends IExtendable> extends JsonDeserializer<T>{
	
	protected Logger logger = LoggerFactory.getLogger(this.getClass());

	@Override
	public T deserialize(JsonParser p, DeserializationContext ctxt) throws IOException, JsonProcessingException {
		try {
			return this.deserialize(p);
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("反序列化异常",e);
		}
	}
	
	/**
	 * json反序列化
	 * @param p
	 * @return
	 * @throws InstantiationException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 * @throws InvocationTargetException
	 * @throws NoSuchMethodException
	 * @throws SecurityException
	 * @throws IOException
	 * @throws NoSuchFieldException
	 */
	protected T deserialize(JsonParser p) throws InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException, NoSuchMethodException, SecurityException, IOException, NoSuchFieldException  {
		//1、读取json传来的内容
		ObjectNode objectNode = p.readValueAsTree();
		
		//1、初始化对象
		T t = this.getInstanceClass().newInstance();
		
		//2、设置原生字段
		this.setValueIntoInstance(t, objectNode);
		
		//3、自定义字段
		this.setValueIntoPropertyValue(t, objectNode);
		
		return t;
	}
	
	/**
	 * 从json中，获取自定义的字段值，设置到PropertyValue 里面
	 * @param t
	 * @param objectNode
	 */
	protected void setValueIntoPropertyValue(T t, ObjectNode objectNode) {
		List<PropertyDef> propertyDefList = t.getPropertyDefList();
		if (CollectionUtils.isEmpty(propertyDefList)) {
			return;
		}
		
		//根据自定义的字段，生成值
		List<PropertyValue> propertyValueList = propertyDefList.stream()
			.filter(propertyDef-> !Constant.YES_INT.equals(propertyDef.getIsPrimitive()))
			.map(propertyDef->{
				PropertyValue propertyValue = new PropertyValue();
				BeanUtils.copyProperties(propertyDef, propertyValue);
				propertyValue.setId(null);
				return propertyValue;
			
			}).collect(Collectors.toList());
		
		propertyValueList.stream().forEach(propertyValue->{
			String fieldName = propertyValue.getCode();
			
			JsonNode fieldNode = objectNode.get(fieldName);
			if (fieldNode == null || fieldNode.isNull()) {
				propertyValue.setValueStr(null);
				return;
			} 
			
			String value = fieldNode.asText();
			if (StringUtils.isBlank(value)) {
				propertyValue.setValueStr(null);
				
			}else{
				value = PropertyValueUtils.validStr(value, propertyValue.getType(), propertyValue.getCode());
				propertyValue.setValueStr(value);
			}
		});
		
		t.setPropertyValueList(propertyValueList);
	}

	/**
	 * 从json中将原生的字段值，设置到 实体
	 * @param propertyValue
	 * @param jsonNode
	 */
	protected void setValueIntoInstance(T t, ObjectNode objectNode) {
		List<Field> primitiveFieldList = ClassReflectionUtils.getPrimitiveField(t.getClass());
		
		Iterator<String> fieldNameIterator = objectNode.fieldNames();
		while (fieldNameIterator.hasNext()) {
			
			String fieldName = fieldNameIterator.next();
			
			Optional<Field> primitive = primitiveFieldList.stream().filter(field->{
				return field.getName().equals(fieldName);
			}).findAny();
			
			primitive.ifPresent(field->{
				String valueStr = objectNode.get(fieldName).asText();
				try {
					this.setPrimitFieldValue(t, field, objectNode.get(fieldName));
					
				} catch ( Exception e) {
					logger.error(String.format("反序列化字段field[%s]:[%s]", field.getName(), valueStr), e);
					throw new RuntimeException(e);
				}
			});
		}
		
	}

	/**
	 * 从json中将原生的字段值，设置到 实体
	 * @param propertyValue
	 * @param jsonNode
	 * @throws Exception 
	 */
	protected void setPrimitFieldValue(T instance, Field field, JsonNode jsonNode) throws Exception  {
		String valueStr = null;
		if (!jsonNode.isValueNode()) {
			valueStr = jsonNode.toString();
			
		} else{
			valueStr = jsonNode.asText();
		}
		
		PropertyValueUtils.setValueToInstance(instance, field ,valueStr);
	}

	public abstract Class<T> getInstanceClass();


}
