package com.els.base.inquiry.utils.json;

import static java.util.stream.Collectors.toList;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.els.base.inquiry.IExtendable;
import com.els.base.inquiry.entity.PropertyDef;
import com.els.base.inquiry.entity.PropertyValue;
import com.els.base.inquiry.utils.ClassReflectionUtils;
import com.els.base.inquiry.utils.PropertyValueUtils;
import com.els.base.utils.reflect.ReflectUtils;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;


/**
 * 可自定义的对象的序列化
 * @author hzy
 *
 */
public class ExtendableObjectJsonSerialzer extends JsonSerializer<IExtendable> {
	
	protected Logger logger = LoggerFactory.getLogger(this.getClass());
	
	private List<String> ignoreList(){
		List<String> ignoreList = new ArrayList<>();
		ignoreList.add("propertyValueList");
		return ignoreList;
	}

	@Override
	public void serialize(IExtendable value, JsonGenerator gen, SerializerProvider serializers)
			throws IOException, JsonProcessingException {
		
		if(value==null){
			gen.writeNull();
		}
		
		Map<String, Object> valueMap = this.buildMap(value);
		
		this.validMap(valueMap, value);
		
		gen.writeObject(valueMap);
	}
	
	
	public Map<String, Object> buildMap(IExtendable value){
		Map<String, Object> valueMap = new HashMap<>();
		
		//把实体的原生字段放到map
		valueMap = this.setPrimitiveFieldIntoMap(valueMap, value);
		
		//将自定义的字段，放到map中
		valueMap = this.setPropertyValue(valueMap, value);
		
		return valueMap;
	}
	
	private void validMap(Map<String, Object> valueMap, IExtendable value) {
		List<PropertyDef> defList = value.getPropertyDefList();
		defList = defList.stream()
			.filter(def -> valueMap.keySet().stream().noneMatch(key-> def.getCode().equals(key)))
			.filter(def -> def!=null)
			.collect(Collectors.toList());
		
		if (CollectionUtils.isNotEmpty(defList)) {
			String fieldName = StringUtils.join(defList, ",");
			throw new RuntimeException(String.format("反序列化，少了字段:%s", fieldName));
		}
		
	}

	private Map<String, Object> setPropertyValue(Map<String, Object> valueMap, IExtendable value) {

		//过滤一些不用的字段
		List<PropertyDef> proDefList = this.filtePropertyDef(value);
		
		if (CollectionUtils.isEmpty(value.getPropertyValueList())) {
			value.setPropertyValueList(new ArrayList<>());
		}
		
		//补全一些字段
		this.completePropertyValue(value, proDefList);
		
		//把自定义的字段的值插入到Map
		value.getPropertyValueList().forEach(properyValue->{
			Object valueObj = PropertyValueUtils.parseStr(properyValue.getValueStr(), properyValue.getType(), properyValue.getCode());
			valueMap.putIfAbsent(properyValue.getCode(), valueObj);
		});
		
		return valueMap;
	}

	private List<PropertyDef> filtePropertyDef(IExtendable value) {
		if (CollectionUtils.isEmpty(value.getPropertyDefList())) {
			return null;
		}
		List<Field> primitiveFieldList = ClassReflectionUtils.getPrimitiveField(value.getClass());
		List<PropertyDef> proDefList = value.getPropertyDefList().stream()
			.filter(proDef-> primitiveFieldList.stream().noneMatch(field-> field.getName().equals(proDef.getCode())))
			.collect(toList());
		
		return proDefList;
	}

	private void completePropertyValue(IExtendable value, List<PropertyDef> proDefList) {
		if (CollectionUtils.isEmpty(proDefList)) {
			return;
		}
		
		List<PropertyValue> absentValueList = proDefList.stream().map(preDef->{
			
			Optional<PropertyValue> proValueOpl = value.getPropertyValueList().stream()
				.filter(proValue-> preDef.getCode().equals(proValue.getCode()))
				.findAny();
			
			return proValueOpl.orElseGet(()-> preDef.buildValue());
		}).collect(toList());
		
		value.setPropertyValueList(absentValueList);
	}

	private Map<String, Object> setPrimitiveFieldIntoMap(Map<String, Object> valueMap, IExtendable value) {
		
		List<Field> primitiveFieldList = ClassReflectionUtils.getPrimitiveField(value.getClass());
		//把实体对象，转化为Map
		
		primitiveFieldList.stream()
			.filter(field->{
				return !this.ignoreList().contains(field.getName());
				
			}).forEach(field->{
				try {
					Object fieldValue = ReflectUtils.getValue(value, field.getName());
					valueMap.put(field.getName(), fieldValue);
					
				} catch ( IllegalArgumentException  e) {
					logger.error(String.format("序列化字段field失败，[%s]", field.getName()), e);
					throw new RuntimeException(e);
				}
			});
		
		return valueMap;
	}

}
