package com.els.base.inquiry.web.controller;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.ParseException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang.time.DateUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import com.els.base.company.entity.Department;
import com.els.base.company.service.CompanyService;
import com.els.base.company.service.DepartmentService;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.service.user.UserService;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.file.entity.FileData;
import com.els.base.file.service.FileManagerFactory;
import com.els.base.inquiry.ITarget;
import com.els.base.inquiry.InquiryCommandInvoker;
import com.els.base.inquiry.command.sup.SupViewDetailCommand;
import com.els.base.inquiry.entity.InquirySupOrder;
import com.els.base.inquiry.entity.InquirySupOrderExample;
import com.els.base.inquiry.entity.TemplateConf;
import com.els.base.inquiry.enumclass.InquiryQuoteStatus;
import com.els.base.inquiry.service.InquirySupOrderService;
import com.els.base.inquiry.utils.excel.OrderExcelUtils;
import com.els.base.inquiry.utils.excel.OrderExcelUtils.ExportParam;
import com.els.base.inquiry.utils.excel.OrderItemExcelUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import jxl.write.WriteException;
import jxl.write.biff.RowsExceededException;


@Api(value="询报价-导入导出")
@Controller
@RequestMapping("inquiryExcel")
public class InquiryExcelController {
	
	@Resource
	protected InquiryCommandInvoker invoker;
	
	@Resource
	protected CompanyService companyService;
	
	@Resource
	protected UserService userService;
	
	@Resource
	protected InquirySupOrderService inquirySupOrderService;
	
	@Resource
	protected DepartmentService departmentService;

	@ApiOperation(httpMethod="POST", value="下载导入模板")
    @RequestMapping("service/downloadExcel")
    @ResponseBody
    public String downloadExcel(@RequestParam(required=true) String templateId, HttpServletResponse response) throws RowsExceededException, WriteException, IOException, ParseException{

    	String fileName = String.format("物料清单模板%s.xls", DateFormatUtils.format(new Date(), "yyyyMMdd"));
    	this.setDownloadHeader(fileName, response);
    	
    	OrderItemExcelUtils.export(templateId, response.getOutputStream());
    	return null;
    }
    
    @ApiOperation(httpMethod="POST", value="上传excel导入数据")
    @RequestMapping("service/uploadForImportExcel")
    @ResponseBody
    public ResponseResult<List<? extends ITarget>> uploadForImportExcel(@RequestParam(required=true) String templateId, MultipartHttpServletRequest request) throws Exception{
    	Map<String, MultipartFile> fileMap = request.getFileMap();

    	if (MapUtils.isEmpty(fileMap)) {
			throw new CommonException("上传文件为空", "file_isNull");
		}
		if (fileMap.size() > 1) {
			throw new CommonException("只接受单个文件导入");
		}

		MultipartFile file = fileMap.entrySet().iterator().next().getValue();

    	return ResponseResult.success(OrderItemExcelUtils.importFromExcel(templateId, file.getInputStream()));
    }
    
    @ApiOperation(httpMethod="POST", value="导出询价单的数据")
    @RequestMapping("service/exportPurOrder")
    @ResponseBody
    public ResponseResult<FileData> exportPurOrder(@RequestParam(required=true) String supOrderId, HttpServletResponse response) throws RowsExceededException, WriteException, IOException, ParseException, InterruptedException{

		InquirySupOrder supOrder = invoker.invoke(new SupViewDetailCommand(supOrderId));
		if (InquiryQuoteStatus.UNQUOTED.getCode().equals(supOrder.getQuoteStatus())) {
			throw new CommonException("该供应商报价单还没有报价，无法导出");
		}
		
		if (InquiryQuoteStatus.REQUOTE.getCode().equals(supOrder.getQuoteStatus())) {
			boolean noneExistQuotedItem = supOrder.getItemList().stream()
					.noneMatch(item-> InquiryQuoteStatus.QUOTED.getCode().equals(item.getQuotationStatus()) 
							|| InquiryQuoteStatus.REQUOTED.getCode().equals(item.getQuotationStatus()));
			
			if (noneExistQuotedItem) {
				throw new CommonException("该供应商报价单还没有报价，无法导出");
			}
		}
		
		TemplateConf templateConf = invoker.getTemplateConfService().queryObjById(supOrder.getTemplateId());
		
		// 获取采购部门
		Optional<Department> purDepartment = Optional.ofNullable(departmentService.queryDepartmentOfUser(supOrder.getPurUserId()));
		// 获取供应部门
		Optional<Department> supDepartment = Optional.ofNullable(departmentService.queryDepartmentOfUser(supOrder.getSupUserId()));
		Map<String, Object> other = new HashMap<>();
		other.put("purDepartmentName", purDepartment.map(Department::getDepartmentName).orElse(""));
		other.put("supDepartmentName", supDepartment.map(Department::getDepartmentName).orElse(""));
		
		
		ExportParam param = OrderExcelUtils.createParam()
    			.setPurUser(userService.queryObjById(supOrder.getPurUserId()))
    			.setSupCompany(this.companyService.queryObjById(supOrder.getSupCompanyId()))
    			.setTemplateConf(templateConf)
    			.setSupOrder(supOrder)
    			.setOther(other);
    	Workbook workbook = OrderExcelUtils.exportToExcel(param);
    	
    	String fileName = String.format("报价单%s.xls", DateFormatUtils.format(new Date(), "yyyyMMdd"));
    	FileData fileData = createEmptyPDF(ProjectUtils.getProjectId(), CompanyUtils.currentCompanyId(), fileName);
    	File file = fileData.toFile();
    	
    	FileOutputStream fos = new FileOutputStream(file);
    	workbook.write(fos);
    	fos.flush();
    	fos.close();
    	workbook.close();
    	
    	return ResponseResult.success(fileData);
    }
    
    private FileData createEmptyPDF(String projectId, String companyId, String filename) throws UnsupportedEncodingException, IOException{
		FileData pdfFile = new FileData();
		pdfFile.setProjectId(projectId);
		pdfFile.setCompanyId(companyId);
		
		pdfFile.setFileName(filename);
		pdfFile.setFileSuffix("xls");
		pdfFile.setIsEncrypt(String.valueOf(Constant.NO_INT));
		pdfFile.setExpiryDay(DateUtils.addMinutes(new Date(), 30));
		FileManagerFactory.getFileManager().createEmptyFile(pdfFile);
		return pdfFile;
	}
    
    @ApiOperation(httpMethod="POST", value="查询报价单")
    @RequestMapping("service/selectSupplier")
    @ResponseBody
    public ResponseResult<List<InquirySupOrder>> selectSupplier(@RequestParam(required=true) String purOrderId){
    	
    	InquirySupOrderExample example = new InquirySupOrderExample();
    	example.createCriteria().andPurOrderIdEqualTo(purOrderId);
    	
    	List<InquirySupOrder> supplierList = this.inquirySupOrderService.queryAllObjByExample(example);
    	return ResponseResult.success(supplierList);
    }
    
    
    private void setDownloadHeader(String fileName, HttpServletResponse response) throws UnsupportedEncodingException{
    	
		StringBuffer header = new StringBuffer("attachment;");
		header.append("filename=\"" + URLEncoder.encode(fileName, "UTF-8") + "\";");
		response.setHeader("Content-Disposition", header.toString());

		response.setCharacterEncoding("UTF-8");
		response.setContentType("application/octet-stream;charset=UTF-8");
    }
    
    
}
