package com.els.base.inquiry.web.controller;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.material.entity.Material;
import com.els.base.material.entity.MaterialCategoryManager;
import com.els.base.material.entity.MaterialCategoryManagerExample;
import com.els.base.material.entity.MaterialExample;
import com.els.base.material.service.MaterialCategoryManagerService;
import com.els.base.material.service.MaterialService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.inquiry.ITarget;
import com.els.base.inquiry.entity.InquiryWaitMaterial;
import com.els.base.inquiry.entity.InquiryWaitMaterialExample;
import com.els.base.inquiry.service.InquiryWaitMaterialService;
import com.els.base.inquiry.web.vo.WaitMaterialTransformVo;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

@Api(value="询报价-待询价物料")
@Controller
@RequestMapping("inquiryWaitMaterial")
public class InquiryWaitMaterialController {
    @Resource
    protected InquiryWaitMaterialService inquiryWaitMaterialService;
    @Resource
    protected MaterialCategoryManagerService materialCategoryManagerService;
    @Resource
    protected MaterialService materialService;

    @ApiOperation(httpMethod="POST", value="创建询报价-待询价物料")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody InquiryWaitMaterial inquiryWaitMaterial) {
        this.inquiryWaitMaterialService.addObj(inquiryWaitMaterial);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="编辑询报价-待询价物料")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody InquiryWaitMaterial inquiryWaitMaterial) {
        Assert.isNotBlank(inquiryWaitMaterial.getId(), "id 为空，保存失败");
        this.inquiryWaitMaterialService.modifyObj(inquiryWaitMaterial);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="删除询报价-待询价物料")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required=true) String id) {
        Assert.isNotBlank(id, "删除失败,id不能为空");
        this.inquiryWaitMaterialService.deleteObjById(id);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="查询询报价-待询价物料")
    @ApiImplicitParams({ 
		@ApiImplicitParam( name = "pageNo",required = false,value = "所在页", paramType = "query", dataType = "String", defaultValue = "0" ),  
		@ApiImplicitParam( name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10" ),  
		@ApiImplicitParam( name = "wapper", required = false, value = "查询条件,属性名请参考 InquiryWaitMaterial", paramType = "body", dataType = "QueryParamWapper" )  
	}) 
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<InquiryWaitMaterial>> findByPage( 
		@RequestParam(defaultValue="1") int pageNo,  
		@RequestParam(defaultValue="10") int pageSize,
		String purOrderNo,
        @RequestParam(defaultValue="1") Integer flag,
		@RequestBody(required=false) QueryParamWapper wapper) {
    	
        InquiryWaitMaterialExample example = new InquiryWaitMaterialExample();
        example.setPageView(new PageView<InquiryWaitMaterial>(pageNo, pageSize));
        
        InquiryWaitMaterialExample.Criteria criteria = example.createCriteria();
        criteria.andIsCreateInquiryOrderEqualTo(Constant.NO_INT);
        
        if (StringUtils.isNotBlank(purOrderNo)) {
			example.or().andInquiryOrderNoEqualTo(purOrderNo);
		}

		//总体需求是这样：需要供应经理只能看到自己负责的待询价清单，而供应经理是通过物料分类去划分的，那么这些人对应着分类，分类对应着物料编码
		//需要开启根据采购人员
        if(flag.equals(Constant.YES_INT)){ //默认是开启的
            this.addCriteria(criteria);
        }

        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        
        PageView<InquiryWaitMaterial> pageData = this.inquiryWaitMaterialService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }

    /**
     * 添加查询条件
     * @param criteria
     */
    private void addCriteria(InquiryWaitMaterialExample.Criteria criteria) {
        //1、找到该人管理的物料分类
        MaterialCategoryManagerExample materialCategoryManagerExample = new MaterialCategoryManagerExample();
        materialCategoryManagerExample.createCriteria()
                .andUserIdEqualTo(SpringSecurityUtils.getLoginUserId());
        List<MaterialCategoryManager> materialCategoryManagers = materialCategoryManagerService.queryAllObjByExample(materialCategoryManagerExample);
        if(CollectionUtils.isNotEmpty(materialCategoryManagers)){

            //2、根据找到该人管理的分类下的物料编码materialCode
            List<String> collect = materialCategoryManagers.stream()
                    //.filter(Objects::nonNull)
                    .filter(x->x!=null&& StringUtils.isNotBlank(x.getCategoryId()))
                    .map(MaterialCategoryManager::getCategoryId).collect(Collectors.toList());

            if(CollectionUtils.isNotEmpty(collect)){

                MaterialExample materialExample = new MaterialExample();
                materialExample.createCriteria().andCategoryIdIn(collect);
                List<Material> materials = materialService.queryAllObjByExample(materialExample);
                if(CollectionUtils.isNotEmpty(materials)){

                    //3.只显示管理的物料编码
                    List<String> materialCodes = materials.stream()
                            .filter(Objects::nonNull)
                            .filter(x->StringUtils.isNotBlank(x.getMaterialCode()))
                            .map(Material::getMaterialCode).collect(Collectors.toList());

                    if(CollectionUtils.isNotEmpty(materialCodes)){

                        //重要业务逻辑
                        criteria.andMaterialCodeIn(materialCodes);
                    }
                }
            }
        }
    }

    @ApiOperation(httpMethod="POST", value="查询询报价-待询价物料,没有经过物料分类筛选的")
    @ApiImplicitParams({
            @ApiImplicitParam( name = "pageNo",required = false,value = "所在页", paramType = "query", dataType = "String", defaultValue = "0" ),
            @ApiImplicitParam( name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10" ),
            @ApiImplicitParam( name = "wapper", required = false, value = "查询条件,属性名请参考 InquiryWaitMaterial", paramType = "body", dataType = "QueryParamWapper" )
    })
    @RequestMapping("service/pur/findByPage")
    @ResponseBody
    public ResponseResult<PageView<InquiryWaitMaterial>> findInquiryWaitMaterialForPurByPage(
            @RequestParam(defaultValue="1") int pageNo,
            @RequestParam(defaultValue="10") int pageSize,
            String purOrderNo,
            @RequestParam(defaultValue="1") Integer flag,
            @RequestBody(required=false) QueryParamWapper wapper) {

        InquiryWaitMaterialExample example = new InquiryWaitMaterialExample();
        example.setPageView(new PageView<InquiryWaitMaterial>(pageNo, pageSize));

        InquiryWaitMaterialExample.Criteria criteria = example.createCriteria();
        criteria.andIsCreateInquiryOrderEqualTo(Constant.NO_INT);

        if (StringUtils.isNotBlank(purOrderNo)) {
            example.or().andInquiryOrderNoEqualTo(purOrderNo);
        }

        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }

        PageView<InquiryWaitMaterial> pageData = this.inquiryWaitMaterialService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }


    @ApiOperation(httpMethod="POST", value="转换待询价清单成询价单物料")
    @RequestMapping("service/transformToPurOrderMaterial")
    @ResponseBody
    public ResponseResult<List<ITarget>> transformToPurOrderMaterial( @RequestBody(required=true) WaitMaterialTransformVo waitMaterialTransformVo) {
    	
    	List<ITarget> targets = inquiryWaitMaterialService.transformToPurOrderMaterial(waitMaterialTransformVo);
		return ResponseResult.success(targets);
    }
}