package com.els.base.inquiry.web.controller;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.time.DateUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.inquiry.entity.SystemPrice;
import com.els.base.inquiry.entity.SystemPriceExample;
import com.els.base.inquiry.service.SystemPriceService;
import com.els.base.inquiry.web.vo.SystemPriceVo;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

@Api(value="询报价-物料的ERP系统价")
@Controller
@RequestMapping("systemPrice")
public class SystemPriceController {
    @Resource
    protected SystemPriceService systemPriceService;

    @ApiOperation(httpMethod="POST", value="创建询报价-物料的ERP系统价")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody SystemPrice systemPrice) {
        this.systemPriceService.addObj(systemPrice);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="编辑询报价-物料的ERP系统价")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody SystemPrice systemPrice) {
        Assert.isNotBlank(systemPrice.getId(), "id 为空，保存失败");
        this.systemPriceService.modifyObj(systemPrice);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="删除询报价-物料的ERP系统价")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required=true) String id) {
        Assert.isNotBlank(id, "删除失败,id不能为空");
        this.systemPriceService.deleteObjById(id);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod="POST", value="查询询报价-物料的ERP系统价")
    @ApiImplicitParams({ 
		@ApiImplicitParam( name = "pageNo",required = false,value = "所在页", paramType = "query", dataType = "String", defaultValue = "0" ),  
		@ApiImplicitParam( name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10" ),  
		@ApiImplicitParam( name = "wapper", required = false, value = "查询条件,属性名请参考 SystemPrice", paramType = "body", dataType = "QueryParamWapper" )  
	}) 
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<SystemPrice>> findByPage( 
		@RequestParam(defaultValue="0") int pageNo,  
		@RequestParam(defaultValue="10") int pageSize, 
		@RequestBody(required=false) QueryParamWapper wapper) {
        SystemPriceExample example = new SystemPriceExample();
        example.setPageView(new PageView<SystemPrice>(pageNo, pageSize));
        
        SystemPriceExample.Criteria criteria = example.createCriteria();
        //填写具体的查询条件，例如
        //criteria.andIdEqualTo("1");
        if (wapper != null) {
            CriteriaUtils.addExample(example, wapper);
        }
        
        PageView<SystemPrice> pageData = this.systemPriceService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
    
    @ApiOperation(httpMethod="POST", value="历史价格对比-物料的ERP系统价")
    @RequestMapping("service/historicalPrice")
    @ResponseBody
    public ResponseResult<List<SystemPrice>> historicalPrice(@RequestParam(required=true) String materialCode, 
    		@RequestParam(required=true)String supCompanySapCode){
    	
    	Assert.isNotBlank(materialCode, "传入的物料编码为空");
    	SystemPriceExample example = new SystemPriceExample();
    	//根据价格有效起始时间降序排列
    	example.setOrderByClause("PRICE_EFFECTIVE_START_TIME DESC");
    	//如果前8个价格
    	example.setPageView(new PageView<>(1, 7));
    	
		SystemPriceExample.Criteria criteria = example.createCriteria();
		//历史价取值为价格有效起始时间在三年内的历史价
		criteria.andPriceEffectiveStartTimeGreaterThanOrEqualTo(DateUtils.addYears(new Date(), -3));
		criteria.andMaterialCodeEqualTo(materialCode);
		criteria.andCompanySapCodeEqualTo(supCompanySapCode);
		
		example.or()
			.andMaterialCodeEqualTo(materialCode)
			.andCompanySapCodeEqualTo(supCompanySapCode)
			.andPriceEffectiveStartTimeIsNull();
		
    	List<SystemPrice> priceList = systemPriceService.queryObjByPage(example).getQueryResult();
    	Assert.isNotEmpty(priceList, "根据条件查询出的历史价数据为空");
    	return ResponseResult.success(priceList);
    }
    
    @ApiOperation(httpMethod="POST", value="同类物料参考比价-物料的ERP系统价")
    @RequestMapping("service/materielPriceParity")
    @ResponseBody
    public ResponseResult<List<SystemPriceVo>> materielPriceParity(@RequestBody (required=true) List<String> materialCodeList,@RequestParam (required=true) BigDecimal untaxedUnitPrice){
    	Assert.isNotEmpty(materialCodeList, "传入的物料编码数据为空");
    	Assert.isNotNull(untaxedUnitPrice, "传入的报价为空");
    	
    	SystemPriceExample example = new SystemPriceExample();
    	SystemPriceExample.Criteria criteria = example.createCriteria();
    	criteria.andMaterialCodeIn(materialCodeList);
    	
    	//物料的系统价为当前有效价格
    	criteria.andPriceEffectiveEndTimeGreaterThanOrEqualTo(new Date());
    	List<SystemPrice> systemPriceList = systemPriceService.queryAllObjByExample(example);
    	if (CollectionUtils.isEmpty(systemPriceList)) {
			return ResponseResult.success(new ArrayList<>());
		}
    	
    	List<SystemPriceVo> list = new ArrayList<SystemPriceVo>(systemPriceList.size());
    	for(SystemPrice sp : systemPriceList){
			SystemPriceVo vo = new SystemPriceVo();
			//将对象信息复制到vo中
			BeanUtils.copyProperties(sp, vo);
			//计算当前报价与系统价的对比结果,保留2位小数
			vo.setPriceRatio(untaxedUnitPrice
				.subtract(sp.getUntaxedUnitPrice())
				.multiply(new BigDecimal(100))
				.divide(sp.getUntaxedUnitPrice(),2,BigDecimal.ROUND_HALF_UP));
			
			list.add(vo);	
		}
    	return ResponseResult.success(list);
    }
}