package com.els.base.msg.mail.web.controller;

import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.msg.mail.entity.MailTemplate;
import com.els.base.msg.mail.entity.MailTemplateExample;
import com.els.base.msg.mail.service.MailTemplateService;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

@Api(value = "消息管理-邮件-模板信息")
@Controller
@RequestMapping("mailTemplate")
public class MailTemplateController {
    @Resource
    protected MailTemplateService mailTemplateService;

    @ApiOperation(httpMethod = "POST", value = "创建邮件模板信息")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody MailTemplate mailTemplate) {
        if (mailTemplate.getTemplateNum() != null && mailTemplate.getTemplateNum().length() > 255
                || mailTemplate.getTemplateTitle() != null && mailTemplate.getTemplateTitle().length() > 255
                || mailTemplate.getTemplateName() != null && mailTemplate.getTemplateName().length() > 255) {
            throw new CommonException("输入的参数长度超标", "database_length_error");
        }
        
        if (!mailTemplate.getBusinessType().matches("[\\w\\_]+")) {
			throw new CommonException("业务类型的格式错误");
		}

        String projectId = ProjectUtils.getProjectId();
//        String currentCompanyId = CompanyUtils.currentCompanyId();

        MailTemplateExample mailTemplateExample = new MailTemplateExample();
        mailTemplateExample.createCriteria()
            .andProjectIdEqualTo(projectId)
//            .andCompanyIdEqualTo(currentCompanyId)
            .andBusinessTypeEqualTo(mailTemplate.getBusinessType());
        
        List<MailTemplate> mailTemplateList = this.mailTemplateService.queryAllObjByExample(mailTemplateExample);
        if (CollectionUtils.isNotEmpty(mailTemplateList)) {
            throw new CommonException("业务类型已经存在，不可重复", "base_is_exists", "业务类型");
        }

        mailTemplateExample.clear();
        mailTemplateExample.createCriteria()
            .andProjectIdEqualTo(projectId)
//            .andCompanyIdEqualTo(currentCompanyId)
            .andTemplateNumEqualTo(mailTemplate.getTemplateNum());
        
        List<MailTemplate> queryAllObjByExample = this.mailTemplateService
                .queryAllObjByExample(mailTemplateExample);
        if (CollectionUtils.isNotEmpty(queryAllObjByExample)) {
            throw new CommonException("邮件模板号已经存在，不可重复", "base_is_exists", "邮件模板号");
        }

        mailTemplate.setProjectId(projectId);
//        mailTemplate.setCompanyId(currentCompanyId);
        mailTemplate.setCreateUser(SpringSecurityUtils.getLoginUser().getNickName());
        
        this.mailTemplateService.addObj(mailTemplate);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "编辑邮件模板信息")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody MailTemplate mailTemplate) {
        if (StringUtils.isBlank(mailTemplate.getId())) {
            throw new CommonException("id 为空，保存失败");
        }

        if (mailTemplate.getTemplateNum() != null && mailTemplate.getTemplateNum().length() > 255
                || mailTemplate.getTemplateTitle() != null && mailTemplate.getTemplateTitle().length() > 255
                || mailTemplate.getTemplateName() != null && mailTemplate.getTemplateName().length() > 255) {
            throw new CommonException("输入的参数长度超标", "database_length_error");
        }

        if (!mailTemplate.getBusinessType().matches("[\\w\\_]+")) {
			throw new CommonException("业务类型的格式错误");
		}
        
        mailTemplate.setModifyDate(new Date());
        this.mailTemplateService.modifyObj(mailTemplate);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "删除邮件模板信息")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required = true) String id) {
        if (StringUtils.isBlank(id)) {
            throw new CommonException("删除失败,id不能为空");
        }
        this.mailTemplateService.deleteObjById(id);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "查询邮件模板信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 MailTemplate", paramType = "body", dataType = "QueryParamWapper") })
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<MailTemplate>> findByPage(@RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        MailTemplateExample example = new MailTemplateExample();
        example.setPageView(new PageView<MailTemplate>(pageNo, pageSize));
        example.setOrderByClause("CREATE_DATE DESC");
        if (wapper != null) {
            MailTemplateExample.Criteria criteria = example.createCriteria();
            CriteriaUtils.addCriterion(criteria, wapper);
        }

        PageView<MailTemplate> pageData = this.mailTemplateService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
}