package com.els.base.msg.im;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.els.base.core.entity.dictionary.DicGroupItem;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.dictonary.DicUtils;
import com.els.base.msg.IMessage;
import com.els.base.msg.IMessageCommand;
import com.els.base.msg.im.entity.ImTemplate;
import com.els.base.msg.im.service.ImTemplateService;
import com.els.base.msg.im.util.ImMessagePushProperties;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.template.BeetlTemplateUtils;
import com.qqt.message.client.MessageClient;
import com.qqt.message.client.impl.MessageClientImpl;

/**
 * 系统消息的发送命令
 * 
 * @author hzy
 *
 */
public class ImMessageCommand implements IMessageCommand {

	private static final long serialVersionUID = 1L;

	private static final Logger logger = LoggerFactory.getLogger(ImMessageCommand.class);

	private static ImTemplateService getImTemplateService() {
		return SpringContextHolder.getOneBean(ImTemplateService.class);
	}

	/** IM消息类型，系统消息 **/
	public static final String TYPE_SYSTEM = "system";
	/** IM消息类型，个人消息 **/
	public static final String TYPE_PERSONAL = "personal";

	private static MessageClient client;

	public ImMessageCommand() {
		try {
			if (client == null) {
				ImMessagePushProperties properties = SpringContextHolder.getOneBean(ImMessagePushProperties.class);
				if (StringUtils.isBlank(properties.getHost())) {
					throw new CommonException("消息服务器IP未配置");
				}
				
				client = new MessageClientImpl(
						properties.getAccessToken(), properties.getAddUrl(), 
						properties.getReadUrl(), properties.getListUrl(), 
						properties.getCountUrl(), properties.getListAllUrl());
//				client = MessageClientFactory.createPushMessageInstance();
			}

		} catch (Exception e) {
			logger.error("消息工具类初始化失败", e);
		}
	}

	@Override
	public void sendMsg(IMessage<?> message) throws Exception {

		Set<String> accounts = this.getAccount(message.getReceiverIdList());
		String type = this.getImMsgType(message);
		String companyCode = message.getCompanyCode();
		String businessTypeCode = message.getBusinessTypeCode();

		if (CollectionUtils.isEmpty(accounts)) {
			logger.warn("接收人的信息为空，无法发送消息");
			return;
		}

		ImTemplate imTemplate = getImTemplateService().queryImTemplateByBusinessType(businessTypeCode);
		if (imTemplate == null) {
			logger.warn("im消息模板为空，无法发送消息");
			return;
		}

		String imContent = imTemplate.getTemplateContent();
		try {
			imContent = BeetlTemplateUtils.renderFromString(imContent, "message", message);
		} catch (Exception e) {
			logger.error("使用模板生成消息内容失败", e);
		}

		Map<String, String> extras = new HashMap<>();
		extras.put("level", message.getMsgLevel().getLevel());
		String page = imTemplate.getPage();
		if (StringUtils.isNotBlank(page)) {
			extras.put("page", BeetlTemplateUtils.renderFromString(page, "message", message));
		}

		String pageParam = imTemplate.getPageParam();
		if (StringUtils.isNotBlank(pageParam)) {
			// key命名为orderId为了兼容之前前端的版本
			extras.put("orderId", BeetlTemplateUtils.renderFromString(pageParam, "message", message));
		}
		// key命名为from为了兼容之前前端的版本
		extras.put("from", imTemplate.getMessageFrom());
		String fromCode = imTemplate.getFromCode();
		if (StringUtils.isNotBlank(fromCode)) {
			// key命名为hisCode为了兼容之前前端的版本
			extras.put("hisCode", BeetlTemplateUtils.renderFromString(fromCode, "message", message));
		}
		logger.info("imContent====" + imContent);
		logger.info("page====" + extras.get("page"));
		logger.info("orderId====" + extras.get("orderId"));
		logger.info("from====" + extras.get("from"));
		logger.info("hisCode====" + extras.get("hisCode"));
		DicGroupItem item = DicUtils.getDicGroupItemByCode("business_type", message.getBusinessTypeCode());
		client.pushMessage(item.getName(), imContent, type, accounts, extras, companyCode, message.getSenderId());
	}

	private String getImMsgType(IMessage<?> message) {
		return TYPE_SYSTEM;
	}

	private Set<String> getAccount(List<String> receiverIdList) {
		Set<String> accounts = new HashSet<>();

		for (int i = 0; CollectionUtils.isNotEmpty(receiverIdList) && i < receiverIdList.size(); i++) {
			accounts.add(receiverIdList.get(i));
		}
		return accounts;
	}

}
