package com.els.base.msg.weixin;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import com.els.base.auth.entity.Role;
import com.els.base.auth.service.RoleService;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.service.user.UserService;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.utils.encryption.DESUtils;
import com.els.base.wechat.member.entity.WxMember;
import com.els.base.wechat.member.service.WxMemberService;
import com.els.base.wechat.oauth.service.WechatOauthService;

import me.chanjar.weixin.mp.bean.result.WxMpUser;

@Component
public class LoginByOpenidService implements WechatOauthService {

	private static Logger logger = LoggerFactory.getLogger(LoginByOpenidService.class); 
	
	@Autowired
	private WxMemberService wxMemberService;
	
	@Autowired
	private UserService userService;
	
	@Autowired
	protected RoleService roleService;

	@Autowired
	protected AuthenticationManager authenticationManager;

	@Override
	public void handleForOpenId(String openId, Map<String, String> paramsMap, HttpServletRequest request,
			HttpServletResponse response) {
		// 根据openid 找 wxMember
		WxMember wxMember = this.wxMemberService.queryByOpenId(openId);
		if (wxMember == null) {
			logger.warn("根据openid 查找 wxMember失败。 openid:{}", openId);
			return;
		}
		
		// 根据wxMember 找 User
		if (StringUtils.isBlank(wxMember.getMemberId())) {
			logger.warn("wxMember 没有绑定MemberId。 openid:{}", openId);
			return;
		}
		User user = this.userService.queryObjById(wxMember.getMemberId());
		
		// 根据user 登录
		if (user == null) {
			logger.warn("wxMember 绑定的User 为空。userId{}, openid:{}", wxMember.getMemberId(), openId);
			return;
		}
		
		this.loginByUser(user, request, response);
	}

	@Override
	public void handleForWxUser(WxMpUser wxMpUser, Map<String, String> paramsMap, HttpServletRequest request,
			HttpServletResponse response) {
		this.handleForOpenId(wxMpUser.getOpenId(), paramsMap, request, response);
	}
	
	private void loginByUser(User user, HttpServletRequest request, HttpServletResponse response) {

		Authentication authentication = new UsernamePasswordAuthenticationToken(user.getLoginName(), getPassword(user));
        Authentication authResult = authenticationManager.authenticate(authentication);
        
        com.els.base.auth.entity.User userAuthed = (com.els.base.auth.entity.User) authResult.getPrincipal();
        Collection<GrantedAuthority> roleList = new ArrayList<>();
        for (Role grantedAuthority : this.roleService.queryUserOwnRoles(user.getId())) {
			
            roleList.add(new SimpleGrantedAuthority(grantedAuthority.getRoleName()));
        }
		userAuthed.setAuthorities(roleList);
		
		// 在当前访问线程中设置 authResult
		SecurityContextHolder.getContext().setAuthentication(authResult);
		request.getSession().setAttribute("SPRING_SECURITY_CONTEXT", SecurityContextHolder.getContext());
	}
	
	private String getPassword(com.els.base.core.entity.user.User user) {
		Assert.isNotBlank(user.getPassword(), "帐号异常");
		
		String password = null;
		try {
			password = DESUtils.decrypt(user.getPassword(), Constant.SECRET_KEY);
			if (StringUtils.isNotBlank(user.getPasswordKey())) {
				password = password.replace(user.getPasswordKey(), "");
				password = DESUtils.decrypt(password, Constant.SECRET_KEY);
			}
		} catch (Exception e) {
			throw new CommonException("帐号异常", e);
		}
		
		return password;

	}

}
