package com.els.base.msg.mail.service.impl;

import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.service.user.UserService;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.msg.mail.dao.MailAccountMapper;
import com.els.base.msg.mail.entity.MailAccount;
import com.els.base.msg.mail.entity.MailAccountExample;
import com.els.base.msg.mail.service.MailAccountService;
import com.els.base.msg.mail.utils.SmtpSendUtils;
import com.els.base.utils.encryption.DESUtils;

@Service("defaultMailAccountService")
public class MailAccountServiceImpl implements MailAccountService {
	@Resource
	protected MailAccountMapper mailAccountMapper;

	@Resource
	protected UserService userService;

	@CacheEvict(value = { "mailAccount" }, allEntries = true)
	@Override
	public void addObj(MailAccount mailAccount) {
		Assert.isNotBlank(mailAccount.getPassword(), "密码不能为空");
		Assert.isNotBlank(mailAccount.getMailAddress(), "邮箱地址不能为空");
		Assert.isNotBlank(mailAccount.getUserLoginName(), "用户帐号不能为空");
		Assert.isNotBlank(mailAccount.getCompanyId(), "企业id不能为空");
		Assert.isNotBlank(mailAccount.getStmpServerHost(), "STMP服务器地址 不能为空");
		Assert.isNotNull(mailAccount.getStmpServerPort(), "STMP服务器端口 不能为空");

		if (!Constant.YES_INT.equals(mailAccount.getStmpServerIsSsl())
				&& !Constant.NO_INT.equals(mailAccount.getStmpServerIsSsl())) {
			// 默认不设置加密
			mailAccount.setStmpServerIsSsl(Constant.NO_INT);
		}

		MailAccountExample mailAccountExample = new MailAccountExample();
		MailAccountExample.Criteria cri = mailAccountExample.createCriteria();

		cri.andCompanyIdEqualTo(mailAccount.getCompanyId()).andMailAddressEqualTo(mailAccount.getMailAddress());

		if (this.mailAccountMapper.countByExample(mailAccountExample) > 0) {
			throw new CommonException("系统已经存在该邮箱，请不要重复添加");
		}

		mailAccountExample.clear();
		cri = mailAccountExample.createCriteria();
		cri.andUserLoginNameEqualTo(mailAccount.getUserLoginName());

		if (this.mailAccountMapper.countByExample(mailAccountExample) > 0) {
			throw new CommonException("该帐号已经设置邮箱，请不要重复添加");
		}

		User user = this.userService.findByUserLoginName(mailAccount.getUserLoginName());
		if (user == null) {
			throw new CommonException("该帐号不存在");
		}

		mailAccount.setUserId(user.getId());

		mailAccountExample.clear();
		cri = mailAccountExample.createCriteria();
		cri.andCompanyIdEqualTo(mailAccount.getCompanyId());

		if (this.mailAccountMapper.countByExample(mailAccountExample) == 0) {
			// 默认第一个邮箱帐号为默认
			mailAccount.setIsDefault(Constant.YES_INT);
		} else {
			mailAccount.setIsDefault(Constant.NO_INT);
		}

		mailAccount.setCreateTime(new Date());
		String password;

		try {
			password = DESUtils.encrypt(mailAccount.getPassword(), Constant.SECRET_KEY);
			mailAccount.setPassword(password);

		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		try {
			SmtpSendUtils.testConnection(mailAccount);
		} catch (Exception e) {
			throw new CommonException("帐号服务器连接验证失败");
		}

		this.mailAccountMapper.insertSelective(mailAccount);
	}

	@Transactional
	@CacheEvict(value = { "mailAccount" }, allEntries = true)
	@Override
	public void deleteObjById(String id) {
		MailAccount mailAccount = this.mailAccountMapper.selectByPrimaryKey(id);
		this.mailAccountMapper.deleteByPrimaryKey(id);
		
		MailAccountExample example = new MailAccountExample();
		example.createCriteria().andCompanyIdEqualTo(mailAccount.getCompanyId());
		if (this.mailAccountMapper.countByExample(example) == 0) {
			throw new CommonException("至少需要一个邮箱帐号来收取邮件消息");
		}
		
		if (this.mailAccountMapper.countByExample(example) >1) {
			return;
		}
		
		MailAccount temp = new MailAccount();
		temp.setIsDefault(Constant.YES_INT);
		this.mailAccountMapper.updateByExampleSelective(temp, example);
	}

	@CacheEvict(value = { "mailAccount" }, allEntries = true)
	@Override
	public void modifyObj(MailAccount mailAccount) {
		Assert.isNotBlank(mailAccount.getId(), "id 为空，无法更新");
		Assert.isNotBlank(mailAccount.getPassword(), "密码不能为空");
		Assert.isNotBlank(mailAccount.getMailAddress(), "邮箱地址不能为空");
		Assert.isNotBlank(mailAccount.getUserLoginName(), "用户帐号不能为空");
		Assert.isNotBlank(mailAccount.getStmpServerHost(), "STMP服务器地址 不能为空");
		Assert.isNotNull(mailAccount.getStmpServerPort(), "STMP服务器端口 不能为空");

		// 修改时不修改默认值
		mailAccount.setIsDefault(null);

		MailAccountExample mailAccountExample = new MailAccountExample();
		MailAccountExample.Criteria cri = mailAccountExample.createCriteria();

		cri.andCompanyIdEqualTo(mailAccount.getCompanyId()).andMailAddressEqualTo(mailAccount.getMailAddress())
				.andIdNotEqualTo(mailAccount.getId());

		if (this.mailAccountMapper.countByExample(mailAccountExample) > 0) {
			throw new CommonException("系统已经存在该邮箱，请不要重复添加");
		}

		mailAccountExample.clear();
		cri = mailAccountExample.createCriteria();
		cri.andUserLoginNameEqualTo(mailAccount.getUserLoginName()).andIdNotEqualTo(mailAccount.getId());

		if (this.mailAccountMapper.countByExample(mailAccountExample) > 0) {
			throw new CommonException("该帐号已经设置邮箱，请不要重复添加");
		}

		User user = this.userService.findByUserLoginName(mailAccount.getUserLoginName());
		if (user == null) {
			throw new CommonException("该帐号不存在");
		}
		mailAccount.setUserId(user.getId());

		if (StringUtils.isNotBlank(mailAccount.getUserId())) {
			mailAccountExample.clear();
			cri = mailAccountExample.createCriteria();
			cri.andUserIdEqualTo(mailAccount.getUserId()).andIdNotEqualTo(mailAccount.getId());

			if (this.mailAccountMapper.countByExample(mailAccountExample) > 0) {
				throw new CommonException("该帐号已经设置邮箱，请不要重复添加");
			}
		}

		if (StringUtils.isNotBlank(mailAccount.getPassword())) {
			MailAccount mail = this.mailAccountMapper.selectByPrimaryKey(mailAccount.getId());

			if (!mail.getPassword().equals(mailAccount.getPassword())) {
				// 修改密码
				try {
					String enPassword = DESUtils.encrypt(mailAccount.getPassword(), Constant.SECRET_KEY);
					mailAccount.setPassword(enPassword);
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
			}
		}
		
		try {
			SmtpSendUtils.testConnection(mailAccount);
		} catch (Exception e) {
			throw new CommonException("帐号服务器连接验证失败");
		}
		this.mailAccountMapper.updateByPrimaryKeySelective(mailAccount);
	}

	@Cacheable(value = "mailAccount", keyGenerator = "redisKeyGenerator")
	@Override
	public MailAccount queryObjById(String id) {
		return this.mailAccountMapper.selectByPrimaryKey(id);
	}

	@Cacheable(value = "mailAccount", keyGenerator = "redisKeyGenerator")
	@Override
	public List<MailAccount> queryAllObjByExample(MailAccountExample example) {
		return this.mailAccountMapper.selectByExample(example);
	}

	@Cacheable(value = "mailAccount", keyGenerator = "redisKeyGenerator")
	@Override
	public PageView<MailAccount> queryObjByPage(MailAccountExample example) {
		PageView<MailAccount> pageView = example.getPageView();
		pageView.setQueryResult(this.mailAccountMapper.selectByExampleByPage(example));
		return pageView;
	}

	@Cacheable(value = "mailAccount", keyGenerator = "redisKeyGenerator")
	@Override
	public MailAccount queryByUserId(String userId) {
		if (StringUtils.isBlank(userId)) {
			throw new CommonException("用户id不能为空");
		}

		MailAccountExample mailAccountExample = new MailAccountExample();
		MailAccountExample.Criteria cri = mailAccountExample.createCriteria();

		cri.andUserIdEqualTo(userId);

		List<MailAccount> accountList = this.mailAccountMapper.selectByExample(mailAccountExample);
		if (CollectionUtils.isEmpty(accountList)) {
			return null;
		}
		return accountList.get(0);
	}

	@Cacheable(value = "mailAccount", keyGenerator = "redisKeyGenerator")
	@Override
	public MailAccount queryDefaultMailAccoutByCompanyId(String companyId) {
//		if (StringUtils.isBlank(companyId)) {
//			throw new CommonException("用户id不能为空");
//		}

		MailAccountExample mailAccountExample = new MailAccountExample();
		MailAccountExample.Criteria cri = mailAccountExample.createCriteria();
		
		if (StringUtils.isNotBlank(companyId)) {
			cri.andCompanyIdEqualTo(companyId);
		}
		cri.andIsDefaultEqualTo(Constant.YES_INT);

		mailAccountExample.setOrderByClause("CREATE_TIME ASC");

		List<MailAccount> list = this.mailAccountMapper.selectByExample(mailAccountExample);
		if (CollectionUtils.isNotEmpty(list)) {
			return list.get(0);
		}

		mailAccountExample.clear();
		cri = mailAccountExample.createCriteria();
		cri.andCompanyIdEqualTo(companyId);
		mailAccountExample.setOrderByClause("CREATE_TIME ASC");

		list = this.mailAccountMapper.selectByExample(mailAccountExample);
		if (CollectionUtils.isEmpty(list)) {
			return null;
		}

		return list.get(0);
	}

	@CacheEvict(value = { "mailAccount" }, allEntries = true)
	@Transactional
	@Override
	public void setMailAccountAsDefault(String companyId, String mailAccountId) {
		if (StringUtils.isBlank(mailAccountId)) {
			throw new CommonException("邮箱帐号id都不能为空");
		}

		MailAccount mailAccount = this.mailAccountMapper.selectByPrimaryKey(mailAccountId);
		if (mailAccount == null) {
			throw new CommonException("该邮箱不存在");
		}

//		if (!companyId.equals(mailAccount.getCompanyId())) {
//			throw new CommonException("该邮箱不属于该公司的，不能设置默认");
//		}

		MailAccountExample example = new MailAccountExample();
		if (StringUtils.isNotBlank(companyId)) {
			example.createCriteria().andCompanyIdEqualTo(companyId);
		}

		// 先设置全部都不是默认
		MailAccount temp = new MailAccount();
		temp.setIsDefault(Constant.NO_INT);
		this.mailAccountMapper.updateByExampleSelective(temp, example);

		// 设置为默认
		temp.setIsDefault(Constant.YES_INT);
		temp.setId(mailAccountId);
		this.mailAccountMapper.updateByPrimaryKeySelective(temp);
	}

	@CacheEvict(value = { "mailAccount" }, allEntries = true)
	@Override
	public void deleteByExample(MailAccountExample example) {
		Assert.isNotNull(example, "参数不能为空");
    	Assert.isNotEmpty(example.getOredCriteria(), "批量删除不能全表删除");
    	
    	this.mailAccountMapper.deleteByExample(example);		
	}

	@CacheEvict(value = { "mailAccount" }, allEntries = true)
	@Transactional
	@Override
	public void addAll(List<MailAccount> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		
		list.forEach(record ->{
			this.mailAccountMapper.insertSelective(record);		
		});		
	}
}