package com.els.base.msg.mail;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.els.base.core.entity.user.User;
import com.els.base.core.service.user.UserService;
import com.els.base.msg.IMessage;
import com.els.base.msg.IMessageCommand;
import com.els.base.msg.mail.entity.MailAccount;
import com.els.base.msg.mail.entity.MailTemplate;
import com.els.base.msg.mail.service.MailAccountService;
import com.els.base.msg.mail.service.MailTemplateService;
import com.els.base.msg.mail.utils.SmtpSendUtils;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.reflect.ReflectUtils;
import com.els.base.utils.template.BeetlTemplateUtils;

/**
 * MailMessageCommand
 * 
 * @date 2017年10月11日 下午5:28:37
 * @version 1.0
 */
public class MailMessageCommand implements IMessageCommand {

	private static final long serialVersionUID = 1L;

	private static Logger logger = LoggerFactory.getLogger(MailMessageCommand.class);

	protected static MailAccountService getMailAccountService() {
		return SpringContextHolder.getOneBean(MailAccountService.class);
	}

	protected static MailTemplateService getMailTemplateService() {
		return SpringContextHolder.getOneBean(MailTemplateService.class);
	}

//	protected static CompanyService getCompanyService() {
//		return SpringContextHolder.getOneBean(CompanyService.class);
//	}
//
//	protected static CompanyUserRefService getCompanyUserRefService() {
//		return SpringContextHolder.getOneBean(CompanyUserRefService.class);
//	}

	protected static UserService getUserService() {
		return SpringContextHolder.getOneBean(UserService.class);
	}

	@Override
	public void sendMsg(IMessage<?> message) throws Exception {
		String typeCode = message.getBusinessTypeCode();
		if (StringUtils.isBlank(message.getSenderId()) && StringUtils.isBlank(message.getCompanyCode())) {
			logger.warn("发送者的信息为空，无法发送邮件");
			return;
		}

		MailAccount sendMailAccount = getSenderMailAccountByUserId(message);

		if (sendMailAccount == null) {
			logger.warn("发送者的信息为空，无法发送邮件");
			return;
		}

		List<String> receiverIdList = message.getReceiverIdList();
		if (CollectionUtils.isEmpty(receiverIdList)) {
			logger.warn("接收人的信息为空，无法发送邮件");
			return;
		}

		MailTemplate mailTemplate = getMailTemplateService().queryMailTemplateByBusinessType(typeCode);
		if (mailTemplate == null) {
			logger.warn("邮箱模板为空，无法发出邮件");
			return;
		}

		for (String receiveId : receiverIdList) {

			MailAccount receiverMailAccount = getReceiverMailAccountByUserId(receiveId);
			if (receiverMailAccount == null) {
				logger.warn("接收人的邮箱信息为空无法发送邮件");
				continue;
			}
			try {
				sendMsg(receiverMailAccount, sendMailAccount, message, mailTemplate);

			} catch (Exception e) {
				logger.error("发送邮件失败", e);
			}
		}
	}

	/**
	 * 根据发送者id查找 发件箱的配置
	 * 
	 * @param sendId
	 * @param message
	 * @return
	 */
	private MailAccount getSenderMailAccountByUserId(IMessage<?> message) {
		MailAccount sendMailAccount = null;
		if (StringUtils.isNotBlank(message.getSenderId())) {
			sendMailAccount = getMailAccountService().queryByUserId(message.getSenderId());
		}

		if (sendMailAccount != null) {
			return sendMailAccount;
		}

		String companyId = this.getCompanyId(message);
		sendMailAccount = getMailAccountService().queryDefaultMailAccoutByCompanyId(companyId);
		return sendMailAccount;
	}

	private MailAccount getReceiverMailAccountByUserId(String receiveId) {

		if (StringUtils.isBlank(receiveId)) {
			return null;
		}

		MailAccount receiverMailAccount = getMailAccountService().queryByUserId(receiveId);
		if (receiverMailAccount != null) {
			return receiverMailAccount;
		}

		User user = getUserService().queryObjById(receiveId);
		if (user != null && StringUtils.isNotBlank(user.getEmail())) {
			MailAccount mailAccount = new MailAccount();
			mailAccount.setUserId(receiveId);
			mailAccount.setMailAddress(user.getEmail());
			mailAccount.setUserName(user.getNickName());
			return mailAccount;
		}

		String companyId = this.getComanyIdByReceiverId(receiveId);
		receiverMailAccount = getMailAccountService().queryDefaultMailAccoutByCompanyId(companyId);

		if (receiverMailAccount != null) {
			return receiverMailAccount;
		}

		if (StringUtils.isNotBlank(companyId)) {
			return this.getMaillAccoutByCompanyId(companyId);
		}
		
		return null;
	}

	/**
	 * 发送单个邮件给收件人
	 * 
	 * @param receiverMailAccount
	 * @param sendMailAccount
	 * @param data
	 * @param mailTemplate
	 * @throws Exception
	 */
	protected void sendMsg(MailAccount receiverMailAccount, MailAccount sendMailAccount, Object data,
			MailTemplate mailTemplate) throws Exception {

		Map<String, Object> mailData = new HashMap<>();
		mailData.put("data", data);
		mailData.put("sender", sendMailAccount);
		mailData.put("receiver", receiverMailAccount);

		String mailContentHtml = mailTemplate.getTemplateContent();
		try {
			mailContentHtml = BeetlTemplateUtils.renderFromString(mailContentHtml, "message", data);
		} catch (Exception e) {
			logger.error("使用模板生成邮件内容失败", e);
		}

		String subject = mailTemplate.getTemplateTitle();
		try {
			subject = BeetlTemplateUtils.renderFromString(subject, "message", data);
		} catch (Exception e) {
			logger.error("使用模板生成邮件标题失败", e);
		}

		logger.debug("--------开始发送邮件------");
		SmtpSendUtils.sendMail(sendMailAccount, receiverMailAccount.getMailAddress(), receiverMailAccount.getUserName(),
				subject, mailContentHtml);
		logger.debug("--------结束发送邮件------");

	}
	
	private String getCompanyId(IMessage<?> message) {
		String companyId = null;
		try {
			Class<?> companyService = Class.forName("com.els.base.company.service.CompanyService");
			Object service = SpringContextHolder.getOneBean(companyService);
			
			if (StringUtils.isBlank(message.getCompanyCode())) {
				return companyId;
			}
			
	//		Company company = getCompanyService().queryCompanyByCode(message.getCompanyCode());
			Object company = ReflectUtils.invokeMethod(service, "queryCompanyByCode", message.getCompanyCode());
			companyId = (String) ReflectUtils.getValue(company, "id");
			
		} catch (Exception e) {
			logger.error("获取公司信息失败", e);
		}
		return companyId;
	}
	
	private String getComanyIdByReceiverId(String receiveId) {
		String companyId = null;
		
		try {
			Class<?> companyUserRefServiceClass = Class.forName("com.els.base.company.service.CompanyUserRefService");
			Object companyUserRefService = SpringContextHolder.getOneBean(companyUserRefServiceClass);
			companyId = (String) ReflectUtils.invokeMethod(companyUserRefService, "queryCompanyIdOfUser", receiveId);
			
		} catch (Exception e) {
			logger.error("获取公司信息失败", e);
		}
		
		return companyId;
	}
	
	private MailAccount getMaillAccoutByCompanyId(String companyId) {
		MailAccount mailAccount = null;
		
		try {
			Class<?> companyService = Class.forName("com.els.base.company.service.CompanyService");
			Object service = SpringContextHolder.getOneBean(companyService);
			
	//		Company company = getCompanyService().queryObjById(companyId);
//			mailAccount.setMailAddress(company.getEmail());
//			mailAccount.setUserName(company.getCompanyFullName());
			
			Object company = ReflectUtils.invokeMethod(service, "queryObjById", companyId);
			String email = (String) ReflectUtils.getValue(company, "email");
			String companyFullName = (String) ReflectUtils.getValue(company, "companyFullName");
			
			mailAccount = new MailAccount();
			mailAccount.setMailAddress(email);
			mailAccount.setUserName(companyFullName);

		} catch (Exception e) {
			logger.error("获取公司信息失败", e);
		}
		
		return mailAccount;
	}

}