package com.els.base.core.service.user.impl;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.dao.user.UserMapper;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.user.User;
import com.els.base.core.entity.user.UserExample;
import com.els.base.core.event.user.UserCreatedEvent;
import com.els.base.core.exception.CommonException;
import com.els.base.core.service.user.UserService;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.encryption.DESUtils;

/**
 * 用户管理
 * 
 * @author hzy
 *
 */
@Service("userService")
public class UserServiceImpl implements UserService {
	
	private static Logger logger = LoggerFactory.getLogger(UserServiceImpl.class);
	
    @Autowired
    private UserMapper userMapper;

    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public void addUser(List<User> usersList) throws Exception {

        //批量插入
        for (User user : usersList) {
           this.addObj(user,false);
        }

    }

    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public void modifyPwd(String userId, String oldPwd, String newPwd) throws Exception {
    	
    	if (StringUtils.isBlank(userId)) throw new CommonException("参数异常，用户id不为空", "base_param_error", "用户id不为空");
    	if (StringUtils.isBlank(oldPwd)) throw new CommonException("参数异常，旧密码不为空", "base_param_error", "旧密码不为空");
    	if (StringUtils.isBlank(newPwd)) throw new CommonException("参数异常，新密码不为空", "base_param_error", "新密码不为空");

        if (!this.checkPassword(userId, oldPwd)) {
            throw new CommonException("密码错误", "base_account_password_worn");
        }
        this.updatePwd(newPwd, userId);
    }

    @Cacheable(value = "userService", keyGenerator="redisKeyGenerator")
    @Override
    public List<User> queryAllUserIsDisabled() {
        UserExample userExample = new UserExample();
        userExample.createCriteria().andIsEnableEqualTo(Constant.NO_INT);
        return this.userMapper.selectByExample(userExample);
    }

    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        if (StringUtils.isBlank(id)) {
            throw new IllegalArgumentException("id 不能少于等于0");
        }
        if ("1".equals(id.trim())) {
            throw new CommonException("管理员,不允许操作", "admin_not_allow_oper");
        }
        this.userMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public void modifyObj(User user) {
        // 检查账号不能为空
        if (user == null || user.getId() == null) {
            throw new CommonException("id 不能为空", "id_is_blank");
        }

        // 检查用户名长度
        if (StringUtils.isNotBlank(user.getLoginName())) {
        	user.setLoginName(user.getLoginName().toLowerCase());
        	
        	if (!user.getLoginName().matches("^\\w([\\w\\-]){4,254}")) {
                throw new CommonException("帐号需要5个以上字符，只能包含字母或数字，或者横杠，或下划线。", "invaild_format", "账号",
                        "需要5个以上字符，只能包含字母或数字，或者横杠，或下划线。");
            }
        	
            if (user.getLoginName().length() > 255) {
                throw new CommonException("用户账号名称不能超过规定长度", "length_canot_exceed", "用户账号名称", 255);
            }

            UserExample loginNameExample = new UserExample();
            loginNameExample.createCriteria()
                .andLoginNameEqualTo(user.getLoginName().toLowerCase())
                .andIdNotEqualTo(user.getId());
            if (this.userMapper.countByExample(loginNameExample) > 0) {
                throw new CommonException("账号名称已存在", "base_is_exists", "账号");
            }
        }

        if (StringUtils.isNotBlank(user.getMobilePhone())) {
            if (user.getMobilePhone().length() > 11) {
                throw new CommonException("用户手机号不能超过规定长度", "length_canot_exceed", "用户手机号", 11);
            }

            UserExample mobilePhoneExample = new UserExample();
            mobilePhoneExample.createCriteria()
                .andMobilePhoneEqualTo(user.getMobilePhone())
                .andIdNotEqualTo(user.getId());
            if (this.userMapper.countByExample(mobilePhoneExample) > 0) {
                throw new CommonException("手机号已存在", "base_the_same_mobile");
            }
        }

        if (StringUtils.isNotBlank(user.getEmail())) {
            if (user.getEmail().length() > 255) {
                throw new CommonException("用户邮箱号不能超过规定长度", "length_canot_exceed", "用户邮箱号", 255);
            }

            UserExample emailExample = new UserExample();
            emailExample.createCriteria()
                .andEmailEqualTo(user.getEmail().toLowerCase())
                .andIdNotEqualTo(user.getId());
            if (this.userMapper.countByExample(emailExample) > 0) {
                throw new CommonException("Email已存在", "base_the_same_email");
            }
        }

        // 检查是否重复值

        if (user.getNickName() != null && user.getNickName().length() > 255) {
            throw new CommonException("昵称长度不能超过255", "length_canot_exceed", "昵称", "255");
        }

        Date date = new Date();
        // 修改用户
        user.setUpdateTime(date);
        this.userMapper.updateByPrimaryKeySelective(user);
    }

    /**
     * 更改密码--修改密码
     * 
     * @throws Exception
     */
    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public void updatePwd(String password, String userId) throws Exception {
        // 修改用户密码
        String passwordKey = StringUtils.leftPad(String.valueOf(RandomUtils.nextInt(1000000)), 6, "0");

        String enPassword = DESUtils.encrypt(DESUtils.encrypt(password, Constant.SECRET_KEY) + passwordKey, Constant.SECRET_KEY);

        Date date = new Date();
        User tmp = new User();
        tmp.setId(userId);
        tmp.setPassword(enPassword);
        tmp.setPasswordKey(passwordKey);
        tmp.setExpireTime(date);
        // this.modifyObj(tmp);
        this.userMapper.updateByPrimaryKeySelective(tmp);
    }

    /**
     * 重置密码 重置密码为123456
     */
    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public void resetPwd(String password, String userId) throws Exception {
        // 修改用户密码
        String passwordKey = StringUtils.leftPad(String.valueOf(RandomUtils.nextInt(1000000)), 6, "0");

        String enPassword = DESUtils.encrypt(DESUtils.encrypt(password, Constant.SECRET_KEY) + passwordKey, Constant.SECRET_KEY);

        User tmp = new User();
        tmp.setId(userId);
        tmp.setPassword(enPassword);
        tmp.setPasswordKey(passwordKey);
        tmp.setExpireTime(null);
        // this.modifyObj(tmp);
        this.userMapper.updateByPrimaryKeySelective(tmp);
    }

    /**
     * 更改密码--验证旧密码
     */
    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public boolean checkPassword(String userId, String oldPwd) {
    	if (StringUtils.isBlank(userId) || StringUtils.isBlank(oldPwd)) {
			return false;
		}
    	
        User user = this.userMapper.selectByPrimaryKey(userId);
        try {

        	String enPassword = null;
            String salt = user.getPasswordKey();
            if (StringUtils.isNotBlank(salt)) {
            	enPassword = DESUtils.encrypt(DESUtils.encrypt(oldPwd, Constant.SECRET_KEY) + salt, Constant.SECRET_KEY);
				
			}else {
				enPassword = DESUtils.encrypt(oldPwd, Constant.SECRET_KEY);
			}

			return enPassword.equals(user.getPassword());

        } catch (Exception e) {
        	logger.error("密码校验失败", e);
            return false;
        }

    }

    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public void disableUserById(String userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        if (user == null) {
            throw new CommonException("找不到用户", "user_not_found");
        }

        User tmp = new User();
        tmp.setId(user.getId());

        Integer isEnable = user.getIsEnable();
        if (Constant.YES_INT.equals(isEnable)) {
            tmp.setIsEnable(Constant.NO_INT);

        } else {
            tmp.setIsEnable(Constant.YES_INT);
        }

        this.userMapper.updateByPrimaryKeySelective(tmp);

    }

    @Cacheable(value = "userService", keyGenerator="redisKeyGenerator")
    @Override
    public User queryObjById(String id) {
        return userMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "userService", keyGenerator="redisKeyGenerator")
    @Override
    public User findByUserLoginName(String params) {
    	Assert.isNotBlank(params, "登录帐号不能为空");
    	
        UserExample userExample = new UserExample();
        userExample.createCriteria()
		.andLoginNameEqualTo(params.toLowerCase());
	
        userExample.or()
		.andLoginNameEqualTo(params.toUpperCase());

        List<User> list = this.userMapper.selectByExample(userExample);
        if (list == null || list.size() == 0) {
            return null;
        }
        return list.get(0);
    }

    @CacheEvict(value = { "userService" }, allEntries = true)
    @Override
    public void disableMembers(String userId, Integer status) {
        User user = new User();
        user.setId(userId);
        user.setIsEnable(status);
        this.userMapper.updateByPrimaryKeySelective(user);
    }

    @CacheEvict(value = { "userService" }, allEntries = true)
    @Transactional
    @Override
    public void addObj(User user) {
        this.addObj(user,true);
    }

    @CacheEvict(value = { "userService" }, allEntries = true)
    @Transactional
    @Override
    public void addObj(User user,Boolean isPublishEvent) {

        Assert.isNotBlank(user.getPassword(), "密码不能为空");
        Assert.isNotBlank(user.getProjectCode(), "项目编码不能为空");
        Assert.isNotBlank(user.getLoginName(), "登录帐号不能为空");
        Assert.isNotBlank(user.getNickName(), "帐号昵称不能为空");

        //用户名不区分大小写，全部都转为小写
        user.setLoginName(user.getLoginName().toLowerCase());

        if (!user.getLoginName().matches("^\\w([\\w\\-]){4,254}")) {
            throw new CommonException("帐号需要5个以上字符，只能包含字母或数字，或者横杠，或下划线。", "invaild_format", "账号",
                    "帐号需要5个以上字符，只能包含字母或数字，或者横杠，或下划线。");
        }

        if (user.getLoginName().length() > 255 ) {
            throw new CommonException("账号长度不能超过255", "length_canot_exceed", "账号", "255");
        }

        if (StringUtils.isNotBlank(user.getEmail())) {
            user.setEmail(user.getEmail().toLowerCase());
        }

        UserExample userEx1 = new UserExample();
        userEx1.createCriteria().andLoginNameEqualTo(user.getLoginName());

        if (this.userMapper.countByExample(userEx1) > 0) {
            throw new CommonException("相同用户已存在", "base_the_same_account");
        }

        if (StringUtils.isNotBlank(user.getMobilePhone())) {
            if (user.getMobilePhone().length() > 11) {
                throw new CommonException("用户手机号不能超过规定长度", "length_canot_exceed", "用户手机号", 11);
            }

            UserExample mobilePhoneExample = new UserExample();
            mobilePhoneExample.createCriteria().andMobilePhoneEqualTo(user.getMobilePhone());
            if (this.userMapper.countByExample(mobilePhoneExample) > 0) {
                throw new CommonException("手机号已存在", "base_the_same_mobile");
            }
        }

        if (StringUtils.isNotBlank(user.getEmail())) {
            if (user.getEmail().length() > 255) {
                throw new CommonException("用户邮箱号不能超过规定长度", "length_canot_exceed", "用户邮箱号", 255);
            }

            UserExample emailExample = new UserExample();
            emailExample.createCriteria().andEmailEqualTo(user.getEmail());
            if (this.userMapper.countByExample(emailExample) > 0) {
                throw new CommonException("Email已存在", "base_the_same_email");
            }
        }

        String passwordKey = StringUtils.leftPad(String.valueOf(RandomUtils.nextInt(1000000)), 6, "0");
        user.setPasswordKey(passwordKey);

        String password;
        try {

            password = DESUtils.encrypt(DESUtils.encrypt(user.getPassword(), Constant.SECRET_KEY) + passwordKey, Constant.SECRET_KEY);
            user.setPassword(password);
            user.setPasswordKey(passwordKey);

            // 新增用户
            this.userMapper.insertSelective(user);

            //如果是指定不开启，那么才不开启
            if(isPublishEvent==null){
                isPublishEvent=true;
            }
            if(isPublishEvent){
                SpringContextHolder.getApplicationContext().publishEvent(new UserCreatedEvent(user));
            }

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }

    }
    
    @CacheEvict(value = { "userService" }, allEntries = true)
    @Transactional
    @Override
	public void addUserFromMdm(User user) {
		Assert.isNotBlank(user.getLoginName(), "登录帐号不能为空");
		
		//用户名不区分大小写，全部都转为小写
        user.setLoginName(user.getLoginName().toLowerCase());
        if (!user.getLoginName().matches("^\\w([\\w\\-]){4,254}")) {
            throw new CommonException("帐号需要5个以上字符，只能包含字母或数字，或者横杠，或下划线。", "invaild_format", "账号",
                    "帐号需要5个以上字符，只能包含字母或数字，或者横杠，或下划线。");
        }

        if (user.getLoginName().length() > 255 ) {
            throw new CommonException("账号长度不能超过255", "length_canot_exceed", "账号", "255");
        }
        UserExample userEx1 = new UserExample();
        userEx1.createCriteria().andLoginNameEqualTo(user.getLoginName());

        if (this.userMapper.countByExample(userEx1) > 0) {
            throw new CommonException("相同用户已存在", "base_the_same_account");
        }
        
		String passwordKey = StringUtils.leftPad(String.valueOf(RandomUtils.nextInt(1000000)), 6, "0");
		user.setPasswordKey(passwordKey);

		String password;
		try {

			password = DESUtils.encrypt(DESUtils.encrypt(User.DEFAULT_PASSWORD, Constant.SECRET_KEY) + passwordKey,
					Constant.SECRET_KEY);
			user.setPassword(password);
			user.setPasswordKey(passwordKey);

			// 新增用户
			this.userMapper.insertSelective(user);
			SpringContextHolder.getApplicationContext().publishEvent(new UserCreatedEvent(user));
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}


    @Cacheable(value = "userService", keyGenerator="redisKeyGenerator")
    @Override
    public List<User> queryAllObjByExample(UserExample example) {
        return this.userMapper.selectByExample(example);
    }

    @Cacheable(value = "userService", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<User> queryObjByPage(UserExample example) {
        PageView<User> pageView = example.getPageView();
        pageView.setQueryResult(this.userMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Cacheable(value = "userService", keyGenerator="redisKeyGenerator")
    @Override
    public boolean findUserIsExist(User user) {
    	if (StringUtils.isBlank(user.getLoginName()) 
    			&& StringUtils.isBlank(user.getMobilePhone())
    			&& StringUtils.isBlank(user.getMobilePhone())) {
			throw new CommonException("参数异常，无法查出是否存在重复用户","base_param_error","无法查出是否存在重复用户");
		}
    	
        UserExample userExample = new UserExample();
        userExample.createCriteria()
                   .andMobilePhoneEqualTo(user.getMobilePhone());
        userExample.or()
                   .andEmailEqualTo(user.getEmail().toLowerCase());
        userExample.or()
                   .andLoginNameEqualTo(user.getLoginName().toLowerCase());

        return this.userMapper.countByExample(userExample) > 0;
    }
    
    
    @CacheEvict(value = { "userService" }, allEntries = true)
	@Override
	public void deleteUserInfo(UserExample example) {
		this.userMapper.deleteByExample(example);
	}

//    public static void main(String[] args) {
//        System.out.println("test_admin".matches("^\\w([\\w\\-]){4,254}"));
//    }

}
