package com.els.base.core.service.dictionary.impl;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import com.els.base.core.dao.dictionary.DicGroupItemMapper;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.dictionary.DicGroup;
import com.els.base.core.entity.dictionary.DicGroupExample;
import com.els.base.core.entity.dictionary.DicGroupItem;
import com.els.base.core.entity.dictionary.DicGroupItemExample;
import com.els.base.core.exception.CommonException;
import com.els.base.core.service.dictionary.DicGroupItemService;
import com.els.base.core.service.dictionary.DicGroupService;

@Service("dicGroupItemService")
public class DicGroupItemServiceImpl implements DicGroupItemService {

    @Resource
    protected DicGroupItemMapper dicGroupItemMapper;

    @Resource
    protected DicGroupService dicGroupService;

    @CacheEvict(value = "dicGroupItem", allEntries = true)
    @Override
    public void addObj(DicGroupItem t) {
        if (StringUtils.isBlank(t.getCode()) || t.getGroupId() == null) {
            throw new CommonException("参数不能为空", "base_canot_be_null", "参数");
        }
        if (t.getCode().length() > 100 || t.getName().length() > 100) {
            throw new CommonException("编码或名称长度不能超过100", "length_canot_exceed", "编码或名称", "100");
        }
        DicGroupItemExample example = new DicGroupItemExample();
        example.createCriteria().andCodeEqualTo(t.getCode()).andGroupIdEqualTo(t.getGroupId());
        if (this.dicGroupItemMapper.countByExample(example) > 0) {
            throw new CommonException("code值已存在", "base_is_exists", "code值");
        }
        this.dicGroupItemMapper.insertSelective(t);
    }

    @CacheEvict(value = "dicGroupItem", allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.dicGroupItemMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = "dicGroupItem", allEntries = true)
    @Override
    public void deleteItemsByGroupId(String dicGroupId) {
        DicGroupItemExample example = new DicGroupItemExample();
        example.createCriteria().andGroupIdEqualTo(dicGroupId);

        this.dicGroupItemMapper.deleteByExample(example);
    }

    @CacheEvict(value = "dicGroupItem", allEntries = true)
    @Override
    public void modifyObj(DicGroupItem t) {
        // 空值的校验
        if (t == null || t.getGroupId() == null  || StringUtils.isBlank(t.getCode())
                || StringUtils.isBlank(t.getValue()) || StringUtils.isBlank(t.getName())) {
            throw new CommonException("参数不完整，请填写必要的信息", "parameter_incomplete");
        }

        // 数据格式的校验
        if (!t.getCode().matches("[\\w\\_\\-]+")) {
            throw new CommonException("编码的格式不正确", "base_invaild_format", "编码");
        }
        if (t.getCode().length() > 100 || t.getName().length() > 100) {
            throw new CommonException("编码或名称长度不能超过100", "length_canot_exceed", "编码或名称", "100");
        }
        this.dicGroupItemMapper.updateByPrimaryKeySelective(t);

    }

    @Cacheable(value = "dicGroupItem", keyGenerator="redisKeyGenerator")
    @Override
    public DicGroupItem queryObjById(String id) {
        return this.dicGroupItemMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "dicGroupItem", keyGenerator="redisKeyGenerator")
    @Override
    public List<DicGroupItem> queryAllObjByExample(DicGroupItemExample example) {
        return this.dicGroupItemMapper.selectByExample(example);
    }

    @Cacheable(value = "dicGroupItem", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<DicGroupItem> queryObjByPage(DicGroupItemExample example) {
        PageView<DicGroupItem> pageView = example.getPageView();
        pageView.setQueryResult(this.dicGroupItemMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Cacheable(value = "dicGroupItem", keyGenerator="redisKeyGenerator")
    @Override
    public List<DicGroupItem> queryItemsByGroupCode(String dicGroupCode) {
        DicGroupExample dicGroupExample = new DicGroupExample();
        dicGroupExample.createCriteria().andCodeEqualTo(dicGroupCode);

        List<DicGroup> groupList = this.dicGroupService.queryAllObjByExample(dicGroupExample);
        if (CollectionUtils.isEmpty(groupList)) {
            return null;
        }

        if (groupList.size() > 1) {
            throw new RuntimeException("数据异常，dicGroupCode下,数据多余一条");
        }

        DicGroup dicGroup = groupList.get(0);
        DicGroupItemExample dicGroupItemExample = new DicGroupItemExample();
        dicGroupItemExample.createCriteria().andGroupIdEqualTo(dicGroup.getId());
        dicGroupExample.setOrderByClause("name ASC");

        return this.queryAllObjByExample(dicGroupItemExample);
    }

    @Cacheable(value = "dicGroupItem", keyGenerator="redisKeyGenerator")
    @Override
    public DicGroupItem queryItems(String dicGroupCode, String dicGroupItemCode) {
        if (StringUtils.isBlank(dicGroupCode) || StringUtils.isBlank(dicGroupItemCode)) {
            throw new NullPointerException("dicGroupCode, dicGroupItemCode参数都不能为空");
        }
        
        DicGroupExample dicGroupExample = new DicGroupExample();
        dicGroupExample.createCriteria().andCodeEqualTo(dicGroupCode);

        List<DicGroup> groupList = this.dicGroupService.queryAllObjByExample(dicGroupExample);
        if (CollectionUtils.isEmpty(groupList)) {
            return null;
        }

        if (groupList.size() > 1) {
            throw new RuntimeException("数据异常，dicGroupCode下,数据多于一条");
        }

        DicGroup dicGroup = groupList.get(0);
        
        DicGroupItemExample dicGroupItemExample = new DicGroupItemExample();
        dicGroupItemExample.createCriteria().andGroupIdEqualTo(dicGroup.getId()).andCodeEqualTo(dicGroupItemCode);
        dicGroupExample.setOrderByClause("name ASC");
        
        List<DicGroupItem> dicGroupItemList = this.dicGroupItemMapper.selectByExample(dicGroupItemExample);
        if (CollectionUtils.isEmpty(dicGroupItemList)) {
			return null;
		}
        
        if (dicGroupItemList.size() > 1) {
            throw new RuntimeException("数据异常，dicGroupCode dicGroupItemCode 下,数据多于一条");
        }

        return dicGroupItemList.get(0);
    }

    @Override
    public DicGroupItem queryItemByValue(String dicGroupCode, String dicGroupItemValue) {
        if (StringUtils.isBlank(dicGroupCode) || StringUtils.isBlank(dicGroupItemValue)) {
            throw new NullPointerException("dicGroupCode, dicGroupItemValue参数都不能为空");
        }

        List<DicGroupItem> list = this.queryItemsByGroupCode(dicGroupCode);
        for (int i = 0; i < list.size(); i++) {
            DicGroupItem dicGroupItem = list.get(i);
            if (dicGroupItem.getValue().equals(dicGroupItemValue)) {
                return dicGroupItem;
            }
        }

        return null;
    }

}
