package com.els.base.log.aop;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import com.els.base.auth.entity.User;
import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.log.entity.OperationLog;
import com.els.base.log.event.OperateOverEvent;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.json.JsonUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

@ConditionalOnProperty(havingValue="true",value="operation.log.is.enable")
@Aspect
@Component
public class OperationLogAspect {

	private static Logger logger = LoggerFactory.getLogger(OperationLogAspect.class);
	
	@Value("${operation.log.is.enable:false}")
	private String operateLogIsEnable;
	
	@Value("${operation.log.http.is.enable:true}")
	private String operateHttpLogIsEnable;
	
	@SuppressWarnings("rawtypes")
	@Around(value = "@annotation(apiOperation)", argNames = "apiOperation")
	public Object addLogAfterOpera(ProceedingJoinPoint joinPoint, ApiOperation apiOperation) throws Throwable {

		if (!Boolean.TRUE.toString().equals(this.operateLogIsEnable)) {
			return joinPoint.proceed();
		}
		
		if (!Boolean.TRUE.toString().equals(this.operateHttpLogIsEnable)) {
			return joinPoint.proceed();
		}
		
		OperationLog operationLog = this.createLog(joinPoint, apiOperation);

		try {
			Object object = joinPoint.proceed();
			
			String result = null;
			if (object instanceof ResponseResult) {
				result = ((ResponseResult) object).getMsg();
			}else{
				result = "success";
			}
			operationLog.setResult(result);
			SpringContextHolder.getApplicationContext().publishEvent(new OperateOverEvent(operationLog));

			return object;

		} catch (Throwable e) {
			operationLog.setResult("error:" + e.getMessage());
			SpringContextHolder.getApplicationContext().publishEvent(new OperateOverEvent(operationLog));
//			throw new RuntimeException(e);
			throw e;
		}
	}
	
	private OperationLog createLog(ProceedingJoinPoint joinPoint, ApiOperation apiOperation){
		OperationLog operationLog = new OperationLog();
		operationLog.setProjectId(ProjectUtils.getProjectId());

		try {
			User user = SpringSecurityUtils.getLoginUser();
			if (user != null) {
				operationLog.setUserId(user.getId());
				operationLog.setUserName(user.getUsername());
			}

			Class<?> classTarget = joinPoint.getTarget().getClass();
			Api api = classTarget.getAnnotation(Api.class);
			operationLog.setTarget(api.value());
			operationLog.setMethod(apiOperation.value());

			HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
			String ip = request.getRemoteAddr();

			String schemem = request.getScheme();
			String serverName = request.getServerName();
			String requestUri = request.getRequestURI();

			operationLog.setOriginIp(ip);
			operationLog.setOriginUrl(schemem + "://" + serverName + requestUri);
			operationLog.setParams(JsonUtils.writeValueAsString(this.filter(joinPoint.getArgs())));

		} catch (Exception e1) {
//			e1.printStackTrace();
			logger.error("日志记录异常", e1);
		}
		
		return operationLog;
	}

	private Object[] filter(Object[] objects) {
		List<Object> rmList = new ArrayList<>();
		for (int i = 0; i < objects.length; i++) {
			if (objects[i] instanceof HttpServletRequest) {
				rmList.add(objects[i]);
			}

			if (objects[i] instanceof HttpServletResponse) {
				rmList.add(objects[i]);
			}

			if (objects[i] instanceof HttpSession) {
				rmList.add(objects[i]);
			}
		}

		if (CollectionUtils.isEmpty(rmList)) {
			return objects;
		}

		for (Object element : rmList) {
			objects = ArrayUtils.removeElement(objects, element);
		}
		return objects;
	}

}
