package com.els.base.performance.web.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

import java.io.UnsupportedEncodingException;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.entity.User;
import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.company.entity.Company;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.performance.entity.KpiProject;
import com.els.base.performance.entity.KpiProjectExample;
import com.els.base.performance.entity.KpiTemplateItem;
import com.els.base.performance.entity.KpiTemplateItemExample;
import com.els.base.performance.service.KpiProjectService;
import com.els.base.performance.service.KpiTemplateItemService;

@Api(value = "绩效管理-评估项表")
@Controller
@RequestMapping("kpiProject")
public class KpiProjectController {
    @Resource
    protected KpiProjectService kpiProjectService;

    @Resource
    protected KpiTemplateItemService kpiTemplateItemService;

    @ApiOperation(httpMethod = "POST", value = "创建评估项表")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody KpiProject kpiProject)
            throws UnsupportedEncodingException {

        if (StringUtils.isBlank(kpiProject.getProjectName())) {
            throw new CommonException("评估项名称为空，保存失败", "base_canot_be_null", "评估项名称");
        }

        if (StringUtils.isBlank(kpiProject.getProjectTypeName())) {
            throw new CommonException("评估项类别名称为空，保存失败", "base_canot_be_null", "评估项类别名称");
        }

        if ((kpiProject.getProjectName() != null && getWordCountCode(kpiProject.getProjectName(), "UTF-8") > 300)
                || (kpiProject.getProjectStandard() != null && getWordCountCode(
                        kpiProject.getProjectStandard(), "UTF-8") > 2000)
                || (kpiProject.getProjectInstruction() != null && getWordCountCode(
                        kpiProject.getProjectInstruction(), "UTF-8") > 2000)
                || (kpiProject.getProjectMethodParameter() != null && getWordCountCode(
                        kpiProject.getProjectMethodParameter(), "UTF-8") > 1000)
                || (kpiProject.getProjectMethodParameterName() != null && getWordCountCode(
                        kpiProject.getProjectMethodParameterName(), "UTF-8") > 300)) {
            throw new CommonException("输入的参数长度超标", "database_length_error");
        }
        if(kpiProject.getProjectHandlType().equals("2")) {
            if (StringUtils.isBlank(kpiProject.getProjectHandlTaxon())) {
                throw new CommonException("文件名称不能为空");
            }
            if (StringUtils.isBlank(kpiProject.getProjectHandlNamespace())) {
                throw new CommonException("命名空间不能为空");
            }
        }
        // 获取当前登录人公司信息
        Company company = CompanyUtils.currentCompany();
        // 获取当前登录人用户信息
        User user = SpringSecurityUtils.getLoginUser();

        kpiProject.setCompany(company);
        kpiProject.setUser(user);

        this.kpiProjectService.addObj(kpiProject);
        return ResponseResult.success();
    }

    /* 按特定的编码格式获取长度 */
    public static int getWordCountCode(String str, String code) throws UnsupportedEncodingException {
        return str.getBytes(code).length;
    }

    @ApiOperation(httpMethod = "POST", value = "编辑评估项表")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody KpiProject kpiProject)
            throws UnsupportedEncodingException {
        if (StringUtils.isBlank(kpiProject.getId())) {
            throw new CommonException("id 为空，保存失败");
        }

        if (StringUtils.isBlank(kpiProject.getProjectName())) {
            throw new CommonException("评估项名称为空，保存失败", "base_canot_be_null", "评估项名称");
        }
        if (StringUtils.isBlank(kpiProject.getProjectTypeName())) {
            throw new CommonException("评估项类别名称为空，保存失败", "base_canot_be_null", "评估项类别名称");
        }

        if ((kpiProject.getProjectName() != null && getWordCountCode(kpiProject.getProjectName(), "UTF-8") > 300)
                || (kpiProject.getProjectStandard() != null && getWordCountCode(
                        kpiProject.getProjectStandard(), "UTF-8") > 2000)
                || (kpiProject.getProjectInstruction() != null && getWordCountCode(
                        kpiProject.getProjectInstruction(), "UTF-8") > 2000)
                || (kpiProject.getProjectMethodParameter() != null && getWordCountCode(
                        kpiProject.getProjectMethodParameter(), "UTF-8") > 1000)
                || (kpiProject.getProjectMethodParameterName() != null && getWordCountCode(
                        kpiProject.getProjectMethodParameterName(), "UTF-8") > 300)) {
            throw new CommonException("输入的参数长度超标", "database_length_error");
        }

        // 获取当前登录人公司信息
        Company company = CompanyUtils.currentCompany();
        // 获取当前登录人用户信息
        User user = SpringSecurityUtils.getLoginUser();

        kpiProject.setCompany(company);
        kpiProject.setUser(user);

        this.kpiProjectService.modifyObj(kpiProject);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "删除评估项表")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required = true) String id) {
        if (StringUtils.isBlank(id)) {
            throw new CommonException("删除失败,id不能为空");
        }
        this.kpiProjectService.deleteObjById(id);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "判断评估项是否被模板引用")
    @RequestMapping("service/selectByIdExists")
    @ResponseBody
    public ResponseResult<String> selectByIdExists(@RequestParam(required = true) String id,
            @RequestParam(required = true) String projectCode) {
        if (StringUtils.isBlank(id)) {
            throw new CommonException("查询失败,id不能为空", "base_canot_be_null", "评估项id");
        }
        if (StringUtils.isBlank(projectCode)) {
            throw new CommonException("查询失败,评估项编码不能为空", "base_canot_be_null", "评估项编码");
        }

        KpiTemplateItemExample kpiTemplateItemExample = new KpiTemplateItemExample();
        KpiTemplateItemExample.Criteria criteria = kpiTemplateItemExample.createCriteria();
        criteria.andKpiProjectIdEqualTo(id);// 评估项id

        List<KpiTemplateItem> kpiTemplateItemList = this.kpiTemplateItemService
                .queryAllObjByExample(kpiTemplateItemExample);

        if (kpiTemplateItemList.size() > 0) {
            throw new CommonException("已被其他引用不能操作", "existing_ref_ass", "操作失败！该评估项【评估项编码:" + projectCode
                    + "】", "其他模板【模板编码:" + kpiTemplateItemList.get(0).getTemplateCode() + "】");
        }

        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "查询评估项表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 KpiProject", paramType = "body", dataType = "QueryParamWapper") })
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<KpiProject>> findByPage(@RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        KpiProjectExample example = new KpiProjectExample();
        example.setPageView(new PageView<KpiProject>(pageNo, pageSize));
        example.setOrderByClause(" CREATE_TIME DESC ");
        KpiProjectExample.Criteria criteria = example.createCriteria();
        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }

        PageView<KpiProject> pageData = this.kpiProjectService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
}