package com.els.base.performance.web.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

import java.io.UnsupportedEncodingException;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.entity.User;
import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.company.entity.Company;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.performance.entity.KpiForm;
import com.els.base.performance.entity.KpiFormExample;
import com.els.base.performance.entity.KpiTemplate;
import com.els.base.performance.entity.KpiTemplateExample;
import com.els.base.performance.entity.KpiTemplateItemExample;
import com.els.base.performance.entity.KpiTemplateVO;
import com.els.base.performance.entity.KpiTemplateWeightExample;
import com.els.base.performance.service.KpiFormService;
import com.els.base.performance.service.KpiTemplateService;

@Api(value = "绩效管理-评估模板表")
@Controller
@RequestMapping("kpiTemplate")
public class KpiTemplateController {
    @Resource
    protected KpiTemplateService kpiTemplateService;

    @Resource
    protected KpiFormService kpiFormService;

    @ApiOperation(httpMethod = "POST", value = "创建评估模板表")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody KpiTemplate kpiTemplate) {
        this.kpiTemplateService.addObj(kpiTemplate);
        return ResponseResult.success();
    }

    /* 按特定的编码格式获取长度 */
    public static int getWordCountCode(String str, String code) throws UnsupportedEncodingException {
        return str.getBytes(code).length;
    }

    @ApiOperation(httpMethod = "POST", value = "判断模板是否评估表引用")
    @RequestMapping("service/selectByIdExists")
    @ResponseBody
    public ResponseResult<String> selectByIdExists(@RequestParam(required = true) String id,
            @RequestParam(required = true) String templateCode) {
        if (StringUtils.isBlank(id)) {
            throw new CommonException("查询失败,id不能为空", "base_canot_be_null", "模板id");
        }
        if (StringUtils.isBlank(templateCode)) {
            throw new CommonException("查询失败,编码不能为空", "base_canot_be_null", "模板编码");
        }

        KpiFormExample example = new KpiFormExample();
        KpiFormExample.Criteria criteria = example.createCriteria();
        criteria.andKpiTemplateIdEqualTo(id);

        List<KpiForm> kpiFormList = this.kpiFormService.queryAllObjByExampleCoustom(example);

        if (kpiFormList.size() > 0) {
            throw new CommonException("已被其他引用不能操作", "existing_ref_ass", "操作失败！该评估模板【模板编码:" + templateCode
                    + "】", "其他评估表【评估表编码:" + kpiFormList.get(0).getFormCode() + "】");
        }

        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "创建评估模板和模板类别权重及明细")
    @RequestMapping("service/createKpiTemplate")
    @ResponseBody
    public ResponseResult<String> createKpiTemplate(@RequestBody KpiTemplateVO kpiTemplateVO)
            throws UnsupportedEncodingException {
        if (StringUtils.isBlank(kpiTemplateVO.getKpiTemplate().getTemplateName())) {
            // throw new CommonException("模板编码为空，保存失败");
            throw new CommonException("模板名称为空，保存失败", "base_canot_be_null", "模板名称");
        }

        if (kpiTemplateVO.getKpiTemplate().getTemplateName() != null
                && getWordCountCode(kpiTemplateVO.getKpiTemplate().getTemplateName(), "UTF-8") > 100) {
            throw new CommonException("输入的参数长度超标", "database_length_error");
        }

        // 获取当前登录人公司信息
        Company company = CompanyUtils.currentCompany();
        // 获取当前登录人用户信息
        User user = SpringSecurityUtils.getLoginUser();

        kpiTemplateVO.setCompany(company);
        kpiTemplateVO.setUser(user);

        this.kpiTemplateService.addKpiTemplate(kpiTemplateVO);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "编辑评估模板表")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody KpiTemplate kpiTemplate) {
        if (StringUtils.isBlank(kpiTemplate.getId())) {
            throw new CommonException("id 为空，保存失败");
        }
        this.kpiTemplateService.modifyObj(kpiTemplate);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "编辑评估模板和模板类别权重及明细")
    @RequestMapping("service/editKpiTemplate")
    @ResponseBody
    public ResponseResult<String> editKpiTemplate(@RequestBody KpiTemplateVO kpiTemplateVO)
            throws UnsupportedEncodingException {

        if (StringUtils.isBlank(kpiTemplateVO.getKpiTemplate().getTemplateCode())) {
            // throw new CommonException("模板编码为空，保存失败");
            throw new CommonException("模板编码为空，保存失败", "base_canot_be_null", "模板编码");
        }
        if (StringUtils.isBlank(kpiTemplateVO.getKpiTemplate().getTemplateName())) {
            // throw new CommonException("模板编码为空，保存失败");
            throw new CommonException("模板名称为空，保存失败", "base_canot_be_null", "模板名称");
        }

        if (kpiTemplateVO.getKpiTemplate().getTemplateName() != null
                && getWordCountCode(kpiTemplateVO.getKpiTemplate().getTemplateName(), "UTF-8") > 100) {
            throw new CommonException("输入的参数长度超标", "database_length_error");
        }

        KpiTemplateWeightExample kpiTemplateWeightExample = new KpiTemplateWeightExample();
        KpiTemplateWeightExample.Criteria kpiTemplateWeightCriteria = kpiTemplateWeightExample
                .createCriteria();
        kpiTemplateWeightCriteria.andKpiTemplateIdEqualTo(kpiTemplateVO.getKpiTemplate().getId());
        kpiTemplateWeightCriteria.andTemplateCodeEqualTo(kpiTemplateVO.getKpiTemplate().getTemplateCode());

        KpiTemplateItemExample kpiTemplateItemExample = new KpiTemplateItemExample();
        KpiTemplateItemExample.Criteria criteria = kpiTemplateItemExample.createCriteria();
        criteria.andKpiTemplateIdEqualTo(kpiTemplateVO.getKpiTemplate().getId());
        criteria.andTemplateCodeEqualTo(kpiTemplateVO.getKpiTemplate().getTemplateCode());

        // 获取当前登录人公司信息
        Company company = CompanyUtils.currentCompany();
        // 获取当前登录人用户信息
        User user = SpringSecurityUtils.getLoginUser();

        kpiTemplateVO.setCompany(company);
        kpiTemplateVO.setUser(user);

        this.kpiTemplateService.editKpiTemplate(kpiTemplateVO, kpiTemplateWeightExample,
                kpiTemplateItemExample);

        return ResponseResult.success();

    }

    @ApiOperation(httpMethod = "POST", value = "删除评估模板表")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required = true) String id) {
        if (StringUtils.isBlank(id)) {
            throw new CommonException("删除失败,id不能为空");
        }
        this.kpiTemplateService.deleteObjById(id);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "删除评估模板和模板类别权重及明细")
    @RequestMapping("service/deleteTemplate")
    @ResponseBody
    public ResponseResult<String> deleteTemplate(@RequestBody KpiTemplateVO kpiTemplateVO) {

        if (StringUtils.isBlank(kpiTemplateVO.getKpiTemplate().getTemplateCode())) {
            throw new CommonException("模板编码为空，删除失败");
        }
        if (StringUtils.isBlank(kpiTemplateVO.getKpiTemplate().getId())) {
            throw new NullPointerException("模板id为空，删除失败");
        }

        KpiTemplateWeightExample kpiTemplateWeightExample = new KpiTemplateWeightExample();
        KpiTemplateWeightExample.Criteria kpiTemplateWeightCriteria = kpiTemplateWeightExample
                .createCriteria();
        kpiTemplateWeightCriteria.andKpiTemplateIdEqualTo(kpiTemplateVO.getKpiTemplate().getId());
        kpiTemplateWeightCriteria.andTemplateCodeEqualTo(kpiTemplateVO.getKpiTemplate().getTemplateCode());

        KpiTemplateItemExample kpiTemplateItemExample = new KpiTemplateItemExample();
        KpiTemplateItemExample.Criteria criteria = kpiTemplateItemExample.createCriteria();
        criteria.andKpiTemplateIdEqualTo(kpiTemplateVO.getKpiTemplate().getId());
        criteria.andTemplateCodeEqualTo(kpiTemplateVO.getKpiTemplate().getTemplateCode());

        this.kpiTemplateService.deleteKpiTemplate(kpiTemplateVO, kpiTemplateWeightExample,
                kpiTemplateItemExample);

        return ResponseResult.success();

    }

    @ApiOperation(httpMethod = "POST", value = "查询评估模板表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
            @ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"),
            @ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 KpiTemplate", paramType = "body", dataType = "QueryParamWapper") })
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<KpiTemplate>> findByPage(@RequestParam(defaultValue = "0") int pageNo,
            @RequestParam(defaultValue = "10") int pageSize,
            @RequestBody(required = false) QueryParamWapper wapper) {
        KpiTemplateExample example = new KpiTemplateExample();
        example.setPageView(new PageView<KpiTemplate>(pageNo, pageSize));
        example.setOrderByClause(" CREATE_TIME DESC ");

        KpiTemplateExample.Criteria criteria = example.createCriteria();
        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }

        PageView<KpiTemplate> pageData = this.kpiTemplateService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }
}