package com.els.base.delivery.service.impl;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.company.entity.Company;
import com.els.base.company.entity.CompanyExample;
import com.els.base.company.service.CompanyService;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.delivery.dao.DeliveryOrderBatchNoMapper;
import com.els.base.delivery.entity.DeliveryOrderBatchNo;
import com.els.base.delivery.entity.DeliveryOrderBatchNoExample;
import com.els.base.delivery.service.DeliveryOrderBatchNoService;
import com.els.base.delivery.utils.DeliveryCodeUtil;

@Service("defaultDeliveryOrderBatchNoService")
public class DeliveryOrderBatchNoServiceImpl implements DeliveryOrderBatchNoService {

    Logger logger = LoggerFactory.getLogger(DeliveryOrderBatchNoServiceImpl.class);

    @Resource
    protected DeliveryOrderBatchNoMapper deliveryOrderBatchNoMapper;
    @Resource
    protected CompanyService companyService;

    @CacheEvict(value={"deliveryOrderBatchNo"}, allEntries=true)
    @Transactional
    @Override
    public String getBatchNo(String companySapCode) {
        Assert.isNotBlank(companySapCode, "供应商sap编码不能为空！");
        CompanyExample companyExample = new CompanyExample();
        //companyExample.createCriteria()
        //        .andCompanySapCodeEqualTo(companySapCode)
        //        .andIsEnableEqualTo(Constant.YES_INT);
        companyExample.createCriteria()
                .andCompanySapCodeEqualTo(companySapCode);
        List<Company> companies = this.companyService.queryAllObjByExample(companyExample);
        Assert.isNotEmpty(companies, "公司不能为空！");
        Company company = companies.get(0);
        String companyRetrieve = company.getCompanyRetrieve();
        Assert.isNotBlank(companyRetrieve, "供应商检索项没有在主数据维护，请维护再创建送货单！");


        //每天生成的送货单批次号不一样
        String format = DateFormatUtils.format(new Date(), "yyyy-MM-dd HH:mm:ss");
        logger.info("今天时间为：{}",format);
        String format1 = DateFormatUtils.format(DateUtils.truncate(new Date(), Calendar.DAY_OF_MONTH), "yyyy-MM-dd HH:mm:ss");
        logger.info("截取的时间为：{}",format1);
        String[] parsePatterns=new String[]{"yyyy-MM-dd HH:mm:ss"};
        Date date = new Date();
        try {
             date = DateUtils.parseDate(format1, parsePatterns);
        } catch (ParseException e) {
            throw  new CommonException("格式转换异常");
        }

        DeliveryOrderBatchNoExample orderBatchNoExample = new DeliveryOrderBatchNoExample();
        orderBatchNoExample.setOrderByClause("BATCH_NO DESC,CREATE_TIME DESC");
        orderBatchNoExample.createCriteria()
                .andCreateTimeGreaterThanOrEqualTo(date)
                .andSupCompanySapCodeEqualTo(company.getCompanySapCode());

        String orderBatchNo="";
        List<DeliveryOrderBatchNo> deliveryOrderBatchNos = this.deliveryOrderBatchNoMapper.selectByExample(orderBatchNoExample);
        if (CollectionUtils.isEmpty(deliveryOrderBatchNos)) {
            orderBatchNo=DeliveryCodeUtil.getBatchNo(companyRetrieve, 0);
            //插入记录
            this.insertDeliveryOrderNo(companySapCode, orderBatchNo,0);
            return orderBatchNo;
        }

        DeliveryOrderBatchNo deliveryOrderBatchNo = deliveryOrderBatchNos.get(0);
        Integer batchNo = deliveryOrderBatchNo.getBatchNo();
        if(batchNo>25){
            throw  new CommonException("送货单批次号已达上限，请明天再生成，谢谢！");
        }
         orderBatchNo = DeliveryCodeUtil.getBatchNo(companyRetrieve, batchNo);
        //插入记录
        this.insertDeliveryOrderNo(companySapCode, orderBatchNo,batchNo);
        return orderBatchNo;
    }

    private void insertDeliveryOrderNo(String companySapCode, String orderBatchNo,int batchNo) {
        DeliveryOrderBatchNo orderNo=new DeliveryOrderBatchNo();
        orderNo.setBatchNo(batchNo+1);
        orderNo.setCreateTime(new Date());
        orderNo.setDeliveryOrderBatchNo(orderBatchNo);
        orderNo.setSupCompanySapCode(companySapCode);
        this.deliveryOrderBatchNoMapper.insertSelective(orderNo);
    }

    @CacheEvict(value={"deliveryOrderBatchNo"}, allEntries=true)
    @Override
    public void addObj(DeliveryOrderBatchNo t) {
        this.deliveryOrderBatchNoMapper.insertSelective(t);
    }

    @CacheEvict(value={"deliveryOrderBatchNo"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.deliveryOrderBatchNoMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"deliveryOrderBatchNo"}, allEntries=true)
    @Override
    public void modifyObj(DeliveryOrderBatchNo t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.deliveryOrderBatchNoMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="deliveryOrderBatchNo", keyGenerator="redisKeyGenerator")
    @Override
    public DeliveryOrderBatchNo queryObjById(String id) {
        return this.deliveryOrderBatchNoMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="deliveryOrderBatchNo", keyGenerator="redisKeyGenerator")
    @Override
    public List<DeliveryOrderBatchNo> queryAllObjByExample(DeliveryOrderBatchNoExample example) {
        return this.deliveryOrderBatchNoMapper.selectByExample(example);
    }

    @Cacheable(value="deliveryOrderBatchNo", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<DeliveryOrderBatchNo> queryObjByPage(DeliveryOrderBatchNoExample example) {
        PageView<DeliveryOrderBatchNo> pageView = example.getPageView();
        pageView.setQueryResult(this.deliveryOrderBatchNoMapper.selectByExampleByPage(example));
        return pageView;
    }

    @CacheEvict(value={"deliveryOrderBatchNo"}, allEntries=true)
    @Transactional
	@Override
	public void addAll(List<DeliveryOrderBatchNo> arg0) {
		if (CollectionUtils.isEmpty(arg0)) {
			return;
		}
		for (DeliveryOrderBatchNo record : arg0) {
			this.deliveryOrderBatchNoMapper.insertSelective(record);
		}
	}

    @CacheEvict(value={"deliveryOrderBatchNo"}, allEntries=true)
	@Override
	public void deleteByExample(DeliveryOrderBatchNoExample arg0) {
		Assert.isNotEmpty(arg0.getOredCriteria(), "删除的条件不能为空");
		this.deliveryOrderBatchNoMapper.deleteByExample(arg0);
	}
}