package com.els.base.delivery.service.impl;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.delivery.dao.DeliveryOrderItemMapper;
import com.els.base.delivery.entity.DeliveryOrderItem;
import com.els.base.delivery.entity.DeliveryOrderItemExample;
import com.els.base.delivery.entity.DeliveryOrderReport;
import com.els.base.delivery.entity.DeliveryOrderReportItemExample;
import com.els.base.delivery.service.DeliveryOrderItemService;
import com.els.base.delivery.service.DeliveryOrderService;
import com.els.base.delivery.utils.DeliveryStatusEnum;
import com.els.base.utils.uuid.UUIDGenerator;

@Service("deliveryOrderItemService")
public class DeliveryOrderItemServiceImpl implements DeliveryOrderItemService {

    private static Logger logger = LoggerFactory.getLogger(DeliveryOrderItemServiceImpl.class);

    @Resource
    protected DeliveryOrderItemMapper deliveryOrderItemMapper;
    @Resource
    protected DeliveryOrderService deliveryOrderService;

    @CacheEvict(value = {"deliveryOrderItem"}, allEntries = true)
    @Override
    public void addObj(DeliveryOrderItem t) {
        this.deliveryOrderItemMapper.insertSelective(t);
    }

    @CacheEvict(value = {"deliveryOrderItem"}, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.deliveryOrderItemMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = {"deliveryOrderItem"}, allEntries = true)
    @Override
    public void modifyObj(DeliveryOrderItem t) {
        Assert.isNotBlank(t.getId(), "id 不能为空", "id_is_blank");
        this.deliveryOrderItemMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "deliveryOrderItem", keyGenerator = "redisKeyGenerator")
    @Override
    public DeliveryOrderItem queryObjById(String id) {
        return this.deliveryOrderItemMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "deliveryOrderItem", keyGenerator = "redisKeyGenerator")
    @Override
    public List<DeliveryOrderItem> queryByDeliveryOrderId(String deliveryOrderId) {
        DeliveryOrderItemExample itemExample = new DeliveryOrderItemExample();
        itemExample.createCriteria().andDeliveryOrderIdEqualTo(deliveryOrderId);
        return this.deliveryOrderItemMapper.selectByExample(itemExample);
    }

    @Cacheable(value = "deliveryOrderItem", keyGenerator = "redisKeyGenerator")
    @Override
    public List<DeliveryOrderItem> queryAllObjByExample(DeliveryOrderItemExample example) {
        return this.deliveryOrderItemMapper.selectByExample(example);
    }

    @Cacheable(value = "deliveryOrderItem", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<DeliveryOrderItem> queryObjByPage(DeliveryOrderItemExample example) {
        PageView<DeliveryOrderItem> pageView = example.getPageView();
        pageView.setQueryResult(this.deliveryOrderItemMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Cacheable(value = "deliveryOrderItem", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<DeliveryOrderItem> queryHisObjByPage(DeliveryOrderItemExample example) {
        PageView<DeliveryOrderItem> pageView = example.getPageView();
        pageView.setQueryResult(this.deliveryOrderItemMapper.selectHisByExampleByPage(example));
        return pageView;
    }

    @CacheEvict(value = {"deliveryOrderItem"}, allEntries = true)
    @Override
    public void updateDeliveryOrderItem(DeliveryOrderItem record, DeliveryOrderItemExample example) {
        deliveryOrderItemMapper.updateByExampleSelective(record, example);
    }

    @CacheEvict(value = {"deliveryOrderItem"}, allEntries = true)
    @Override
    public void deleteByExample(DeliveryOrderItemExample example) {
        if (example == null || CollectionUtils.isEmpty(example.getOredCriteria())) {
            return;
        }

        this.deliveryOrderItemMapper.deleteByExample(example);
    }

    @CacheEvict(value = {"deliveryOrderItem"}, allEntries = true)
    @Override
    public void insertHis(String hisItemId, String id, String cause, Date operateTime) {
        this.deliveryOrderItemMapper.insertHis(hisItemId, id, cause, operateTime);
    }


    @CacheEvict(value = {"deliveryOrderItem"}, allEntries = true)
    @Transactional
    @Override
    public void cancellation(List<String> deliveryOrderItemIdList) {

        for (String id : deliveryOrderItemIdList) {

            DeliveryOrderItem deliveryOrderItem = this.queryObjById(id);
            // 数据校验
            if (deliveryOrderItem == null) {
                throw new CommonException("没有找到该发货单行", "do_not_exists", "发货单行");
            }
            if ((deliveryOrderItem.getDeliveryStatus() == null)) {
                throw new CommonException("送货单状态不能为空", "base_canot_be_null", "送货单状态");
            }

            // 如果已经收获的，不允许作废
            if (DeliveryStatusEnum.ALL_RECEIVED.getValue().equals(deliveryOrderItem.getDeliveryStatus())
                    || DeliveryStatusEnum.HK_RECEIVED.getValue().equals(deliveryOrderItem.getDeliveryStatus())) {
                throw new CommonException("该送货单行客户有收货，不可作废发货，谢谢！", "the_customer_has_received_the_goods");
            }
            if (!DeliveryStatusEnum.ONWAY.getValue().equals(deliveryOrderItem.getDeliveryStatus())) {
                throw new CommonException("第"+deliveryOrderItem.getDeliveryOrderItemNo()+"非在途状态无法作废");
            }
            if (Constant.NO_INT.equals(deliveryOrderItem.getIsEnable())) {
                throw new CommonException("第"+deliveryOrderItem.getDeliveryOrderItemNo()+"行已经作废");
            }

            // 减少采购订单里面的可发货数量和在途数量
            this.deliveryOrderService.updatePurchaseOrderForDelivery(deliveryOrderItem.getPurOrderItemId(),
                    deliveryOrderItem.getDeliveryQuantity().multiply(new BigDecimal(-1)));

            DeliveryOrderItemExample deliveryOrderItemExample = new DeliveryOrderItemExample();
            deliveryOrderItemExample.createCriteria().andIdEqualTo(deliveryOrderItem.getId());

            DeliveryOrderItem tempItem = new DeliveryOrderItem();
            tempItem.setIsEnable(Constant.NO_INT);
            this.updateDeliveryOrderItem(tempItem, deliveryOrderItemExample);

            // 设置为0

            // 当前记录添加到历史纪录（发货头记录）
            String hisItemId = UUIDGenerator.generateUUID();
            this.insertHis(hisItemId, deliveryOrderItem.getId(), "作废发货单行", new Date());

        }
    }

    /**
     * 送货单报表
     */
    @Override
    public PageView<DeliveryOrderReport> queryDeliveryOrderReportByPage(DeliveryOrderReportItemExample example) {
        PageView<DeliveryOrderReport> pageView = example.getPageView();
        pageView.setQueryResult(this.deliveryOrderItemMapper.queryDeliveryOrderReportByPage(example));
        return pageView;
    }

    /**
     * 货单报表导出
     */
    @Override
    public List<DeliveryOrderReport> queryDeliveryOrderReportForExcel(DeliveryOrderReportItemExample example) {
        return this.deliveryOrderItemMapper.queryDeliveryOrderReportForExcel(example);
    }

    @CacheEvict(value = {"deliveryOrderItem"}, allEntries = true)
    @Transactional
	@Override
	public void addAll(List<DeliveryOrderItem> arg0) {
		if (CollectionUtils.isEmpty(arg0)) {
			return;
		}
		for (DeliveryOrderItem record : arg0) {
			this.deliveryOrderItemMapper.insertSelective(record);
		}
	}
}