package com.els.base.purchase.service.impl;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.company.entity.Company;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.delivery.entity.DeliveryOrder;
import com.els.base.delivery.entity.DeliveryOrderExample;
import com.els.base.delivery.entity.DeliveryOrderItem;
import com.els.base.delivery.entity.DeliveryOrderItemExample;
import com.els.base.delivery.entity.DeliveryPackage;
import com.els.base.delivery.entity.DeliveryPackageExample;
import com.els.base.delivery.service.DeliveryOrderItemService;
import com.els.base.delivery.service.DeliveryOrderService;
import com.els.base.delivery.service.DeliveryPackageService;
import com.els.base.delivery.utils.DeliveryOrderPrintUtils;
import com.els.base.file.entity.FileData;
import com.els.base.material.entity.SupplierMaterial;
import com.els.base.material.entity.SupplierMaterialExample;
import com.els.base.material.service.SupplierMaterialService;
import com.els.base.purchase.dao.LabelPrintRecordMapper;
import com.els.base.purchase.entity.LabelPrintRecord;
import com.els.base.purchase.entity.LabelPrintRecordExample;
import com.els.base.purchase.entity.LabelPrintRecordExtend;
import com.els.base.purchase.service.LabelPrintRecordService;
import com.els.base.purchase.utils.LabelPrintRecordStatusEnum;
import com.els.base.purchase.utils.SysUtil;
import com.els.base.purchase.vo.KnMaterialQrcodeVO;

@Service("labelPrintRecordService")
public class LabelPrintRecordServiceImpl implements LabelPrintRecordService {

	Logger logger = LoggerFactory.getLogger(this.getClass());

    @Resource
    protected LabelPrintRecordMapper labelPrintRecordMapper;
    @Resource
    protected SupplierMaterialService supplierMaterialService;
    @Resource
    protected DeliveryOrderService deliveryOrderService;
    @Resource
    protected DeliveryOrderItemService deliveryOrderItemService;
    @Resource
    protected DeliveryPackageService deliveryPackageService;

	@Override
	@CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
	public long getOldRidSerialNumber(String projectId, Company company) {
		//默认从1开始
		long serialNumber=0L;

		//查看当天供应商二维码记录中是否产生记录，如果有产生，那么查询最后一条生成流水号
		LabelPrintRecordExample labelPrintRecordExample =new LabelPrintRecordExample();
		labelPrintRecordExample.setOrderByClause("  LAST_UPDATE_TIME DESC , RID DESC ");
		LabelPrintRecordExample.Criteria criteria = labelPrintRecordExample.createCriteria();
		criteria.andProjectIdEqualTo(projectId);
		criteria.andCompanyIdEqualTo(company.getId());
		criteria.andRidIsNotNull();
		//criteria.andPrintStatusEqualTo(LabelPrintRecordStatusEnum.PRINT.getValue());
		Date date= getMorningDate();
		criteria.andLastUpdateTimeGreaterThan(date);
		List<LabelPrintRecord> list = labelPrintRecordMapper.selectByExample(labelPrintRecordExample);
		if(CollectionUtils.isEmpty(list)){
			return serialNumber;
		}
		LabelPrintRecord labelPrintRecord=list.get(0);
		String rid=labelPrintRecord.getRid();
		String num= StringUtils.substring(rid,rid.length()-5);
		if(StringUtils.isNotBlank(num)){
			return Long.parseLong(num);
		}else{
			return 0L;
		}
	}

	/**
	 * 获取凌晨时间
	 * @return
	 */
	private Date getMorningDate(){
		Calendar c = Calendar.getInstance();
		c.set(Calendar.HOUR_OF_DAY, 0);
		c.set(Calendar.MINUTE, 0);
		c.set(Calendar.SECOND, 0);
		c.set(Calendar.MILLISECOND, 0);
		Date date=c.getTime();
		return date;
	}

	@CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
    @Override
    public void addObj(LabelPrintRecord t) {
        this.labelPrintRecordMapper.insertSelective(t);
    }

    @CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.labelPrintRecordMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
    @Override
    public void modifyObj(LabelPrintRecord t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.labelPrintRecordMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "labelPrintRecord", keyGenerator="redisKeyGenerator")
    @Override
    public LabelPrintRecord queryObjById(String id) {
        return this.labelPrintRecordMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "labelPrintRecord", keyGenerator="redisKeyGenerator")
    @Override
    public List<LabelPrintRecord> queryAllObjByExample(LabelPrintRecordExample example) {
        return this.labelPrintRecordMapper.selectByExample(example);
    }

    @Override
    public PageView<LabelPrintRecord> queryObjByPage(LabelPrintRecordExample example) {
        PageView<LabelPrintRecord> pageView = example.getPageView();
        pageView.setQueryResult(this.labelPrintRecordMapper.selectByExampleByPage(example));
        return pageView;
    }


	/**
	 * 组装内箱数据
	 */
	public  List<LabelPrintRecordExtend>  assemblyInnerBox(String printType,List<LabelPrintRecord> items, String projectId, Company company,long serialNumber, int isToExcel) throws Exception {
		//打印数据列表
		List<LabelPrintRecordExtend> printList = new ArrayList<LabelPrintRecordExtend>();
		long serialNumberTmp=serialNumber;
		//数据的组装与校验
		for (LabelPrintRecord labelPrintRecord : items) {

			if(StringUtils.isBlank(labelPrintRecord.getCompanySapCode())){
				throw  new CommonException("公司SAP编码不能为空，请补全之后再打印，谢谢！");
			}

			//rid拼装
			String newRid = "A"
					+ SysUtil.removeZero(labelPrintRecord.getCompanySapCode())
					+ DateFormatUtils.format(new Date(), "yyyyMMdd")
					.substring(2, DateFormatUtils.format(new Date(), "yyyyMMdd").length());

			BigDecimal deliveryQuantity = labelPrintRecord.getDeliveryQuantity();
			logger.info("发货数量为："+deliveryQuantity);

			//送货单数为0的情况
			if(deliveryQuantity==null||deliveryQuantity.compareTo(BigDecimal.ZERO)==0){
				continue;
			}

			String innerPackageQuantity= labelPrintRecord.getInnerPackageQuantity();
			logger.info("最小包装数为："+innerPackageQuantity);

			//内箱数乱填的情况
			if(!StringUtils.isNumeric(innerPackageQuantity)){
				throw new CommonException("内箱的最小包装数数量格式不对");
			}

			//内箱填负数的时候
			if(Integer.parseInt(labelPrintRecord.getInnerPackageQuantity())<0){
				throw new CommonException("内箱的数量或件数不能小于0");
			}

			//内箱数没有维护的时候
			if(labelPrintRecord.getInnerPackageQuantity()==null||Integer.parseInt(labelPrintRecord.getInnerPackageQuantity())==0){
				continue;
			}

			int printPage = deliveryQuantity.divide(new BigDecimal(labelPrintRecord.getInnerPackageQuantity()) ,0, RoundingMode.UP).intValue();
			logger.info("二维码的页数为："+printPage);

			//3、检查发货数量，是否是内包装数的整数倍
			BigDecimal lastPackageQuantity = deliveryQuantity.divideAndRemainder(new BigDecimal(labelPrintRecord.getInnerPackageQuantity()))[1];

				for(int i=0; i<printPage; i++){
					//构建拓展数据
					LabelPrintRecordExtend labelPrintRecordExtend = new LabelPrintRecordExtend();
					BeanUtils.copyProperties(labelPrintRecord, labelPrintRecordExtend);

					if (i == (printPage - 1) && lastPackageQuantity.compareTo(BigDecimal.ZERO) > 0) {
						//如果发货数量不是 内箱包装数的整数倍，在最后一页，二维码数量为 余数
						labelPrintRecordExtend.setInnerPackageQuantity(lastPackageQuantity.toString());
						//labelPrintRecordExtend.setInnerBoxQuantity();
					}

					if (labelPrintRecordExtend.getProductDate() == null) {
						labelPrintRecordExtend.setProductDate(new Date());
					}

					String productBatch = StringUtils.defaultIfBlank(labelPrintRecord.getProductBatch(), "");// 生产批次
					String material = StringUtils.defaultIfBlank(labelPrintRecord.getMaterial(), "");
					String materialVersion = StringUtils.defaultIfBlank(labelPrintRecord.getMaterialVersion(), "");
					String msdRank = StringUtils.defaultIfBlank(labelPrintRecord.getMsdRank(), "");
					String esdRank = StringUtils.defaultIfBlank(labelPrintRecord.getEsdRank(), "");
					String ledRank = StringUtils.defaultIfBlank(labelPrintRecord.getLedRank(), "");
					String pcbRank = StringUtils.defaultIfBlank(labelPrintRecord.getPcbRank(), "");
					String brand = StringUtils.defaultIfBlank(labelPrintRecord.getBrand(), "");
					labelPrintRecordExtend.setProductBatch(productBatch);


					labelPrintRecordExtend.setDeliveryOrderNo(labelPrintRecord.getDeliveryOrderNo());


				    //RID的设置
					serialNumberTmp=serialNumberTmp+1L;
					//serialNumber=serialNumber+1L;

					//如果打印类型为手工打印，那么需要从供应商料号里面寻找改sap编码
					if("1".equals(printType)){

						if(StringUtils.isBlank(labelPrintRecord.getSupplierMaterialId())){
							throw new CommonException("手工打印，没有供应商料号ID，不允许打印，请补充后再打印");
						}
						logger.info("供应商料号ID为："+labelPrintRecord.getSupplierMaterialId());
						SupplierMaterial supplierMaterial =supplierMaterialService.queryObjById(labelPrintRecord.getSupplierMaterialId());

						if(supplierMaterial==null){
							throw new CommonException("手工打印,供应商物料未维护，不允许打印，请补充后再打印");
						}
						logger.info("供应商料号sap编码为："+supplierMaterial.getCompanySapCode());
						newRid = "A"
								+ SysUtil.removeZero(supplierMaterial.getCompanySapCode())
								+ DateFormatUtils.format(new Date(), "yyyyMMdd")
								.substring(2, DateFormatUtils.format(new Date(), "yyyyMMdd").length());

						//设置一下
						labelPrintRecordExtend.setCompanySapCode(supplierMaterial.getCompanySapCode());
					}

					labelPrintRecordExtend.setLn(labelPrintRecordExtend.getCompanySapCode() + "#" + DateFormatUtils.format(labelPrintRecordExtend.getProductDate(), "yyyyMMdd") + "#" + labelPrintRecord.getProductBatch());
					labelPrintRecordExtend.setRid(newRid+StringUtils.leftPad(String.valueOf(serialNumberTmp),5,'0'));

					// 大条码内容
					String bigBarCodeContent = labelPrintRecordExtend.getRid() + "&" +material+ "&" + materialVersion + "&"
							+ labelPrintRecordExtend.getInnerPackageQuantity() + "&" + labelPrintRecordExtend.getCompanySapCode()
							+ "#" + DateFormatUtils.format(labelPrintRecordExtend.getProductDate(), "yyyyMMdd") + "#" + productBatch + "&" + msdRank + "#" + esdRank
							+ "#" + ledRank + "#" + pcbRank + "&" + brand + "&" + labelPrintRecordExtend.getProductArea()
							+ "&" + labelPrintRecordExtend.getPoitem() + "&" + labelPrintRecordExtend.getVersion() + "&"
							+ labelPrintRecordExtend.getSupplierMaterial();
					labelPrintRecordExtend.setBigBarCodeContent(bigBarCodeContent);

					// 下部小条码内容//华阳物料
					String samllDownBarCodeContent = labelPrintRecordExtend.getMaterial();
					labelPrintRecordExtend.setSamllDownBarCodeContent(samllDownBarCodeContent);

					// 上部小条码内容//RID信息
					String samllUpBarCodeContent = labelPrintRecordExtend.getRid();
					labelPrintRecordExtend.setSamllUpBarCodeContent(samllUpBarCodeContent);


					String bigBarCodeBase64=DeliveryOrderPrintUtils.getBase64QRCode(bigBarCodeContent);
					String  samllDownBarCodeBase64=DeliveryOrderPrintUtils.getBase64QRCode(samllDownBarCodeContent);
					String  samllUpBarCodeBase64=DeliveryOrderPrintUtils.getBase64QRCode(samllUpBarCodeContent);

					labelPrintRecordExtend.setSamllDownBarCodeImagePath(samllDownBarCodeBase64);
					labelPrintRecordExtend.setSamllUpBarCodeImagePath(samllUpBarCodeBase64);
					labelPrintRecordExtend.setBigBarCodeImagePath(bigBarCodeBase64);

					printList.add(labelPrintRecordExtend);
				}

				labelPrintRecord.setRid(newRid+StringUtils.leftPad(String.valueOf(serialNumberTmp),5,'0'));
				labelPrintRecord.setLastUpdateTime(new Date());
				if (isToExcel != 1) {
					labelPrintRecord.setPrintTime(new Date());
					labelPrintRecord.setPrintStatus(LabelPrintRecordStatusEnum.PRINT.getValue());
				}
				labelPrintRecordMapper.updateByPrimaryKeySelective(labelPrintRecord);

			}

    	return printList;
	}


	/**
	 * 组装外箱数据
	 */
	public  List<LabelPrintRecordExtend>  assemblyOutterBox(String printType,List<LabelPrintRecord> items, String projectId, Company company,long serialNumber, int isToExcel) throws Exception {
		//打印数据
		List<LabelPrintRecordExtend> printList = new ArrayList<LabelPrintRecordExtend>();

		long serialNumberTemp=serialNumber;

		for (LabelPrintRecord labelPrintRecord : items) {

			if(StringUtils.isBlank(labelPrintRecord.getCompanySapCode())){
				throw  new CommonException("公司SAP编码不能为空，请补全之后再打印，谢谢！");
			}

			String newRid = "A"
					+ SysUtil.removeZero(labelPrintRecord.getCompanySapCode())
					+ DateFormatUtils.format(new Date(), "yyyyMMdd")
					.substring(2, DateFormatUtils.format(new Date(), "yyyyMMdd").length());

			//送货数量
			BigDecimal  deliveryQuantity =labelPrintRecord.getDeliveryQuantity();
			if(deliveryQuantity==null||deliveryQuantity.compareTo(BigDecimal.ZERO)==0){
				continue;
			}

			//格式的校验
			/*if (!StringUtils.isNumeric(labelPrintRecord.getOutterBoxQuantity().toString())||labelPrintRecord.getOutterBoxQuantity() <0) {
				throw new CommonException("外箱的数量格式不对");
			}*/

			//外包装数
			Long  outterPackageQuantity = labelPrintRecord.getOutterPackageQuantity();
			if(outterPackageQuantity==null||outterPackageQuantity==0){
				continue;
			}

			long printPage = deliveryQuantity.divide(new BigDecimal(outterPackageQuantity) ,0, RoundingMode.UP).longValue();
			labelPrintRecord.setOutterBoxQuantity(printPage);

			BigDecimal lastPackageQuantity = deliveryQuantity.divideAndRemainder(new BigDecimal(outterPackageQuantity))[1];

				for(int i=0; i<printPage; i++){
					//构建打印数据对象
					LabelPrintRecordExtend labelPrintRecordExtend = new LabelPrintRecordExtend();
					BeanUtils.copyProperties(labelPrintRecord, labelPrintRecordExtend);

					if (i == (printPage - 1) && lastPackageQuantity.compareTo(BigDecimal.ZERO) > 0) {
						//如果发货数量不是 内箱包装数的整数倍，在最后一页，二维码数量为 余数
						labelPrintRecordExtend.setOutterPackageQuantity(lastPackageQuantity.longValue());
					}

					if (labelPrintRecordExtend.getProductDate() == null) {
						labelPrintRecordExtend.setProductDate(new Date());
					}

					String productBatch = StringUtils.defaultIfBlank(labelPrintRecord.getProductBatch(), "");// 生产批次
					String material = StringUtils.defaultIfBlank(labelPrintRecord.getMaterial(), "");
					String materialVersion = StringUtils.defaultIfBlank(labelPrintRecord.getMaterialVersion(), "");
					String msdRank = StringUtils.defaultIfBlank(labelPrintRecord.getMsdRank(), "");
					String esdRank = StringUtils.defaultIfBlank(labelPrintRecord.getEsdRank(), "");
					String ledRank = StringUtils.defaultIfBlank(labelPrintRecord.getLedRank(), "");
					String pcbRank = StringUtils.defaultIfBlank(labelPrintRecord.getPcbRank(), "");
					String brand = StringUtils.defaultIfBlank(labelPrintRecord.getBrand(), "");
					labelPrintRecordExtend.setProductBatch(productBatch);

					serialNumberTemp=serialNumberTemp+1L;


					//如果打印类型为手工打印，那么需要从供应商料号里面寻找改sap编码
					if("1".equals(printType)){

						if(StringUtils.isBlank(labelPrintRecord.getSupplierMaterialId())){
							throw new CommonException("手工打印，没有供应商料号ID，不允许打印，请补充后再打印");
						}

						logger.info("供应商料号ID为："+labelPrintRecord.getSupplierMaterialId());
						SupplierMaterial supplierMaterial =supplierMaterialService.queryObjById(labelPrintRecord.getSupplierMaterialId());

						if(supplierMaterial==null){
							throw new CommonException("手工打印,供应商物料未维护，不允许打印，请补充后再打印");
						}

						logger.info("供应商sap编码为："+supplierMaterial.getCompanySapCode());
						newRid = "A"
								+ SysUtil.removeZero(supplierMaterial.getCompanySapCode())
								+ DateFormatUtils.format(new Date(), "yyyyMMdd")
								.substring(2, DateFormatUtils.format(new Date(), "yyyyMMdd").length());
						//设置一下
						labelPrintRecordExtend.setCompanySapCode(supplierMaterial.getCompanySapCode());
					}

					labelPrintRecordExtend.setLn(labelPrintRecordExtend.getCompanySapCode() + "#" + DateFormatUtils.format(labelPrintRecordExtend.getProductDate(), "yyyyMMdd") + "#" + labelPrintRecord.getProductBatch());
					labelPrintRecordExtend.setRid(newRid+StringUtils.leftPad(String.valueOf(serialNumberTemp),5,'0'));

					// 大条码内容
					String bigBarCodeContent = labelPrintRecordExtend.getRid() + "&" +material+ "&" + materialVersion + "&"
							+ labelPrintRecordExtend.getOutterPackageQuantity() + "&" + labelPrintRecordExtend.getCompanySapCode()
							+ "#" + DateFormatUtils.format(labelPrintRecordExtend.getProductDate(), "yyyyMMdd")  + "#" + productBatch + "&" + msdRank + "#" + esdRank
							+ "#" + ledRank + "#" + pcbRank + "&" + brand + "&" + labelPrintRecordExtend.getProductArea()
							+ "&" + labelPrintRecordExtend.getPoitem() + "&" + labelPrintRecordExtend.getVersion() + "&"
							+ labelPrintRecordExtend.getSupplierMaterial();
					labelPrintRecordExtend.setBigBarCodeContent(bigBarCodeContent);

					// 下部小条码内容//华阳物料
					String samllDownBarCodeContent = labelPrintRecordExtend.getMaterial();
					labelPrintRecordExtend.setSamllDownBarCodeContent(samllDownBarCodeContent);

					// 上部小条码内容//RID信息
					String samllUpBarCodeContent = labelPrintRecordExtend.getRid();
					labelPrintRecordExtend.setSamllUpBarCodeContent(samllUpBarCodeContent);


					//二维码
					String bigBarCodeBase64=DeliveryOrderPrintUtils.getBase64QRCode(bigBarCodeContent);
					String  samllDownBarCodeBase64=DeliveryOrderPrintUtils.getBase64QRCode(samllDownBarCodeContent);
					String  samllUpBarCodeBase64=DeliveryOrderPrintUtils.getBase64QRCode(samllUpBarCodeContent);
					labelPrintRecordExtend.setSamllDownBarCodeImagePath(samllDownBarCodeBase64);
					labelPrintRecordExtend.setSamllUpBarCodeImagePath(samllUpBarCodeBase64);
					labelPrintRecordExtend.setBigBarCodeImagePath(bigBarCodeBase64);

					printList.add(labelPrintRecordExtend);// 有多少包装件数就打印多少个
				}
				labelPrintRecord.setRid(newRid+StringUtils.leftPad(String.valueOf(serialNumberTemp),5,'0'));
				labelPrintRecord.setLastUpdateTime(new Date());
				if (isToExcel != 1) {
					labelPrintRecord.setPrintTime(new Date());
					labelPrintRecord.setPrintStatus(LabelPrintRecordStatusEnum.PRINT.getValue());
				}
				labelPrintRecordMapper.updateByPrimaryKeySelective(labelPrintRecord);
			}
		return printList;
	}

	/**
	 * 获取供应商料号信息列表,参数供应商ID，客户物料号
	 */
	private List<SupplierMaterial> getSupperMaterialList(LabelPrintRecord labelPrintRecord){
		if (StringUtils.isBlank(labelPrintRecord.getMaterial())&& StringUtils.isBlank(labelPrintRecord.getSupplierMaterial())) {
			throw new CommonException("供应商料号和客户物料号不能全部为空");
		}
		List<SupplierMaterial> supplierMaterialList = new ArrayList<SupplierMaterial>();
		if (StringUtils.isBlank(labelPrintRecord.getSupplierMaterial())) {
			SupplierMaterialExample example2 = new SupplierMaterialExample();
			example2.createCriteria().andCompanyIdEqualTo(labelPrintRecord.getCompanyId())
					.andMaterialCodeEqualTo(labelPrintRecord.getMaterial());
			supplierMaterialList = this.supplierMaterialService.queryAllObjByExample(example2);

			if (CollectionUtils.isEmpty(supplierMaterialList)) {
				throw new CommonException("物料为" + labelPrintRecord.getMaterial() + "没有在供应商物料信息管理中维护",
						"delivery_row_material", labelPrintRecord.getMaterial(), "没有在供应商物料信息管理中维护");
			}

		} else if (StringUtils.isBlank(labelPrintRecord.getMaterial())) {
			SupplierMaterialExample example2 = new SupplierMaterialExample();
			example2.createCriteria().andCompanyIdEqualTo(labelPrintRecord.getCompanyId())
					.andSupplierMaterialEqualTo(labelPrintRecord.getSupplierMaterial());
			supplierMaterialList = this.supplierMaterialService.queryAllObjByExample(example2);
			if (CollectionUtils.isEmpty(supplierMaterialList)) {
				throw new CommonException("物料为" + labelPrintRecord.getSupplierMaterial() + "没有供应商物料信息管理中维护",
						"delivery_row_material", labelPrintRecord.getSupplierMaterial(), "没有在供应商物料信息管理中维护");
			}
		} else {
			SupplierMaterialExample example2 = new SupplierMaterialExample();
			example2.createCriteria().andCompanyIdEqualTo(labelPrintRecord.getCompanyId())
					.andSupplierMaterialEqualTo(labelPrintRecord.getSupplierMaterial())
					.andMaterialCodeEqualTo(labelPrintRecord.getMaterial());
			supplierMaterialList = this.supplierMaterialService.queryAllObjByExample(example2);
			if (CollectionUtils.isEmpty(supplierMaterialList)) {
				throw new CommonException("物料为" + labelPrintRecord.getMaterial() + "没有在供应商物料信息管理维护",
						"delivery_row_material", labelPrintRecord.getMaterial(), "没有在供应商物料信息管理中维护");
			}
		}
    	return  supplierMaterialList;
	}


	//根据送货单数据查询送货单行数据
	public List<DeliveryOrderItem> checkDeliveryOrderItem(DeliveryOrder deliveryOrder){
		DeliveryOrderItemExample deliveryOrderItemExample = new DeliveryOrderItemExample();
		deliveryOrderItemExample.createCriteria().andDeliveryOrderIdEqualTo(deliveryOrder.getId());
		List<DeliveryOrderItem> deliveryOrderItemList = this.deliveryOrderItemService.queryAllObjByExample(deliveryOrderItemExample);
		if (CollectionUtils.isEmpty(deliveryOrderItemList)) {
			throw new CommonException("行数据不存在", "do_not_exists", "行数据");
		}
		return deliveryOrderItemList;
	}

	//根据送货单ID查询送货单
	List<DeliveryOrder> checkDeliveryOrderList(List<String> ids){
		DeliveryOrderExample deliveryOrderExample = new DeliveryOrderExample();
		deliveryOrderExample.createCriteria().andIdIn(ids);
		List<DeliveryOrder> items = this.deliveryOrderService.queryAllObjByExample(deliveryOrderExample);
		if (CollectionUtils.isEmpty(items)) {
			throw new CommonException("没有查询到数据", "not_query_to_data");
		}
		return items;
	}

	//根据送货单行物料号进行查找
	public  List<SupplierMaterial>  checkSupplierMaterialList(DeliveryOrderItem deliveryOrderItem){
		SupplierMaterialExample supplierMaterialExample = new SupplierMaterialExample();
		       supplierMaterialExample.createCriteria()
					   .andCompanyIdEqualTo(deliveryOrderItem.getCompanyId())
					   .andCompanySapCodeEqualTo(deliveryOrderItem.getCompanyCode())
					   .andMaterialCodeEqualTo(deliveryOrderItem.getMaterialNo());
		List<SupplierMaterial> supplierMaterialList = this.supplierMaterialService.queryAllObjByExample(supplierMaterialExample);
		if (CollectionUtils.isEmpty(supplierMaterialList)) {
			throw new CommonException("送货行中物料编码为" + deliveryOrderItem.getMaterialNo() + "没有在供应商物料信息管理中维护", "delivery_row_material", deliveryOrderItem.getMaterialNo(), "没有在供应商物料信息管理中维护");
		}
		return  supplierMaterialList;
	}

	@Override
    public List<LabelPrintRecord> createPrintDataDelivery(List<String> ids) {
		//打印数据列表
        List<LabelPrintRecord> list = new ArrayList<LabelPrintRecord>();

        //根据送货单ID查询送货单
        List<DeliveryOrder> items = checkDeliveryOrderList(ids);

        for (DeliveryOrder deliveryOrder : items) {

        	//添加校验 发货状态:1 未发货，2 在途，3收货,4 香港到货"
			if(deliveryOrder.getDeliveryStatus()!= 2){
				if(deliveryOrder.getDeliveryStatus()==1){
					throw new CommonException("送货单号为："+deliveryOrder.getDeliveryOrderNo()+"所在行的送货单未发货，不可打印！");
				}else if(deliveryOrder.getDeliveryStatus()==3){
					throw new CommonException("送货单号为："+deliveryOrder.getDeliveryOrderNo()+"所在行的送货单已收货，不可打印！");
				}else if(deliveryOrder.getDeliveryStatus()==4){
					throw new CommonException("送货单号为："+deliveryOrder.getDeliveryOrderNo()+"所在行的送货单为香港到货，不可打印！");
				}
			}
			
			//根据送货单数据查询送货单行数据
			List<DeliveryOrderItem> deliveryOrderItemList = checkDeliveryOrderItem(deliveryOrder);

            for (DeliveryOrderItem deliveryOrderItem : deliveryOrderItemList) {
            	//根据送货单行数据查找所关联的包装明细
                DeliveryPackageExample example = new DeliveryPackageExample();
                example.createCriteria()
                    .andDeliveryOrderIdEqualTo(deliveryOrder.getId())
					.andDeliveryOrderNoEqualTo(deliveryOrder.getDeliveryOrderNo())
					.andDeliveryOrderItemIdEqualTo(deliveryOrderItem.getId())
                    .andDeliveryOrderItemNoEqualTo(deliveryOrderItem.getDeliveryOrderItemNo());
                List<DeliveryPackage> deliveryPackageList = this.deliveryPackageService.queryAllObjByExample(example);

				//如果有包装明细就按照包装明细打印
                if (CollectionUtils.isNotEmpty(deliveryPackageList)) {
                	list.addAll(createLabelWithPackage(deliveryOrder, deliveryOrderItem, deliveryPackageList));
                }else{
                	throw new CommonException("没有包装明细信息，无法打印");
                }
            }

			//根据送货单数据查询送货单行数据
//			List<DeliveryOrderItem> deliveryOrderItemList = checkDeliveryOrderItem(deliveryOrder);
//			Map<String,BigDecimal> map = new HashMap<>();
//			BigDecimal count =null;
//			for (DeliveryOrderItem deliveryOrderItem : deliveryOrderItemList) {
//				count=count.add(deliveryOrderItem.getDeliveryQuantity());
//			}
//			map.put(deliveryOrderItemList.get(0).getMaterialNo(), count);
//			list=createLabelPrintRecord(map);
			
        }
        return list;
        
        
//            for (DeliveryOrderItem deliveryOrderItem : deliveryOrderItemList) {
//            	//根据送货单行数据查找所关联的包装明细
//                DeliveryPackageExample example = new DeliveryPackageExample();
//                example.createCriteria()
//                    .andDeliveryOrderIdEqualTo(deliveryOrder.getId())
//					.andDeliveryOrderNoEqualTo(deliveryOrder.getDeliveryOrderNo())
//					.andDeliveryOrderItemIdEqualTo(deliveryOrderItem.getId())
//                    .andDeliveryOrderItemNoEqualTo(deliveryOrderItem.getDeliveryOrderItemNo());
//                List<DeliveryPackage> deliveryPackageList = this.deliveryPackageService.queryAllObjByExample(example);
//
//				//如果有包装明细就按照包装明细打印
//                if (CollectionUtils.isNotEmpty(deliveryPackageList)) {
//                	list.addAll(createLabelWithPackage(deliveryOrder, deliveryOrderItem, deliveryPackageList));
//                }else{
//                	list.addAll(createLabelWithoutPackage(deliveryOrder, deliveryOrderItem));
//                }
//            }
    }
	
	/**
	 * 
	 * @param map
	 * @return
	 */
	private List<LabelPrintRecord> createLabelPrintRecord(Map<String, BigDecimal> map) {
		
		return null;
	}

	/**
	 * 没有包装明细的计算方法
	 * @param deliveryOrder
	 * @param deliveryOrderItem
	 * @return
	 */
	private List<LabelPrintRecord> createLabelWithoutPackage(DeliveryOrder deliveryOrder, DeliveryOrderItem deliveryOrderItem){
		List<LabelPrintRecord> list = new ArrayList<>();
		
    	LabelPrintRecord labelPrintRecord = new LabelPrintRecord();

		//根据供应商料号查找供应商料号信息
		List<SupplierMaterial> supplierMaterialList =checkSupplierMaterialList(deliveryOrderItem);

        //复制料号信息
		BeanUtils.copyProperties(supplierMaterialList.get(0), labelPrintRecord);

		//由“湿敏度MSD等级MsLv#静电防护等级ESDLv#LED亮度等级LedLv#PCB定义”构成
		String spec = labelPrintRecord.getMsdRank() + "#" + labelPrintRecord.getEsdRank()
				+ "#" + labelPrintRecord.getLedRank() + "#" + labelPrintRecord.getPcbRank();
		labelPrintRecord.setSpec(spec);
		labelPrintRecord.setCreateTime(new Date());
		labelPrintRecord.setCreateUser(SpringSecurityUtils.getLoginUserName());
        labelPrintRecord.setDeliveryOrderNo(deliveryOrder.getDeliveryOrderNo());
        labelPrintRecord.setPurchaseOrderNo(deliveryOrderItem.getPurOrderNo());
        labelPrintRecord.setPurchaseOrderItemNo(deliveryOrderItem.getPurOrderItemNo());
        labelPrintRecord.setPoitem(deliveryOrderItem.getPurOrderNo() + "#" + deliveryOrderItem.getPurOrderItemNo());

		//由“供应商代码、生产日期D/C、批次LOT”构成
		/*String rId = SysUtil.getNextRid(deliveryOrder.getCompanyCode());
		labelPrintRecord.setRid(rId);*/

		labelPrintRecord.setPrintStatus(LabelPrintRecordStatusEnum.UN_PRINT.getValue());
		labelPrintRecord.setPrintTime(new Date());
		labelPrintRecord.setProductDate(new Date());

		String  material=StringUtils.defaultIfBlank(deliveryOrderItem.getMaterialNo(),supplierMaterialList.get(0).getMaterialCode());
		labelPrintRecord.setMaterial(material);

		//添加供应商料号ID
		String supplierMaterialId=StringUtils.defaultIfBlank(labelPrintRecord.getSupplierMaterialId(),supplierMaterialList.get(0).getId());
		labelPrintRecord.setSupplierMaterialId(supplierMaterialId);

		//数量的计算

		//送货数量
		BigDecimal deliveryQuantity=deliveryOrderItem.getDeliveryQuantity();
		if(deliveryQuantity==null||deliveryQuantity.compareTo(BigDecimal.ZERO)==0){
			throw  new CommonException("发货数量不能为0或者为空");
		}
		labelPrintRecord.setDeliveryQuantity(deliveryQuantity);

		//最小包装数
		String minPackage="0";
		if(CollectionUtils.isNotEmpty(supplierMaterialList)){
			if(StringUtils.isNotEmpty(supplierMaterialList.get(0).getMinPackage())){
				minPackage=supplierMaterialList.get(0).getMinPackage();
			}
		}
		labelPrintRecord.setInnerPackageQuantity(minPackage);

		//内箱数量
		Long innerBoxQuantity=0L;
		if(StringUtils.isNotBlank(minPackage)&&Long.valueOf(minPackage)!=0){
			BigDecimal temp = deliveryQuantity.divide(new BigDecimal(minPackage),0, BigDecimal.ROUND_UP);
			innerBoxQuantity =	temp.longValue();
		}
		labelPrintRecord.setInnerBoxQuantity(innerBoxQuantity);


		//外箱数
		String  packQuantity=deliveryOrderItem.getPackQuantity();
		if(StringUtils.isBlank(packQuantity)){
			packQuantity="0";
		}
		labelPrintRecord.setOutterBoxQuantity(Long.valueOf(packQuantity));

		//外箱包装数
		Long outterPackageQuantity=labelPrintRecord.getOutterPackageQuantity();
		if(outterPackageQuantity==null){
			outterPackageQuantity=0L;
		}
		labelPrintRecord.setOutterPackageQuantity(outterPackageQuantity);

		labelPrintRecord.setId(null);

		//特殊，遗留问题 TODO
		if(CollectionUtils.isNotEmpty(supplierMaterialList)){
			labelPrintRecord.setPurCompanyId(supplierMaterialList.get(0).getPurCompanyCode());
		}

		//取得是供应商的sapcode，从送货单来
		labelPrintRecord.setCompanySapCode(deliveryOrder.getCompanyCode());

        list.add(labelPrintRecord);
        return list;
	}

	/**
	 * 有包装明细的计算方法
	 * @param deliveryOrder
	 * @param deliveryOrderItem
	 * @param deliveryPackageList
	 * @return
	 */
	private List<LabelPrintRecord> createLabelWithPackage(DeliveryOrder deliveryOrder, DeliveryOrderItem deliveryOrderItem, List<DeliveryPackage> deliveryPackageList){
		List<LabelPrintRecord> list = new ArrayList<>();
		for (DeliveryPackage deliveryPackage : deliveryPackageList) {

            LabelPrintRecord labelPrintRecord = new LabelPrintRecord();

            //根据供应商料号查找供应商料号信息
			List<SupplierMaterial> supplierMaterialList =checkSupplierMaterialList(deliveryOrderItem);

			//复制供应商料号里面的信息
			BeanUtils.copyProperties(supplierMaterialList.get(0), labelPrintRecord);

            //数据的拼装
			String spec = labelPrintRecord.getMsdRank() + "#" + labelPrintRecord.getEsdRank()
					+ "#" + labelPrintRecord.getLedRank() + "#" + labelPrintRecord.getPcbRank();// 由“湿敏度MSD等级MsLv#静电防护等级ESDLv#LED亮度等级LedLv#PCB定义”构成
			labelPrintRecord.setSpec(spec);
			labelPrintRecord.setCreateTime(new Date());
			labelPrintRecord.setCreateUser(SpringSecurityUtils.getLoginUserName());
            labelPrintRecord.setDeliveryOrderNo(deliveryOrderItem.getDeliveryOrderNo());
            labelPrintRecord.setPurchaseOrderNo(deliveryOrderItem.getPurOrderNo());
            labelPrintRecord.setPurchaseOrderItemNo(deliveryOrderItem.getPurOrderItemNo());
            labelPrintRecord.setPoitem(deliveryOrderItem.getPurOrderNo() + "#" + deliveryOrderItem.getPurOrderItemNo());

            //由“供应商代码、生产日期D/C、批次LOT”构成
			/*String rId = SysUtil.getNextRid(deliveryOrder.getCompanyCode());
			labelPrintRecord.setRid(rId);*/

			labelPrintRecord.setPrintStatus(LabelPrintRecordStatusEnum.UN_PRINT.getValue());
			labelPrintRecord.setPrintTime(new Date());
			labelPrintRecord.setProductDate(new Date());

			String  material=StringUtils.defaultIfBlank(deliveryOrderItem.getMaterialNo(),supplierMaterialList.get(0).getMaterialCode());
			labelPrintRecord.setMaterial(material);

			//添加供应商料号ID
			String supplierMaterialId=StringUtils.defaultIfBlank(labelPrintRecord.getSupplierMaterialId(),supplierMaterialList.get(0).getId());
			labelPrintRecord.setSupplierMaterialId(supplierMaterialId);

			//数量计算

			//发货数量校验
			if(deliveryPackage.getTotalQuantity()==null){
				throw new CommonException("发货数量不能为空，请检查，谢谢");
			}
			BigDecimal totalQuantity =new BigDecimal(deliveryPackage.getTotalQuantity());
			if(totalQuantity.compareTo(BigDecimal.ZERO)<=0){
				throw new CommonException("发货数量不能为0,不能为空，并且不能小于0，请检查，谢谢！");
			}
			labelPrintRecord.setDeliveryQuantity(totalQuantity);

			//外箱数
			Integer packageQuantity=deliveryPackage.getPackageQuantity();
			if(packageQuantity==null){
				packageQuantity=0;
			}
			labelPrintRecord.setOutterBoxQuantity(packageQuantity.longValue());

			//外箱包装数
			Integer deliveryQuantity=deliveryPackage.getDeliveryQuantity();
			if(deliveryQuantity==null){
				deliveryQuantity=0;
			}
			labelPrintRecord.setOutterPackageQuantity(deliveryQuantity.longValue());


			//内箱包装数
			String minPackage="0";
			if(CollectionUtils.isNotEmpty(supplierMaterialList)){
				minPackage=supplierMaterialList.get(0).getMinPackage();
				if(StringUtils.isEmpty(minPackage)){
					minPackage="0";
				}
			}
			labelPrintRecord.setInnerPackageQuantity(minPackage);

			//内箱数
			Long innerBoxQuantity=0L;
			if(StringUtils.isNotBlank(minPackage)&&Integer.parseInt(minPackage)!=0){
				BigDecimal temp=totalQuantity.divide(new BigDecimal(minPackage),0, BigDecimal.ROUND_UP);
				innerBoxQuantity =	temp.longValue();
			}
			labelPrintRecord.setInnerBoxQuantity(innerBoxQuantity);


			labelPrintRecord.setId(null);
			//TODO 特殊，遗留问题 ,ID存采购编码了
			if(CollectionUtils.isNotEmpty(supplierMaterialList)){
				labelPrintRecord.setPurCompanyId(supplierMaterialList.get(0).getPurCompanyCode());
			}

			//取得是供应商的sapcode，从送货单来
			labelPrintRecord.setCompanySapCode(deliveryOrder.getCompanyCode());

            list.add(labelPrintRecord);
        }
		
		return list;
	}


	/**
	 字符简称	中文含义	 备注
	 RID	唯一码	必填，自动生成
	 CPN	华阳通用物料编码	必填
	 REV	华阳通用物料版本（1-2位数字或字母，待PLM定）	必填，目前默认NA
	 QTY	数量	必填
	 L/N	由“供应商代码、生产日期D/C、批次LOT”构成	必填
	 DESC	物料描述	可选
	 SPN	供应商物料编码	可选
	 SPEC	由“湿敏度等级MsLv（1-2位）、ESD（1-2位）、LED亮度等级LedLv（7位）、PCB定义（5位）”构成	可选
	 Brand	品牌（货交香港、转厂必填）	可选，货交香港、转厂必填
	 Origin	原产地（货交香港、转厂必填）	可选，货交香港、转厂必填
	 SVER	软件版本（建议软件部定义时不要超过18位，勿使用“#”）	可选
	 PO	采购订单#采购订单行项	可选
	 NW	净重（货交香港、转厂必填）	可选，货交香港、转厂必填
	 GW	毛重（货交香港、转厂必填）	可选，货交香港、转厂必填
	 */
	@Transactional
    @CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
	@Override
	public List<LabelPrintRecord> saveLabelPrintRecord(String projectId, Company company, List<LabelPrintRecord> items) {

		if(CollectionUtils.isEmpty(items)){
    		throw new CommonException("二维码数据不能为空", "base_canot_be_null","二维码数据");
    	}
		//循环数据的校验
		List<LabelPrintRecord> labelPrintRecordList=new ArrayList<LabelPrintRecord>();

		for (LabelPrintRecord labelPrintRecord : items) {

			//校验是否合法
			check(labelPrintRecord);

			//设置初始值
			labelPrintRecord.setProjectId(projectId);
			labelPrintRecord.setSupplierCode(company.getCompanyCode());
			labelPrintRecord.setCompanyId(company.getId());
			labelPrintRecord.setSupplierName(company.getCompanyName());

			//根据查询获取供应商物料数据，如果页面没有传，那么使用供应商物料数据
			List<SupplierMaterial> supplierMaterialList = getSupperMaterialList(labelPrintRecord);
			SupplierMaterial supplierMaterial=supplierMaterialList.get(0);

			//初始化数据
			labelPrintRecord=init(supplierMaterial,labelPrintRecord);

			//送货数量校验
			BigDecimal deliveryQuantity =labelPrintRecord.getDeliveryQuantity();
			if(deliveryQuantity==null||deliveryQuantity.compareTo(BigDecimal.ZERO)<=0) {
				throw new CommonException("发货数量不能为空，也不能为0,或者为负数，请检查，谢谢");
			}
			if(!StringUtils.isNumeric(deliveryQuantity.toString())){
				throw new CommonException("发货数量格式不对，请检查，谢谢");
			}
			labelPrintRecord.setDeliveryQuantity(deliveryQuantity);

			//最小包装数校验
			String innerPackageQuantity=labelPrintRecord.getInnerPackageQuantity();
			if(StringUtils.isEmpty(innerPackageQuantity)){//如果为空，那么从供应商料号那边拿取
				if(StringUtils.isNotEmpty(supplierMaterial.getMinPackage())){
					innerPackageQuantity=supplierMaterial.getMinPackage();
				}else{
					innerPackageQuantity="0";
				}
			}
			labelPrintRecord.setInnerPackageQuantity(innerPackageQuantity);

			//内包箱数校验
			//内包箱数=送货数量/最小包装数。如果不够装，那么需要进1 ,比如 有10件货，最小包装为3 ，那么需要内包箱数为4
			Long innerBoxQuantity =labelPrintRecord.getInnerBoxQuantity();
			if(StringUtils.isNotEmpty(innerPackageQuantity)&&Integer.parseInt(innerPackageQuantity)!=0){
				innerBoxQuantity = deliveryQuantity.divide(new BigDecimal(innerPackageQuantity) ,0, RoundingMode.UP).longValue();
			}
			labelPrintRecord.setInnerBoxQuantity(innerBoxQuantity);

			//外包箱数校验
			Long outterPackageQuantity=labelPrintRecord.getOutterPackageQuantity();
			if(outterPackageQuantity==null){
				outterPackageQuantity=0L;
			}
			labelPrintRecord.setOutterPackageQuantity(outterPackageQuantity);

			//外箱数
			Long outterBoxQuantity=labelPrintRecord.getOutterBoxQuantity();
			if(outterPackageQuantity!=null&&outterPackageQuantity!=0){
				outterBoxQuantity = deliveryQuantity.divide(new BigDecimal(outterPackageQuantity) ,0, RoundingMode.UP).longValue();
			}
			labelPrintRecord.setOutterBoxQuantity(outterBoxQuantity);

			//内外箱都为0的时候不给打印
			if(outterPackageQuantity==0&&Long.valueOf(labelPrintRecord.getInnerPackageQuantity())==0){
				throw  new CommonException("未维护内、外包装数，无法打印二维码，谢谢！");
			}

			//判断是导出保存，还是直接保存
			if(labelPrintRecord.getPrintStatus()==null||labelPrintRecord.getPrintStatus().equals(LabelPrintRecordStatusEnum.UN_PRINT.getValue())){
				labelPrintRecord.setPrintStatus(LabelPrintRecordStatusEnum.UN_PRINT.getValue());
			}else{
				labelPrintRecord.setPrintStatus(LabelPrintRecordStatusEnum.PRINT.getValue());
			}

			//保存数据库
			if(StringUtils.isNotBlank(labelPrintRecord.getId())){
				this.labelPrintRecordMapper.updateByPrimaryKeySelective(labelPrintRecord);
			}else{
				this.labelPrintRecordMapper.insert(labelPrintRecord);
			}

			//把数据返回去
			labelPrintRecordList.add(labelPrintRecord);
		}
		return labelPrintRecordList;
	}


	private  void check(LabelPrintRecord labelPrintRecord){
		//添加供应商sapcode校验
		if (StringUtils.isBlank(labelPrintRecord.getCompanySapCode())) {
			throw new CommonException("客户物料号："+labelPrintRecord.getMaterial()+"所在行的"+"供应商SAP编码不能为空");
		}

		//供应商料号数据的校验
		if (StringUtils.isBlank(labelPrintRecord.getMaterial())&& StringUtils.isBlank(labelPrintRecord.getSupplierMaterial())) {
			throw new CommonException("供应商料号和客户物料号不能全部为空");
		}
	}


	/**
	 * 根究供应商料号初始化信息
	 * @param supplierMaterial
	 * @param labelPrintRecord
	 * @return
	 */
	private LabelPrintRecord init(SupplierMaterial supplierMaterial,LabelPrintRecord labelPrintRecord){

		//添加供应商料号ID
		String supplierMaterialId=StringUtils.defaultIfBlank(labelPrintRecord.getSupplierMaterialId(),supplierMaterial.getId());
		labelPrintRecord.setSupplierMaterialId(supplierMaterialId);

		//============需要组装的字段=======================
		//"ln": "1001749#20170801#pci",
		String productBatch = labelPrintRecord.getProductBatch() == null ? "" : labelPrintRecord.getProductBatch();// 生产批次
		Date  productDate = labelPrintRecord.getProductDate()==null?new Date():labelPrintRecord.getProductDate();// 生产日期
		String  productDateStr="";
		if (labelPrintRecord.getProductDate() != null) {
			productDateStr = DateFormatUtils.format(labelPrintRecord.getProductDate(), "yyyyMMdd");
		}
		labelPrintRecord.setLn(labelPrintRecord.getSupplierCode() + "#" + productDateStr + "#" + productBatch);

		//labelPrintRecord.setProductDate(new Date());


		//============根据供应商物料获取信息的字段 END=======================

		//"material": "123805001406",
		String material=StringUtils.defaultIfBlank(labelPrintRecord.getMaterial(),supplierMaterial.getMaterialCode());
		labelPrintRecord.setMaterial(material);

		String materialId=StringUtils.defaultIfBlank(labelPrintRecord.getMaterialId(),supplierMaterial.getMaterialId());
		labelPrintRecord.setMaterialId(materialId);

		//"materialDesc": "SOC,ISO,36P,3mm,DIP/V,FUSE(10A),AOWA",
		String materialDesc=StringUtils.defaultIfBlank(labelPrintRecord.getMaterialDesc(),supplierMaterial.getMaterialDesc());
		labelPrintRecord.setMaterialDesc(materialDesc);

		//"materialVersion": null,
		String materialVersion=StringUtils.defaultIfBlank(labelPrintRecord.getMaterialVersion(),supplierMaterial.getMaterialVersion());
		labelPrintRecord.setMaterialVersion(materialVersion);

		//"version": "Rj",
		String version=StringUtils.defaultIfBlank(labelPrintRecord.getVersion(),supplierMaterial.getVersion());
		labelPrintRecord.setVersion(version);

		//"printSize": "80*80",
		String printSize=StringUtils.defaultIfBlank(labelPrintRecord.getPrintSize(),supplierMaterial.getPrintSize());
		labelPrintRecord.setPrintSize(printSize);

		//productArea
		String productArea=StringUtils.defaultIfBlank(labelPrintRecord.getProductArea(),supplierMaterial.getProductArea());
		labelPrintRecord.setProductArea(productArea);


		// spec由“湿敏度MSD等级MsLv#静电防护等级ESDLv#LED亮度等级LedLv#PCB定义”构成
		String msdRank=StringUtils.defaultIfBlank(labelPrintRecord.getMsdRank(),supplierMaterial.getMsdRank());
		String esdRank=StringUtils.defaultIfBlank(labelPrintRecord.getEsdRank(),supplierMaterial.getEsdRank());
		String ledRank=StringUtils.defaultIfBlank(labelPrintRecord.getLedRank(),supplierMaterial.getLedRank());
		String pcbRank=StringUtils.defaultIfBlank(labelPrintRecord.getPcbRank(),supplierMaterial.getPcbRank());
		labelPrintRecord.setMsdRank(msdRank);
		labelPrintRecord.setEsdRank(esdRank);
		labelPrintRecord.setLedRank(ledRank);
		labelPrintRecord.setPcbRank(pcbRank);
		String spec = msdRank+ "#" + esdRank + "#" + ledRank+ "#" + pcbRank;
		labelPrintRecord.setSpec(spec);


		//输入字段
		//"purchaseOrderNo": "caigou",
		String  purchaseOrderNo=StringUtils.defaultIfBlank(labelPrintRecord.getPurchaseOrderNo(),"");
		labelPrintRecord.setPurchaseOrderNo(purchaseOrderNo);

		//"purchaseOrderItemNo": "dingdan",
		String purchaseOrderItemNo=StringUtils.defaultIfBlank(labelPrintRecord.getPurchaseOrderItemNo(),"");
		labelPrintRecord.setPurchaseOrderItemNo(purchaseOrderItemNo);

		//"poitem": "caigou#dingdan",
		labelPrintRecord.setPoitem(purchaseOrderNo+"#"+purchaseOrderItemNo);

		//	"brand": null,
		String  brand=StringUtils.defaultIfBlank(labelPrintRecord.getBrand(),supplierMaterial.getBrand());
		labelPrintRecord.setBrand(brand);

		if(labelPrintRecord.getNetWeight()==null){
			labelPrintRecord.setNetWeight(supplierMaterial.getNetWeight());
		}


		if(labelPrintRecord.getGrossWeight()==null){
			labelPrintRecord.setGrossWeight(supplierMaterial.getGrossWeight());
		}

		labelPrintRecord.setPrintStatus(LabelPrintRecordStatusEnum.UN_PRINT.getValue());

		String purCompanyId = StringUtils.defaultIfBlank(labelPrintRecord.getPurCompanyId(),"");// 客户公司编码
		String purCompanyName =StringUtils.defaultIfBlank( labelPrintRecord.getPurCompanyName(),"");// 客户公司名称
		labelPrintRecord.setPurCompanyId(purCompanyId);
		labelPrintRecord.setPurCompanyName(purCompanyName);


		labelPrintRecord.setCreateTime(new Date());

		String supplierMaterialDesc=StringUtils.defaultIfBlank(labelPrintRecord.getSupplierMaterialDesc(),"");
		labelPrintRecord.setSupplierMaterialDesc(supplierMaterialDesc);


		labelPrintRecord.setCreateUser(SpringSecurityUtils.getLoginUserName());
		labelPrintRecord.setLastUpdateTime(new Date());
		labelPrintRecord.setCreateTime(new Date());
		labelPrintRecord.setLastUpdateUser(SpringSecurityUtils.getLoginUserName());
		labelPrintRecord.setPrintTime(new Date());

		return labelPrintRecord;
	}


	@Override
	@CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
	public List<LabelPrintRecord> exportToExcel(List<String> ids) {
		LabelPrintRecordExample example = new LabelPrintRecordExample();
		LabelPrintRecordExample.Criteria criteria = example.createCriteria();
		criteria.andIdIn(ids);
		
		List<LabelPrintRecord> list = labelPrintRecordMapper.selectByExample(example);
		return list;
//		List<LabelPrintRecordExtend> results = new ArrayList<>();
//		
//		if (list != null && !list.isEmpty()) {
//			for(LabelPrintRecord record : list) {
//				String rId = SysUtil.getNextRid(record.getSupplierCode());
//				record.setRid(rId);
//				// 大条码内容
//				String material = record.getMaterial();// 客户物料
//				String materialVersion = record.getMaterialVersion() == null ? "" : record.getMaterialVersion();
//				String productDateStr = "";
//				if (record.getProductDate() != null) {
//					productDateStr = DateFormatUtils.format(record.getProductDate(), "yyyyMMdd");
//				}
//				String productBatch = record.getProductBatch() == null ? "" : record.getProductBatch();// 生产批次
//				record.setLn(record.getSupplierCode() + "#" + productDateStr + "#" + productBatch);
//				String msdRank = record.getMsdRank() == null ? "" : record.getMsdRank();
//				String esdRank = record.getEsdRank() == null ? "" : record.getEsdRank();
//				String ledRank = record.getLedRank() == null ? "" : record.getLedRank();
//				String pcbRank = record.getPcbRank() == null ? "" : record.getPcbRank();
//				String brand = record.getBrand() == null ? "" : record.getBrand();
//				String bigBarCodeContent = record.getRid() + "&" +material+ "&" + materialVersion + "&"
//						+ record.getOutterPackageQuantity() + "&" + record.getSupplierCode()
//						+ "#" + productDateStr + "#" + productBatch + "&" + msdRank + "#" + esdRank
//						+ "#" + ledRank + "#" + pcbRank + "&" + brand + "&" + record.getProductArea()
//						+ "&" + record.getPoitem() + "&" + record.getVersion() + "&"
//						+ record.getSupplierMaterial();
//				LabelPrintRecordExtend recordExtend = new LabelPrintRecordExtend();
//				BeanUtils.copyProperties(record, recordExtend);
//				recordExtend.setBigBarCodeContent(bigBarCodeContent);
//				results.add(recordExtend);
//			}
//		}
//		
//		return results;
	}

	@Override
	@CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
	public List<LabelPrintRecordExtend> innerBoxList(String printType,List<LabelPrintRecord> items, String projectId, Company company) {
		try {
			long serialNumber=getOldRidSerialNumber(projectId,company);
			return this.assemblyInnerBox(printType,items,projectId,company,serialNumber, 1);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	@Override
	@CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
	public List<LabelPrintRecordExtend> outterBoxBoxList(String printType,List<LabelPrintRecord> items, String projectId, Company company) {
		try {
			long serialNumber=getOldRidSerialNumber(projectId,company);
			return this.assemblyOutterBox(printType,items, projectId, company,serialNumber, 1);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}


	@Override
	@Transactional
	@CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
	public synchronized  List<FileData> printQRcode(String printType,List<LabelPrintRecord> items, String projectId, Company company) throws Exception {
		List<FileData> fileDataList = new ArrayList<FileData>();

		logger.info("保存二维码打印数据");
		items=this.saveLabelPrintRecord(projectId, company, items);

		long serialNumber=getOldRidSerialNumber(projectId,company);
		logger.info("获取供应商最后打印的流水号为："+serialNumber);

		logger.info("组装内箱二维码数据");
		List<LabelPrintRecordExtend> innerBoxPrintList = this.assemblyInnerBox(printType,items,projectId,company,serialNumber, 0);

		FileData innerBoxFileData =null;
		if(CollectionUtils.isNotEmpty(innerBoxPrintList)) {
			logger.info("打印内箱");
			innerBoxFileData = DeliveryOrderPrintUtils.generateDeliveryOrderInnerBoxPdf(innerBoxPrintList);
		}

		//流水号加上内箱数
		serialNumber=serialNumber+innerBoxPrintList.size();
		logger.info("打印完内箱的流水号为："+serialNumber);

		logger.info("组装内箱二维码数据");
		List<LabelPrintRecordExtend> outerBoxPrintList = this.assemblyOutterBox(printType,items,projectId,company,serialNumber, 0);

		FileData outerBoxPrintFileData =null;
		if(CollectionUtils.isNotEmpty(outerBoxPrintList)) {
			logger.info("打印外箱");
			outerBoxPrintFileData = DeliveryOrderPrintUtils.generateDeliveryOrderOuterBoxPdf(outerBoxPrintList);
		}

		if(innerBoxFileData!=null){
			fileDataList.add(innerBoxFileData);
		}
		if(outerBoxPrintFileData!=null){
			fileDataList.add(outerBoxPrintFileData);
		}
		logger.info("二维码数据"+fileDataList);
		return fileDataList;
	}

	@Override
	public List<FileData> knPrintMaterialQRcode(String printType, 
												List<LabelPrintRecord> items, 
												String projectId,
												Company company) throws Exception {
		
		//1、校验数字是否正确
		//2、按照输入的数量进行分箱
		// 送货数量 = 188个，每箱10个，分18个整箱，一个散箱。
		//3、生成打印的pdf
		
		
		List<FileData> fileDataList = new ArrayList<FileData>();

//		logger.info("保存二维码打印数据");
//		items=this.saveLabelPrintRecord(projectId, company, items);
		
		List<KnMaterialQrcodeVO> materialQrcodeList = this.assemblyKnMaterialQrcodeData(items, projectId, company);
		
		//FileData fileData = DeliveryOrderPrintUtils.generateKnMaterialQrcodePdf(materialQrcodeList, "MI");
		//fileDataList.add(fileData);
		
		return fileDataList;
	}
	
	private List<KnMaterialQrcodeVO> assemblyKnMaterialQrcodeData(List<LabelPrintRecord> items, 
																  String projectId, 
																  Company company) {
		
		List<KnMaterialQrcodeVO> materialQrcodeList = new ArrayList<>();
		
		for(LabelPrintRecord printRecord : items) {
			
			
			KnMaterialQrcodeVO vo = new KnMaterialQrcodeVO();
			
			vo.setBatchNo(printRecord.getProductBatch());//批次号
			vo.setCheckBatch("010001125956");//检验批
			vo.setFigureType("MJ560AW12L-20500");//图号
			vo.setMaterialCode("5001236411");//物料号
			vo.setMaterialName("有固定板MS570DWR-30301");//名称
			vo.setPlantNo("3010");//工厂号
			vo.setProductOrderNo("");//入库任务单号
			vo.setQuanlityCheck("");//质检
			vo.setQuantity("3");//数量
			vo.setSiString("Q");//特殊库存标识，如Q（项目库存）
			vo.setSivString("20D17166c001C0");//特殊库存值，如项目号
			vo.setSpecification("");//规格
			vo.setStore("2202");//仓库
			vo.setProjectId(projectId);
			vo.setPurCompanyId("companyId");
			
			materialQrcodeList.add(vo);
		}
		
		return materialQrcodeList;
	}

	@CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
	@Transactional
	@Override
	public void addAll(List<LabelPrintRecord> arg0) {
		if (CollectionUtils.isEmpty(arg0)) {
			return;
		}
		for (LabelPrintRecord record : arg0) {
			this.labelPrintRecordMapper.insertSelective(record);
		}
	}

	@CacheEvict(value = { "labelPrintRecord" }, allEntries = true)
	@Override
	public void deleteByExample(LabelPrintRecordExample arg0) {
		Assert.isNotEmpty(arg0.getOredCriteria(), "删除的条件不能为空");
		this.labelPrintRecordMapper.deleteByExample(arg0);
	}
	
}