package com.els.base.purchase.web.controller;

import java.io.IOException;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.company.entity.Company;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.delivery.entity.DeliveryOrder;
import com.els.base.delivery.entity.DeliveryOrderExample;
import com.els.base.delivery.entity.DeliveryOrderItem;
import com.els.base.delivery.entity.DeliveryOrderItemExample;
import com.els.base.delivery.entity.DeliveryPackage;
import com.els.base.delivery.entity.DeliveryPackageExample;
import com.els.base.delivery.service.DeliveryOrderItemService;
import com.els.base.delivery.service.DeliveryOrderService;
import com.els.base.delivery.service.DeliveryPackageService;
import com.els.base.delivery.utils.DeliveryOrderPrintUtils;
import com.els.base.file.entity.FileData;
import com.els.base.material.entity.Material;
import com.els.base.material.entity.MaterialExample;
import com.els.base.material.entity.SupplierMaterial;
import com.els.base.material.entity.SupplierMaterialExample;
import com.els.base.material.service.MaterialService;
import com.els.base.material.service.SupplierMaterialService;
import com.els.base.purchase.entity.LabelPrintRecord;
import com.els.base.purchase.entity.LabelPrintRecordExample;
import com.els.base.purchase.entity.LabelPrintRecordExtend;
import com.els.base.purchase.service.LabelPrintRecordService;
import com.els.base.purchase.utils.LabelPrintRecordStatusEnum;
import com.els.base.purchase.vo.KnMaterialQrcodeVO;
import com.els.base.utils.excel.ExcelUtils;
import com.els.base.utils.excel.TitleAndModelKey;
import com.google.zxing.WriterException;
import com.itextpdf.text.DocumentException;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import jxl.write.WritableWorkbook;
import jxl.write.WriteException;
import jxl.write.biff.RowsExceededException;

@Api(value = "标签打印记录")
@Controller
@RequestMapping("labelPrintRecord")
public class LabelPrintRecordController {

    private static Logger logger = LoggerFactory.getLogger(LabelPrintRecordController.class);

    private static final String cuttentLock = "lockThis";

    @Resource
    protected LabelPrintRecordService labelPrintRecordService;

    @Resource
    private DeliveryOrderService deliveryOrderService;

    @Resource
    private DeliveryOrderItemService deliveryOrderItemService;

    @Resource
    private DeliveryPackageService deliveryPackageService;

    @Resource
    private SupplierMaterialService supplierMaterialService;

    @Resource
    private MaterialService materialService;

    @ApiOperation(httpMethod = "POST", value = "删除二维码打印记录")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestParam(required = true) String id) {
        if (StringUtils.isBlank(id)) {
            throw new CommonException("id为空，操作失败", "id_is_blank");
        }
        this.labelPrintRecordService.deleteObjById(id);
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "批量删除二维码打印记录")
    @RequestMapping("service/deleteByIds")
    @ResponseBody
    public ResponseResult<String> deleteByIds(@RequestBody(required = true) List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new CommonException("id列表为空，操作失败", "id_is_blank");
        }
        for (String id : ids) {
            if (StringUtils.isBlank(id)) {
                throw new CommonException("id为空，操作失败", "id_is_blank");
            }
            this.labelPrintRecordService.deleteObjById(id);
        }
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "二维码打印记录，查询条件暂定为客户、客户名称、采购订单、物料编码、供应商物料编码、打印状态（下拉列表）、创建日期")
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<LabelPrintRecord>> findByPage(
            @ApiParam(value = "所在页", defaultValue = "0") @RequestParam(defaultValue = "0") int pageNo,
            @ApiParam(value = "每页数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize,
            @ApiParam(value = "查询条件,属性名请参考 LabelPrintRecord") @RequestBody(required = false) QueryParamWapper wapper) {
        LabelPrintRecordExample example = new LabelPrintRecordExample();
        example.setPageView(new PageView<LabelPrintRecord>(pageNo, pageSize));
        example.setOrderByClause(" PRINT_TIME DESC , CREATE_TIME DESC ");

        LabelPrintRecordExample.Criteria criteria = example.createCriteria();
        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }
        criteria.andCompanyIdEqualTo(CompanyUtils.currentCompanyId());

        PageView<LabelPrintRecord> pageData = this.labelPrintRecordService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "二维码未打印记录,区别于二维码记录是，查询全部，不分页")
    @RequestMapping("service/findAll")
    @ResponseBody
    public ResponseResult<List<LabelPrintRecord>> findAll(@ApiParam(value = "查询条件,属性名请参考 LabelPrintRecord")
                                                          @RequestBody(required = false) QueryParamWapper wapper) {
        LabelPrintRecordExample example = new LabelPrintRecordExample();
        LabelPrintRecordExample.Criteria criteria = example.createCriteria();
        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }
        criteria.andCompanyIdEqualTo(CompanyUtils.currentCompanyId());
        criteria.andPrintStatusEqualTo(LabelPrintRecordStatusEnum.UN_PRINT.getValue());
        List<LabelPrintRecord> pageData = this.labelPrintRecordService.queryAllObjByExample(example);
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "供应商送货-送货单管理-打印内箱二维码,生成数据以供打印。参数送货单头（包含里面的行数据）id数组")
    @RequestMapping("service/createPrintDataForDelivery")
    @ResponseBody
    public ResponseResult<List<FileData>> createPrintDataDelivery(@RequestBody List<String> ids) throws WriterException, IOException, DocumentException, InterruptedException {
        if (CollectionUtils.isEmpty(ids)) {
            throw new CommonException("数据不能为空", "base_canot_be_null", "数据");
        }
//		list = this.labelPrintRecordService.createPrintDataDelivery(ids);
        //List<LabelPrintRecord> labelList = this.createPrintDelivery(ids);
        //List<KnMaterialQrcodeVO> voList = this.createKnMaterialQrcodeVO(ids);
        List<FileData> list = this.createPrintDelivery(ids);
        return ResponseResult.success(list);
    }


    private List<FileData> createPrintDelivery(List<String> ids) throws WriterException, IOException, DocumentException, InterruptedException {
        Assert.isNotEmpty(ids, "送货单ID不能为空");

        List<FileData> fileDataList = new ArrayList<>();
        List<KnMaterialQrcodeVO> voList = new ArrayList<>();
        //根据送货单ID查询送货单
        List<DeliveryOrder> items = checkDeliveryOrderList(ids);
        for (DeliveryOrder deliveryOrder : items) {
            //添加校验 发货状态:1 未发货，2 在途，3收货,4 香港到货"
            logger.info("添加校验 发货状态:1 未发货，2 在途，3收货,4 香港到货");
            if (deliveryOrder.getDeliveryStatus() != 2) {
                if (deliveryOrder.getDeliveryStatus() == 1) {
                    throw new CommonException("送货单号为：" + deliveryOrder.getDeliveryOrderNo() + "所在行的送货单未发货，不可打印！");
                } else if (deliveryOrder.getDeliveryStatus() == 3) {
                    throw new CommonException("送货单号为：" + deliveryOrder.getDeliveryOrderNo() + "所在行的送货单已收货，不可打印！");
                } else if (deliveryOrder.getDeliveryStatus() == 4) {
                    throw new CommonException("送货单号为：" + deliveryOrder.getDeliveryOrderNo() + "所在行的送货单为香港到货，不可打印！");
                }
            }
            //根据送货单数据查询送货单行数据
            logger.info("根据送货单数据查询送货单行数据");
            List<DeliveryOrderItem> deliveryOrderItemList = checkDeliveryOrderItem(deliveryOrder);
            logger.info("根据送货单行数据查找所关联的包装明细");
            for (DeliveryOrderItem deliveryOrderItem : deliveryOrderItemList) {
                // 根据送货单行数据查找所关联的包装明细
                DeliveryPackageExample example = new DeliveryPackageExample();
                example.createCriteria().andDeliveryOrderIdEqualTo(deliveryOrder.getId())
                        .andDeliveryOrderNoEqualTo(deliveryOrder.getDeliveryOrderNo())
                        .andDeliveryOrderItemIdEqualTo(deliveryOrderItem.getId())
                        .andDeliveryOrderItemNoEqualTo(deliveryOrderItem.getDeliveryOrderItemNo());
                List<DeliveryPackage> deliveryPackageList = this.deliveryPackageService.queryAllObjByExample(example);
                //根据物料编码在物料主数据中查出仓库、仓位
                logger.info("根据物料编码在物料主数据中查出仓库、仓位");
                MaterialExample materialExample = new MaterialExample();
                materialExample.createCriteria().andMaterialCodeEqualTo(deliveryOrderItem.getMaterialNo())
                        .andFactoryNameEqualTo(deliveryOrderItem.getFactoryCode());
                List<Material> material = this.materialService.queryAllObjByExample(materialExample);
                if (CollectionUtils.isNotEmpty(material)) {
                    deliveryOrderItem.setWarehouseCode(material.get(0).getWarehouse());
                    deliveryOrderItem.setPosition(material.get(0).getWarehouseBin());
                }
                logger.info("如果有包装明细就按照包装明细打印");
                // 如果有包装明细就按照包装明细打印
                if (CollectionUtils.isEmpty(deliveryPackageList)) {
                    throw new CommonException("没有包装明细信息，无法打印");
//					labelList.addAll(createLabelWithPackage(deliveryOrder, deliveryOrderItem, deliveryPackageList));
                }
                logger.info("处理是否自动检测");
                for (DeliveryPackage deliveryPackage : deliveryPackageList) {
                    logger.info("knArray");
                    List<KnMaterialQrcodeVO> knArray = this.generateVO(deliveryOrderItem, deliveryPackage);
                    for (KnMaterialQrcodeVO knMaterialQrcodeVO : knArray) {
                        //
                        logger.info("knArray");
                        if (CollectionUtils.isNotEmpty(material)) {
                            if ("Y".equals(material.get(0).getAutomaticDetection())) {
                                knMaterialQrcodeVO.setAutomaticDetection("是");
                            } else {
                                knMaterialQrcodeVO.setAutomaticDetection("否");
                            }
                            //
                            String standPackQuantity = material.get(0).getStandPackQuantity();
                            if (StringUtils.isEmpty(standPackQuantity)) {
                                standPackQuantity = "";
                            }
                            String standPackSpce = material.get(0).getStandPackSpce();
                            if (StringUtils.isEmpty(standPackSpce)) {
                                standPackSpce = "";
                            }
                            String minimumPackingQuantity = material.get(0).getMinimumPackingQuantity();
                            if (StringUtils.isEmpty(minimumPackingQuantity)) {
                                minimumPackingQuantity = "";
                            }
                            String minPackSpce = material.get(0).getMinPackSpce();
                            if (StringUtils.isEmpty(minPackSpce)) {
                                minPackSpce = "";
                            }
                            String checkBatch = standPackQuantity + "/" + standPackSpce + " " + minimumPackingQuantity + "/" + minPackSpce;
                            knMaterialQrcodeVO.setCheckBatch(checkBatch);
                        }

                    }
                    voList.addAll(knArray);
                }

            }
        }
        logger.info("生成PDF");
        FileData fileData = DeliveryOrderPrintUtils.generateKnMaterialQrcodePdf(voList);
        fileDataList.add(fileData);
//        for(KnMaterialQrcodeVO vo : voList){
//        	for(LabelPrintRecord label : labelList){
//        		for(int i=0;i<label.getOutterPackageQuantity().intValue();i++){
//        			FileData fileData = DeliveryOrderPrintUtils.generateKnMaterialQrcodePdf(vo, "MI");
//        			fileDataList.add(fileData);
//        		}
//        	}
//        }
        return fileDataList;
    }

    private List<KnMaterialQrcodeVO> generateVO(DeliveryOrderItem deliveryOrderItem, DeliveryPackage deliveryPackage) throws IOException {
        int packageNum = deliveryPackage.getPackageQuantity();
        List<KnMaterialQrcodeVO> voList = new ArrayList<>(packageNum);

        for (int i = 0; i < packageNum; i++) {
            KnMaterialQrcodeVO vo = new KnMaterialQrcodeVO();
            vo.setMaterialName(StringUtils.defaultIfBlank(deliveryOrderItem.getMaterialDesc(), ""));
            vo.setStore(StringUtils.defaultIfBlank(deliveryOrderItem.getWarehouseCode(), ""));//仓库
            vo.setBatchNo(StringUtils.defaultIfBlank(deliveryOrderItem.getBatch(), ""));//批次号
            vo.setSpecification(StringUtils.defaultIfBlank(deliveryOrderItem.getSpecification(), ""));//规格
            vo.setFigureType(StringUtils.defaultIfBlank(deliveryOrderItem.getMapNo(), ""));//图号
            vo.setQuantity(String.valueOf(deliveryPackage.getDeliveryQuantity()));//数量
            vo.setCheckBatch("");//单元包装量
            vo.setMaterialCode(StringUtils.defaultIfBlank(deliveryOrderItem.getMaterialNo(), ""));//物料号
            vo.setPlantNo(StringUtils.defaultIfBlank(deliveryOrderItem.getFactoryCode(), ""));//工厂号
            vo.setQuanlityCheck(deliveryOrderItem.getQuanlityTestStatus());//质检
            vo.setLocation(StringUtils.defaultIfBlank(deliveryOrderItem.getWarehouseCode(), ""));//仓库名称
            vo.setPosition(StringUtils.defaultIfBlank(deliveryOrderItem.getPosition(), ""));//仓位
            vo.setDeliverOrderNo(StringUtils.defaultIfBlank(deliveryOrderItem.getDeliveryOrderNo(), ""));//送货单号
            vo.setDeliverOrderItemNo(StringUtils.defaultIfBlank(deliveryOrderItem.getDeliveryOrderItemNo(), ""));//送货单行号
            vo.setSiString(StringUtils.defaultIfBlank(deliveryOrderItem.getCourseAssignType(), ""));//特殊库存标识，如Q（项目库存）
            vo.setProjectId(deliveryOrderItem.getProjectId());
            vo.setPurCompanyId(deliveryOrderItem.getPurCompanyId());
            vo.setSivString(StringUtils.defaultIfBlank(deliveryOrderItem.getWbsNo(), ""));//特殊库存值，如项目号

            /*
             * [S]PLANT_NO[M]{0}
             * [S]MATERIAL_NO[M]{1}
             * [S]QTY[M]{2}
             * [S]BATCH_NO[M]{3}
             * [S]LOCATION[M]{4}
             * [S]ASN_NO[M]{5}
             * [S]ASN_ITEM_NO[M]{6}
             * [S]SI[M]{7}
             * [S]SIV[M]{8}
             * [E]
             * PLANT_NO: 工厂号
             * MATERIAL_NO: 物料号
             * QTY: 数量
             * BATCH_NO: 批次号
             * LOCATION: 仓库
             * ASN_NO: ASN订单号（送货单号）
             * ASN_ITEM_NO: ASN订单行号（送货单行号）
             * SI: 特殊库存标识（如Q 项目库存）
             * SIV: 特殊库存值（如项目号）
             */
            String s = "[S]PLANT_NO[M]%s"
                    + "[S]MATERIAL_NO[M]%s"
                    + "[S]QTY[M]%s"
                    + "[S]BATCH_NO[M]%s"
                    + "[S]LOCATION[M]%s"
                    + "[S]ASN_NO[M]%s"
                    + "[S]ASN_ITEM_NO[M]%s"
                    + "[S]SI[M]%s"
                    + "[S]SIV[M]%s"
                    + "[E]";

            //物料二维码
            String barcodeBase64Str = String.format(s, vo.getPlantNo(),
                    vo.getMaterialCode(),
                    vo.getQuantity(),
                    vo.getBatchNo(),
                    vo.getLocation(),
                    vo.getDeliverOrderNo(),
                    vo.getDeliverOrderItemNo(),
                    vo.getSiString(),
                    vo.getSivString());
            String barcodeBase64 = DeliveryOrderPrintUtils.createPdf417QRCode(barcodeBase64Str);
            vo.setScanPicPath(barcodeBase64);
            voList.add(vo);
        }

        return voList;
    }


    /**
     * 包装明细计算方法
     *
     * @param deliveryOrder
     * @param deliveryOrderItem
     * @param deliveryPackageList
     * @return
     */
    private List<LabelPrintRecord> createLabelWithPackage(DeliveryOrder deliveryOrder, DeliveryOrderItem deliveryOrderItem, List<DeliveryPackage> deliveryPackageList) {
        List<LabelPrintRecord> list = new ArrayList<>();

        for (DeliveryPackage deliveryPackage : deliveryPackageList) {

            LabelPrintRecord labelPrintRecord = new LabelPrintRecord();

            //根据供应商料号查找供应商料号信息
            List<SupplierMaterial> supplierMaterialList = checkSupplierMaterialList(deliveryOrderItem);

            //复制供应商料号里面的信息
            BeanUtils.copyProperties(supplierMaterialList.get(0), labelPrintRecord);

            //数据的拼装
            String spec = labelPrintRecord.getMsdRank() + "#" + labelPrintRecord.getEsdRank()
                    + "#" + labelPrintRecord.getLedRank() + "#" + labelPrintRecord.getPcbRank();// 由“湿敏度MSD等级MsLv#静电防护等级ESDLv#LED亮度等级LedLv#PCB定义”构成
            labelPrintRecord.setSpec(spec);
            labelPrintRecord.setCreateTime(new Date());
            labelPrintRecord.setCreateUser(SpringSecurityUtils.getLoginUserName());
            labelPrintRecord.setDeliveryOrderNo(deliveryOrderItem.getDeliveryOrderNo());
            labelPrintRecord.setPurchaseOrderNo(deliveryOrderItem.getPurOrderNo());
            labelPrintRecord.setPurchaseOrderItemNo(deliveryOrderItem.getPurOrderItemNo());
            labelPrintRecord.setPoitem(deliveryOrderItem.getPurOrderNo() + "#" + deliveryOrderItem.getPurOrderItemNo());

            //由“供应商代码、生产日期D/C、批次LOT”构成
			/*String rId = SysUtil.getNextRid(deliveryOrder.getCompanyCode());
			labelPrintRecord.setRid(rId);*/

            labelPrintRecord.setPrintStatus(LabelPrintRecordStatusEnum.UN_PRINT.getValue());
            labelPrintRecord.setPrintTime(new Date());
            labelPrintRecord.setProductDate(new Date());

            String material = StringUtils.defaultIfBlank(deliveryOrderItem.getMaterialNo(), supplierMaterialList.get(0).getMaterialCode());
            labelPrintRecord.setMaterial(material);

            //数量计算

            //发货数量校验
            if (deliveryPackage.getTotalQuantity() == null) {
                throw new CommonException("发货数量不能为空，请检查，谢谢");
            }
            BigDecimal totalQuantity = new BigDecimal(deliveryPackage.getTotalQuantity());
            if (totalQuantity.compareTo(BigDecimal.ZERO) <= 0) {
                throw new CommonException("发货数量不能为0,不能为空，并且不能小于0，请检查，谢谢！");
            }
            labelPrintRecord.setDeliveryQuantity(totalQuantity);

            //外箱数
            Integer packageQuantity = deliveryPackage.getPackageQuantity();
            if (packageQuantity == null) {
                packageQuantity = 0;
            }
            labelPrintRecord.setOutterBoxQuantity(packageQuantity.longValue());

            //外箱包装数
            Integer deliveryQuantity = deliveryPackage.getDeliveryQuantity();
            if (deliveryQuantity == null) {
                deliveryQuantity = 0;
            }
            labelPrintRecord.setOutterPackageQuantity(deliveryQuantity.longValue());

            list.add(labelPrintRecord);
        }
        return list;
    }

    //根据送货单行物料号进行查找
    private List<SupplierMaterial> checkSupplierMaterialList(DeliveryOrderItem deliveryOrderItem) {
        SupplierMaterialExample supplierMaterialExample = new SupplierMaterialExample();
        supplierMaterialExample.createCriteria()
                .andCompanyIdEqualTo(deliveryOrderItem.getCompanyId())
                .andCompanySapCodeEqualTo(deliveryOrderItem.getCompanyCode())
                .andMaterialCodeEqualTo(deliveryOrderItem.getMaterialNo());
        List<SupplierMaterial> supplierMaterialList = this.supplierMaterialService.queryAllObjByExample(supplierMaterialExample);
        if (CollectionUtils.isEmpty(supplierMaterialList)) {
            throw new CommonException("送货行中物料编码为" + deliveryOrderItem.getMaterialNo() + "没有在供应商物料信息管理中维护", "delivery_row_material", deliveryOrderItem.getMaterialNo(), "没有在供应商物料信息管理中维护");
        }
        return supplierMaterialList;
    }


    //根据送货单数据查询送货单行数据
    private List<DeliveryOrderItem> checkDeliveryOrderItem(DeliveryOrder deliveryOrder) {
        DeliveryOrderItemExample deliveryOrderItemExample = new DeliveryOrderItemExample();
        deliveryOrderItemExample.createCriteria().andDeliveryOrderIdEqualTo(deliveryOrder.getId());
        deliveryOrderItemExample.setOrderByClause("DELIVERY_ORDER_ITEM_NO ASC,MATERIAL_NO ASC ,DELIVERY_DATE DESC");
        List<DeliveryOrderItem> deliveryOrderItemList = this.deliveryOrderItemService.queryAllObjByExample(deliveryOrderItemExample);
        if (CollectionUtils.isEmpty(deliveryOrderItemList)) {
            throw new CommonException("行数据不存在", "do_not_exists", "行数据");
        }
        return deliveryOrderItemList;
    }


    //根据送货单ID查询送货单
    private List<DeliveryOrder> checkDeliveryOrderList(List<String> ids) {
        DeliveryOrderExample deliveryOrderExample = new DeliveryOrderExample();
        deliveryOrderExample.createCriteria().andIdIn(ids);
        List<DeliveryOrder> items = this.deliveryOrderService.queryAllObjByExample(deliveryOrderExample);
        if (CollectionUtils.isEmpty(items)) {
            throw new CommonException("没有查询到数据", "not_query_to_data");
        }
        return items;
    }


    @ApiOperation(httpMethod = "POST", value = "通过物料号查询打印记录")
    @RequestMapping("service/findByMaterial")
    @ResponseBody
    public ResponseResult<PageView<LabelPrintRecord>> findByMaterial(
            @ApiParam(value = "所在页", defaultValue = "0") @RequestParam(defaultValue = "0") int pageNo,
            @ApiParam(value = "每页数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize,
            @ApiParam("物料号") String material) {

        if (StringUtils.isBlank(material)) {
            throw new CommonException("查询失败,物料号不存在", "do_not_exists", "物料号");
        }

        LabelPrintRecordExample example = new LabelPrintRecordExample();
        example.setPageView(new PageView<LabelPrintRecord>(pageNo, pageSize));
        example.createCriteria().andMaterialEqualTo(material).andCompanyIdEqualTo(CompanyUtils.currentCompanyId());

        PageView<LabelPrintRecord> pageData = this.labelPrintRecordService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "手工打印二维码，保存二维码数据")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<List<LabelPrintRecord>> create(@RequestBody List<LabelPrintRecord> items) {
        if (CollectionUtils.isEmpty(items)) {
            throw new CommonException("数据不能为空", "base_canot_be_null", "数据");
        }
        List<LabelPrintRecord> list = this.labelPrintRecordService.saveLabelPrintRecord(ProjectUtils.getProjectId(), CompanyUtils.currentCompany(), items);
        return ResponseResult.success(list);
    }


    @ApiOperation(httpMethod = "POST", value = "打印二维码，输出到pdf")
    @RequestMapping("service/printFromUser")
    @ResponseBody
    public ResponseResult<List<FileData>> printFromUser(
            @ApiParam(value = "打印类型,默认0为送货单打印，1为手工打印", defaultValue = "0") @RequestParam(defaultValue = "0") String printType,
            @ApiParam(value = "打印数据") @RequestBody List<LabelPrintRecord> items) throws Exception {
        logger.info("二维码打印开始");

        if (CollectionUtils.isEmpty(items)) {
            throw new CommonException("数据不能为空", "base_canot_be_null", "数据");
        }
        //打印内箱和外箱二维码
        List<FileData> fileDataList = this.labelPrintRecordService.printQRcode(printType, items, ProjectUtils.getProjectId(), CompanyUtils.currentCompany());
        logger.info("二维码结束");
        return ResponseResult.success(fileDataList);
    }

    @ApiOperation(httpMethod = "POST", value = "康尼_打印二维码，输出到pdf")
    @RequestMapping("service/kn/printMaterialQRcode")
    @ResponseBody
    public ResponseResult<List<FileData>> knPrintFromUser(
            @ApiParam(value = "打印类型,默认0为送货单打印，1为手工打印", defaultValue = "0") @RequestParam(defaultValue = "0") String printType,
            @ApiParam(value = "打印数据") @RequestBody List<LabelPrintRecord> items) throws Exception {
        logger.info("二维码打印开始");

        if (CollectionUtils.isEmpty(items)) {
            throw new CommonException("数据不能为空", "base_canot_be_null", "数据");
        }
        //打印内箱和外箱二维码
        List<FileData> fileDataList = this.labelPrintRecordService.knPrintMaterialQRcode(printType, items, ProjectUtils.getProjectId(), CompanyUtils.currentCompany());
        logger.info("二维码结束");
        return ResponseResult.success(fileDataList);
    }

    @ApiOperation(httpMethod = "GET", value = "手工打印二维码，输出到EXCEL")
    @RequestMapping("service/exportToExcel")
    @ResponseBody
    public void exportToExcel(
            @ApiParam(value = "打印类型,默认0为送货单打印，1为手工打印", defaultValue = "0") @RequestParam(defaultValue = "0") String printType,
            @ApiParam(value = "表主键ID列表，多个用英文逗号(,)隔开", defaultValue = "") String ids,
            HttpServletResponse response) {

        if (org.springframework.util.StringUtils.isEmpty(ids)) {
            throw new CommonException("数据不能为空", "base_canot_be_null", "数据");
        }
        String[] idArray = ids.split(",");

        // 根据拿到的数据进行数据组装
        List<String> idList = new ArrayList<>();
        for (String id : idArray) {
            idList.add(id);
        }
        List<LabelPrintRecord> lists = this.labelPrintRecordService.exportToExcel(idList);

        Company company = CompanyUtils.currentCompany();
        String projectId = ProjectUtils.getProjectId();

        List<LabelPrintRecordExtend> innerBoxList = null;
        List<LabelPrintRecordExtend> outtererBoxList = null;
        synchronized (cuttentLock) {
            innerBoxList = this.labelPrintRecordService.innerBoxList(printType, lists, projectId, company);
            outtererBoxList = this.labelPrintRecordService.outterBoxBoxList(printType, lists, projectId, company);
        }

        try {
            response.reset();
            StringBuffer header = new StringBuffer("attachment;");
            header.append("filename=\"" + URLEncoder.encode("手工打印二维码.xls", "UTF-8") + "\";");
            response.setHeader("Content-Disposition", header.toString());
            response.setCharacterEncoding("UTF-8");
            response.setContentType("application/vnd.ms-excel");
            OutputStream outputStream = response.getOutputStream();

            WritableWorkbook writableWorkbook = this.exportInnerBox(outputStream, innerBoxList);
            this.exportOuterBox(outputStream, outtererBoxList, writableWorkbook);

            writableWorkbook.write();
            outputStream.flush();
            writableWorkbook.close();
            outputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
            response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
        }
        return;
    }

    @ApiOperation(httpMethod = "GET", value = "手工打印二维码，输出到EXCEL2")
    @RequestMapping("service/exportToExcel2")
    @ResponseBody
    public void exportToExcel2(
            @ApiParam(value = "打印类型,默认0为送货单打印，1为手工打印", defaultValue = "0") @RequestParam(defaultValue = "0") String printType,
            @ApiParam(value = "表主键ID列表，多个用英文逗号(,)隔开", defaultValue = "") String ids, HttpServletResponse response) {

        if (org.springframework.util.StringUtils.isEmpty(ids)) {
            throw new CommonException("数据不能为空", "base_canot_be_null", "数据");
        }
        String[] idArray = ids.split(",");

        // 根据拿到的数据进行数据组装
        List<String> idList = new ArrayList<>();
        for (String id : idArray) {
            idList.add(id);
        }
        List<LabelPrintRecord> lists = this.labelPrintRecordService.createPrintDataDelivery(idList);

        Company company = CompanyUtils.currentCompany();
        String projectId = ProjectUtils.getProjectId();

        List<LabelPrintRecordExtend> innerBoxList = null;
        List<LabelPrintRecordExtend> outtererBoxList = null;
        synchronized (cuttentLock) {
            innerBoxList = this.labelPrintRecordService.innerBoxList(printType, lists, projectId, company);
            outtererBoxList = this.labelPrintRecordService.outterBoxBoxList(printType, lists, projectId, company);
        }
//		List<LabelPrintRecordExtend> innerBoxList = this.labelPrintRecordService.innerBoxList(lists, projectId, company);
//		List<LabelPrintRecordExtend> outtererBoxList = this.labelPrintRecordService.outterBoxBoxList(lists, projectId, company);

        try {
            response.reset();
            StringBuffer header = new StringBuffer("attachment;");
            header.append("filename=\"" + URLEncoder.encode("手工打印二维码.xls", "UTF-8") + "\";");
            response.setHeader("Content-Disposition", header.toString());
            response.setCharacterEncoding("UTF-8");
            response.setContentType("application/vnd.ms-excel");
            OutputStream outputStream = response.getOutputStream();

//            List<TitleAndModelKey> titleAndModelKeys = new ArrayList<TitleAndModelKey>();
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("标签RID", "rid"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("客户物料编码", "material"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("单件数量", "deliveryQuantity"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商代码#生产日期#批次", "ln"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商物料编码", "supplierMaterial"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("MSD等级#ESD等级#LED等级#PCB等级", "spec"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("入库二维码", "bigBarCodeContent"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("创建日期", "createTime"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品牌", "brand"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产地", "productArea"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("物料描述", "materialDesc"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("华阳物料版本", "materialVersion"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单号", "purchaseOrderNo"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单行号", "purchaseOrderItemNo"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("软件版本", "version"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("MSD等级", "msdRank"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("LED等级", "ledRank"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("PCB等级", "pcbRank"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("打印尺寸", "printSize"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("打印日期", "printTime"));
//            
//            WritableWorkbook writableWorkbook = ExcelUtils.exportDataToExcel(outputStream, titleAndModelKeys, innerBoxList, "内箱", null, 0);
//            ExcelUtils.exportDataToExcel(outputStream, titleAndModelKeys, outtererBoxList, "外箱", null, 1, writableWorkbook);
            WritableWorkbook writableWorkbook = this.exportInnerBox(outputStream, innerBoxList);
            this.exportOuterBox(outputStream, outtererBoxList, writableWorkbook);
            writableWorkbook.write();
            outputStream.flush();
            writableWorkbook.close();
            outputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
            response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
        }
        return;
    }

    private WritableWorkbook exportInnerBox(OutputStream outputStream, List<LabelPrintRecordExtend> innerBoxList) throws IOException, RowsExceededException, WriteException, ParseException {
        List<TitleAndModelKey> titleAndModelKeys = new ArrayList<TitleAndModelKey>();
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("标签RID", "rid"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("客户物料编码", "material"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("单件数量", "innerPackageQuantity"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商代码#生产日期#批次", "ln"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商物料编码", "supplierMaterial"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("MSD等级#ESD等级#LED等级#PCB等级", "spec"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("入库二维码", "bigBarCodeContent"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("创建日期", "createTime"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品牌", "brand"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产地", "productArea"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("物料描述", "materialDesc"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("华阳物料版本", "materialVersion"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单号", "purchaseOrderNo"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单行号", "purchaseOrderItemNo"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("软件版本", "version"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("MSD等级", "msdRank"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("LED等级", "ledRank"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("PCB等级", "pcbRank"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("打印尺寸", "printSize"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("打印日期", "printTime"));

        WritableWorkbook writableWorkbook = ExcelUtils.exportDataToExcel(outputStream, titleAndModelKeys, innerBoxList, "内箱", null, 0);
        return writableWorkbook;
    }

    private void exportOuterBox(OutputStream outputStream, List<LabelPrintRecordExtend> outtererBoxList, WritableWorkbook writableWorkbook) throws RowsExceededException, WriteException, IOException, ParseException {
        List<TitleAndModelKey> titleAndModelKeys = new ArrayList<TitleAndModelKey>();
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("标签RID", "rid"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("客户物料编码", "material"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("单件数量", "outterPackageQuantity"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商代码#生产日期#批次", "ln"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商物料编码", "supplierMaterial"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("MSD等级#ESD等级#LED等级#PCB等级", "spec"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("入库二维码", "bigBarCodeContent"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("创建日期", "createTime"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品牌", "brand"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产地", "productArea"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("物料描述", "materialDesc"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("华阳物料版本", "materialVersion"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单号", "purchaseOrderNo"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单行号", "purchaseOrderItemNo"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("软件版本", "version"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("MSD等级", "msdRank"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("LED等级", "ledRank"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("PCB等级", "pcbRank"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("毛重", "grossWeight"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("净重", "netWeight"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("打印尺寸", "printSize"));
        titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("打印日期", "printTime"));

        ExcelUtils.exportDataToExcel(outputStream, titleAndModelKeys, outtererBoxList, "外箱", null, 1, writableWorkbook);
    }

//	@ApiOperation(httpMethod = "GET", value = "手工打印二维码，输出到EXCEL2")
//	@RequestMapping("service/exportToExcel2")
//	@ResponseBody
//	public void exportToExcel2(@ApiParam(value = "表主键ID列表，多个用英文逗号(,)隔开", defaultValue = "") String ids, HttpServletResponse response) {
//
//		if (org.springframework.util.StringUtils.isEmpty(ids)) {
//			throw new CommonException("数据不能为空", "base_canot_be_null", "数据");
//		}
//		String[] idArray = ids.split(",");
//
//		// 根据拿到的数据进行数据组装
//		List<String> idList = new ArrayList<>();
//		for(String id : idArray) {
//			idList.add(id);
//		}
////		List<LabelPrintRecordExtend> labelPrintRecords = this.labelPrintRecordService.exportToExcel(idList);
//		List<LabelPrintRecord> lists = this.labelPrintRecordService.createPrintDataDelivery(idList);
//		List<LabelPrintRecordExtend> labelPrintRecords = new ArrayList<>();
//		if (lists != null) {
//			for(LabelPrintRecord record : lists) {
//				String rId = SysUtil.getNextRid(record.getSupplierCode());
//				record.setRid(rId);
//				// 大条码内容
//				String material = record.getMaterial();// 客户物料
//				String materialVersion = record.getMaterialVersion() == null ? "" : record.getMaterialVersion();
//				String productDateStr = "";
//				if (record.getProductDate() != null) {
//					productDateStr = DateFormatUtils.format(record.getProductDate(), "yyyyMMdd");
//				}
//				String productBatch = record.getProductBatch() == null ? "" : record.getProductBatch();// 生产批次
//				record.setLn(record.getSupplierCode() + "#" + productDateStr + "#" + productBatch);
//				String msdRank = record.getMsdRank() == null ? "" : record.getMsdRank();
//				String esdRank = record.getEsdRank() == null ? "" : record.getEsdRank();
//				String ledRank = record.getLedRank() == null ? "" : record.getLedRank();
//				String pcbRank = record.getPcbRank() == null ? "" : record.getPcbRank();
//				String brand = record.getBrand() == null ? "" : record.getBrand();
//				String bigBarCodeContent = record.getRid() + "&" +material+ "&" + materialVersion + "&"
//						+ record.getOutterPackageQuantity() + "&" + record.getSupplierCode()
//						+ "#" + productDateStr + "#" + productBatch + "&" + msdRank + "#" + esdRank
//						+ "#" + ledRank + "#" + pcbRank + "&" + brand + "&" + record.getProductArea()
//						+ "&" + record.getPoitem() + "&" + record.getVersion() + "&"
//						+ record.getSupplierMaterial();
//				LabelPrintRecordExtend recordExtend = new LabelPrintRecordExtend();
//				BeanUtils.copyProperties(record, recordExtend);
//				recordExtend.setBigBarCodeContent(bigBarCodeContent);
//				labelPrintRecords.add(recordExtend);
//			}
//		}
//		
//		try {
//            response.reset();
//            StringBuffer header = new StringBuffer("attachment;");
//            header.append("filename=\"" + URLEncoder.encode("手工打印二维码.xls", "UTF-8") + "\";");
//            response.setHeader("Content-Disposition", header.toString());
//            response.setCharacterEncoding("UTF-8");
//            response.setContentType("application/vnd.ms-excel");
//            OutputStream outputStream = response.getOutputStream();
//            
//            List<TitleAndModelKey> titleAndModelKeys = new ArrayList<TitleAndModelKey>();
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("标签RID", "rid"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("客户物料编码", "material"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("单件数量", "deliveryQuantity"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商代码#生产日期#批次", "ln"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商物料编码", "supplierMaterial"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("MSD等级#ESD等级#LED等级#PCB等级", "spec"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("入库二维码", "bigBarCodeContent"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("创建日期", "createTime"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品牌", "brand"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("产地", "productArea"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("物料描述", "materialDesc"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("华阳物料版本", "materialVersion"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单号", "purchaseOrderNo"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单行号", "purchaseOrderItemNo"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("软件版本", "version"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("MSD等级", "msdRank"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("LED等级", "ledRank"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("PCB等级", "pcbRank"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("打印尺寸", "printSize"));
//            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("打印日期", "printTime"));
//            
//
//            WritableWorkbook writableWorkbook = ExcelUtils.exportDataToExcel(outputStream, titleAndModelKeys,
//            		labelPrintRecords, "手工打印二维码", null, 0);
//            writableWorkbook.write();
//            outputStream.flush();
//            writableWorkbook.close();
//            outputStream.close();
//        } catch (Exception e) {
//            e.printStackTrace();
//            response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
//        }
//		return;
//	}

}