package com.els.base.purchase.web.controller;

import java.io.OutputStream;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.company.entity.Company;
import com.els.base.company.service.CompanyPurRefService;
import com.els.base.company.service.CompanyService;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.file.entity.FileData;
import com.els.base.purchase.entity.PurchaseOrder;
import com.els.base.purchase.entity.PurchaseOrderExample;
import com.els.base.purchase.entity.PurchaseOrderItem;
import com.els.base.purchase.service.PurchaseOrderItemService;
import com.els.base.purchase.service.PurchaseOrderService;
import com.els.base.purchase.service.interceptor.PrintPurchaseOrderInterceptor;
import com.els.base.purchase.utils.PurchaseOrderSendStatusEnum;
import com.els.base.purchase.utils.PurchaseOrderStatusEnum;
import com.els.base.utils.SpringContextHolder;
import com.els.base.utils.excel.DateConverter;
import com.els.base.utils.excel.ExcelUtils;
import com.els.base.utils.excel.TitleAndModelKey;
import com.qqt.message.client.JsonUtil;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import jxl.write.WritableWorkbook;

@Api(value = "采购方订单头")
@Controller
@RequestMapping("purchaseOrderHeader")
public class PurchaseOrderHeaderController {
    Logger logger = LoggerFactory.getLogger(PurchaseOrderHeaderController.class);
    @Resource
    protected CompanyPurRefService companyPurRefService;
    @Resource
    protected CompanyService companyService;
    @Resource
    protected PurchaseOrderService purchaseOrderHeaderService;
    @Resource
    protected PurchaseOrderItemService purchaseOrderItemService;

    @ApiOperation(httpMethod = "POST", value = "批量发送采购方订单：入参（订单id数组）")
    @RequestMapping("service/send")
    @ResponseBody
    public ResponseResult<String> send(@RequestBody List<String> orderIdList) {

        if (CollectionUtils.isEmpty(orderIdList)) {
            throw new CommonException("id为空，操作失败", "id_is_blank");
        }
        for (String orderId : orderIdList) {
            PurchaseOrder purchaseOrderHeader = this.purchaseOrderHeaderService.queryObjById(orderId);
            if (purchaseOrderHeader == null) {
                throw new CommonException("采购方订单orderId[ " + orderId + " ]不存在", "do_not_exists",
                        "采购方订单orderId");
            }
            // 如果是已发送或者变更已发送
            if (purchaseOrderHeader.getOrderSendStatus() == PurchaseOrderSendStatusEnum.UPDATE_SENDED.getValue()
                    || purchaseOrderHeader.getOrderSendStatus() == PurchaseOrderSendStatusEnum.SENDED.getValue()) {
                // 如果是未确认或者已确认
                if (purchaseOrderHeader.getOrderStatus() != PurchaseOrderStatusEnum.BACK.getValue()
                        && purchaseOrderHeader.getOrderStatus() != PurchaseOrderStatusEnum.PART_BACK.getValue()) {
                    throw new CommonException("采购方订单[ " + purchaseOrderHeader.getOrderNo() + " ]已经发送,请勿重复发送");
                }
            }
            purchaseOrderHeader.setUserId(SpringSecurityUtils.getLoginUserId());
            purchaseOrderHeader.setUserName(SpringSecurityUtils.getLoginUserName());
            purchaseOrderHeader.setOrderStatus(PurchaseOrderStatusEnum.UNCONFIRM.getValue());
            this.purchaseOrderHeaderService.sendOrder(purchaseOrderHeader);
            // this.purchaseOrderHeaderService.sendPurchaseOrder(purchaseOrderHeader);
        }
        return ResponseResult.success();
    }


    @ApiOperation(httpMethod = "POST", value = "批量删除采购方订单：入参（订单id数组）")
    @RequestMapping("service/delete")
    @ResponseBody
    public ResponseResult<String> delete(@RequestBody List<String> orderIdList) {

        if (CollectionUtils.isEmpty(orderIdList)) {
            throw new CommonException("id为空，操作失败", "id_is_blank");
        }
        for (String orderId : orderIdList) {
            PurchaseOrder purchaseOrderHeader = this.purchaseOrderHeaderService.queryObjById(orderId);
            if (purchaseOrderHeader == null) {
                throw new CommonException("采购方订单orderId[ " + orderId + " ]不存在", "do_not_exists",
                        "采购方订单orderId");
            }
            // 如果是已发送或者变更已发送
            purchaseOrderHeader.setUserId(SpringSecurityUtils.getLoginUserId());
            purchaseOrderHeader.setUserName(SpringSecurityUtils.getLoginUserName());
            purchaseOrderHeader.setOrderStatus(PurchaseOrderStatusEnum.UNCONFIRM.getValue());
            this.purchaseOrderHeaderService.deleteOrder(purchaseOrderHeader);
            // this.purchaseOrderHeaderService.sendPurchaseOrder(purchaseOrderHeader);
        }
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "修改采购方订单：入参（整个订单实体）")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody PurchaseOrder purchaseOrderHeader) {
        this.purchaseOrderHeaderService.updateForSap(purchaseOrderHeader);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "查询采购方订单头")
    @RequestMapping("service/findTobeSentOrderByPage")
    @ResponseBody
    public ResponseResult<PageView<PurchaseOrder>> findTobeSentOrderByPage(
            @ApiParam(value = "所在页", defaultValue = "0") @RequestParam(defaultValue = "0") int pageNo,
            @ApiParam(value = "每页数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize,
            @ApiParam(value = "查询条件,属性名请参考 PurchaseOrderHeader") @RequestBody(required = false) QueryParamWapper wapper) {

        List<Company> companyList = companyPurRefService.queryAllCompanyByUserId(
                SpringSecurityUtils.getLoginUserId(), null);
        if (CollectionUtils.isEmpty(companyList)) {
            return ResponseResult.success(new PageView<PurchaseOrder>());
        }

        List<Integer> sendStatusList = new ArrayList<>();
        sendStatusList.add(PurchaseOrderSendStatusEnum.NOTSEND.getValue());
        sendStatusList.add(PurchaseOrderSendStatusEnum.UPDATE_NOTSEND.getValue());

        List<Integer> orderStatusList = new ArrayList<>();
        orderStatusList.add(PurchaseOrderStatusEnum.BACK.getValue());
        orderStatusList.add(PurchaseOrderStatusEnum.PART_BACK.getValue());

        PurchaseOrderExample example = new PurchaseOrderExample();
        example.setPageView(new PageView<PurchaseOrder>(pageNo, pageSize));

//        example.createCriteria().andOrderSendStatusIn(sendStatusList);
        example.createCriteria()
            .andOrderSendStatusIn(sendStatusList)
            .andIsEnableNotEqualTo(Constant.NO_INT);
 
        example.or()
            .andOrderSendStatusEqualTo(PurchaseOrderSendStatusEnum.UPDATE_NOTSEND.getValue())
            .andSendTimeIsNotNull();

        example.or().andOrderStatusIn(orderStatusList);

        for (PurchaseOrderExample.Criteria criteria : example.getOredCriteria()) {
            criteria.andProjectIdEqualTo(ProjectUtils.getProjectId());
            criteria.andCompanyIdEqualTo(CompanyUtils.currentCompanyId());

            if (wapper != null) {
                CriteriaUtils.addCriterion(criteria, wapper);
            }
        }

        //example.setOrderByClause("LAST_UPDATE_TIME DESC , CREATE_TIME DESC");
        example.setOrderByClause(" ORDER_DATE DESC , LAST_UPDATE_TIME DESC , CREATE_TIME DESC");
        PageView<PurchaseOrder> pageData = this.purchaseOrderHeaderService.queryObjNoItemsByPage(example,
                SpringSecurityUtils.getLoginUserId());
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "查询采购方订单头")
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<PurchaseOrder>> findByPage(
            @ApiParam(value = "所在页", defaultValue = "0") @RequestParam(defaultValue = "0") int pageNo,
            @ApiParam(value = "每页数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize,
            @ApiParam(value = "查询条件,属性名请参考 PurchaseOrderHeader") @RequestBody(required = false) QueryParamWapper wapper) {
        List<Company> companyList = companyPurRefService.queryAllCompanyByUserId(
                SpringSecurityUtils.getLoginUserId(), null);
        if (CollectionUtils.isEmpty(companyList)) {
            return ResponseResult.success(new PageView<PurchaseOrder>());
        }

        List<String> companyIds = new ArrayList<String>();
        for (Company company : companyList) {
            companyIds.add(company.getId());
        }

        List<Integer> isEnableValues = new ArrayList<Integer>();
        isEnableValues.add(1);
        isEnableValues.add(2);

        PurchaseOrderExample example = new PurchaseOrderExample();
        example.setPageView(new PageView<PurchaseOrder>(pageNo, pageSize));
        PurchaseOrderExample.Criteria criteria = example.createCriteria();
        criteria.andProjectIdEqualTo(ProjectUtils.getProjectId());
        criteria.andIsEnableIn(isEnableValues);
        criteria.andCompanyIdEqualTo(CompanyUtils.currentCompanyId());
        
        if(companyIds.size()>1000){
            int a = companyIds.size() / 1000;
            int b = companyIds.size() % 1000;
            
            List<String> list1 = new ArrayList<String>();
            List<String> list2 = new ArrayList<String>();
            List<String> list3 = new ArrayList<String>();
            list3.add(StringUtils.join(companyIds.subList(0, 1000), ","));
            example.createCriteria().andSupCompanyIdIn(list3);
            for (int i = 1; i < a; i++) {
                list1.add(StringUtils.join(companyIds.subList(i * 1000, (i + 1) * 1000), ","));
                list2.addAll(list1);
                list1.removeAll(list1);
                example.or().andSupCompanyIdIn(list2);
                list2.removeAll(list2);
            }
            if (b != 0) {
                list1.add( StringUtils.join(companyIds.subList(a * 1000, companyIds.size()), ",") );
                list2.addAll(list1);
                example.or().andSupCompanyIdIn(list2);
            }
        }else{
            criteria.andSupCompanyIdIn(companyIds);
        }
        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }
        example.setOrderByClause(" ORDER_DATE DESC , LAST_UPDATE_TIME DESC , CREATE_TIME DESC");
        PageView<PurchaseOrder> pageData = this.purchaseOrderHeaderService.queryObjNoItemsByPage(example, SpringSecurityUtils.getLoginUserId());
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "GET", value = "通过id查询采购方订单头历史")
    @RequestMapping("service/findHeaderHisById")
    @ResponseBody
    public ResponseResult<PageView<PurchaseOrder>> findHeaderHisById(
            @ApiParam(value = "所在页", defaultValue = "0") @RequestParam(defaultValue = "0") int pageNo,
            @ApiParam(value = "每页数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize,
            @ApiParam("采购订单头id") String id) {
        PurchaseOrderExample example = new PurchaseOrderExample();
        example.setPageView(new PageView<PurchaseOrder>(pageNo, pageSize));
        if (StringUtils.isBlank(id)) {
            throw new CommonException("id为空，操作失败", "id_is_blank");
        }

        PurchaseOrderExample.Criteria criteria = example.createCriteria();
        criteria.andIdEqualTo(id);
        example.setOrderByClause("version DESC");
        PageView<PurchaseOrder> pageData = this.purchaseOrderHeaderService.queryHisObjByPage(example);
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "采购订单打印")
    @RequestMapping("service/print")
    @ResponseBody
    public ResponseResult<FileData> print(@ApiParam("采购订单头id") String orderNo) {
        FileData fileData = null;
        if (StringUtils.isBlank(orderNo)) {
            throw new CommonException("查询失败,采购订单号不存在", "do_not_exists", "采购订单号");
        }
        String companyId = CompanyUtils.currentCompanyId();
        String projectId = ProjectUtils.getProjectId();
        PurchaseOrderExample purchaseOrderExample = new PurchaseOrderExample();
        purchaseOrderExample.createCriteria().andOrderNoEqualTo(orderNo);
        List<PurchaseOrder> purchaseOrderList = purchaseOrderHeaderService
                .queryAllObjByExample(purchaseOrderExample);
        String supplierShortName = "";
        if (CollectionUtils.isNotEmpty(purchaseOrderList)) {
            PurchaseOrder purchaseOrder = purchaseOrderList.get(0);
            if (StringUtils.isNotBlank(purchaseOrder.getSupCompanyId())) {
                Company company = companyService.queryObjById(purchaseOrder.getSupCompanyId());
                supplierShortName = company.getCompanyName();
            }
        }
        // 发布sap修改事件
        Map<String, PrintPurchaseOrderInterceptor> interceptors = SpringContextHolder
                .getBeans(PrintPurchaseOrderInterceptor.class);
        if (interceptors != null && interceptors.size() != 0) {

            Iterator<String> keyIterator = interceptors.keySet().iterator();
            while (keyIterator.hasNext()) {
                PrintPurchaseOrderInterceptor interceptor = interceptors.get(keyIterator.next());
                fileData = interceptor.preHandle(supplierShortName, orderNo, companyId, projectId);
                if (fileData == null) {
                    throw new CommonException("文件打印失败", "base_fail");
                }
            }
        }
        return ResponseResult.success(fileData);
    }

    @ApiOperation(httpMethod = "POST", value = "采购订单批量打印")
    @RequestMapping("service/printForBatch")
    @ResponseBody
    public ResponseResult<FileData> printForBatch(@ApiParam("采购订单头id") @RequestBody List<String> orderNoList) {
        FileData fileData = null;
        List<PurchaseOrder> purchaseOrders = new ArrayList<PurchaseOrder>();
        for (String orderNo : orderNoList) {
            PurchaseOrder purchaseOrder = null;
            if (StringUtils.isBlank(orderNo)) {
                throw new CommonException("查询失败,采购订单号不存在", "do_not_exists", "采购订单号");
            }
            String companyId = CompanyUtils.currentCompanyId();
            String projectId = ProjectUtils.getProjectId();
            PurchaseOrderExample purchaseOrderExample = new PurchaseOrderExample();
            purchaseOrderExample.createCriteria().andOrderNoEqualTo(orderNo);
            List<PurchaseOrder> purchaseOrderList = purchaseOrderHeaderService
                    .queryAllObjByExample(purchaseOrderExample);
            String supplierShortName = "";
            if (CollectionUtils.isNotEmpty(purchaseOrderList)) {
                purchaseOrder = purchaseOrderList.get(0);
                if (StringUtils.isNotBlank(purchaseOrder.getSupCompanyId())) {
                    Company company = companyService.queryObjById(purchaseOrder.getSupCompanyId());
                    supplierShortName = company.getCompanyName();
                }
            }
            purchaseOrder.setCompanyId(companyId);
            purchaseOrder.setProjectId(projectId);
            purchaseOrder.setSupShortName(supplierShortName);
            purchaseOrders.add(purchaseOrder);
        }

        // 发布sap修改事件
        Map<String, PrintPurchaseOrderInterceptor> interceptors = SpringContextHolder
                .getBeans(PrintPurchaseOrderInterceptor.class);
        if (interceptors != null && interceptors.size() != 0) {
            Iterator<String> keyIterator = interceptors.keySet().iterator();
            while (keyIterator.hasNext()) {
                PrintPurchaseOrderInterceptor interceptor = interceptors.get(keyIterator.next());
                logger.info("purchaseOrders-----==-" + purchaseOrders.size());
                fileData = interceptor.preHandleForBatch(purchaseOrders);
                if (fileData == null) {
                    throw new CommonException("文件打印失败", "base_fail");
                }
            }
        }
        return ResponseResult.success(fileData);
    }

    @ApiOperation(value = "采购订单批量导出Excel", httpMethod = "GET")
    @RequestMapping(value = "service/downloadExcel")
    @ResponseBody
    public ModelAndView downloadExcel(String queryParams, HttpServletResponse response) {
        try {

            List<Company> supCompanyList = this.companyPurRefService.queryAllCompanyByUserId(
                    SpringSecurityUtils.getLoginUserId(), null);
            List<String> supCompanyIdList = new ArrayList<>();

            for (int i = 0; CollectionUtils.isNotEmpty(supCompanyList) && i < supCompanyList.size(); i++) {
                supCompanyIdList.add(supCompanyList.get(i).getId());
            }

            List<PurchaseOrderItem> purOrderList = null;
            if (CollectionUtils.isEmpty(supCompanyIdList)) {
                purOrderList = new ArrayList<>();

            } else {
                PurchaseOrderExample purchaseOrder = new PurchaseOrderExample();
                PurchaseOrderExample.Criteria createCriteria = purchaseOrder.createCriteria();
                createCriteria.andCompanyIdEqualTo(CompanyUtils.currentCompanyId())
                        .andProjectIdEqualTo(ProjectUtils.getProjectId()).andSupCompanyIdIn(supCompanyIdList);

                if (StringUtils.isNotEmpty(queryParams)) {
                    QueryParamWapper queryParamWapper = JsonUtil.convertValue(queryParams,
                            QueryParamWapper.class);
                    CriteriaUtils.addCriterion(createCriteria, queryParamWapper);
                }

                purOrderList = this.purchaseOrderItemService.selectForExcel(purchaseOrder);
            }

            response.reset();
            StringBuffer header = new StringBuffer("attachment;");
            String companyShortName = CompanyUtils.currentCompany().getCompanyName();

            header.append("filename=\"" + URLEncoder.encode("" + companyShortName + "PO明细.xls", "UTF-8")
                    + "\";");
            response.setHeader("Content-Disposition", header.toString());
            response.setCharacterEncoding("UTF-8");
            response.setContentType("application/vnd.ms-excel");
            OutputStream outputStream = response.getOutputStream();
            // logger.info("--------------list" + purOrderList.size());
            for (PurchaseOrderItem purchaseOrderItem : purOrderList) {
                BigDecimal quantity = purchaseOrderItem.getQuantity();// 订单数量
                BigDecimal receivedQuantity = purchaseOrderItem.getReceivedQuantity();// 已经收货数量
                BigDecimal noReceivedQuantity = BigDecimal.ZERO;
                if (purchaseOrderItem.getQuantity() != null) {
                    if (purchaseOrderItem.getReceivedQuantity() != null) {
                        noReceivedQuantity = quantity.subtract(receivedQuantity);
                    } else {
                        noReceivedQuantity = purchaseOrderItem.getQuantity();
                    }
                }
                purchaseOrderItem.setNoReceivedQuantity(noReceivedQuantity);
                if (purchaseOrderItem.getIsEnable() != null && purchaseOrderItem.getIsEnable() == 2) {
                    purchaseOrderItem.setFrozenFlag("已冻结");
                } else {
                    purchaseOrderItem.setFrozenFlag("未冻结");
                }
            }
            List<TitleAndModelKey> titleAndModelKeys = new ArrayList<TitleAndModelKey>();
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商编码", "companyCode"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商简称", "companyShortName"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单号", "orderNo"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("项目", "orderItemNo"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品号", "materialCode"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品名规格", "materialName"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("订单数量", "quantity"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("已交数量", "receivedQuantity"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("未交数量", "noReceivedQuantity"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("冻结标识", "frozenFlag"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("单位", "orderUnit"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("净价", "noTaxPrice"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("价格单位", "priceUnit"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购金额", "nonTaxAmount"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("币种", "currency"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("税率", "saleTaxCode"));
            // titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("凭证日期", "orderDate"));
            TitleAndModelKey titleAndModelKey = ExcelUtils.createTitleAndModelKey("凭证日期", "orderDate");
            DateConverter dateConverter = new DateConverter().setDateFormat("yyyy-MM-dd");
            titleAndModelKey.setToObjConverter(dateConverter);
            titleAndModelKey.setToStrConverter(dateConverter);
            titleAndModelKeys.add(titleAndModelKey);
            // titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("交货日期", "deliveredDate"));
            TitleAndModelKey titleAndModelKey2 = ExcelUtils.createTitleAndModelKey("交货日期", "deliveredDate");
            DateConverter dateConverter2 = new DateConverter().setDateFormat("yyyy-MM-dd");
            titleAndModelKey2.setToObjConverter(dateConverter2);
            titleAndModelKey2.setToStrConverter(dateConverter2);
            titleAndModelKeys.add(titleAndModelKey2);
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商物料号", "supMaterialCode"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品牌", "brand"));

            WritableWorkbook writableWorkbook = ExcelUtils.exportDataToExcel(outputStream, titleAndModelKeys,
                    purOrderList, "采购订单", null, 0);
            writableWorkbook.write();
            outputStream.flush();
            writableWorkbook.close();
            outputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
            response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
        }
        return null;
    }

    @ApiOperation(value = "采购订单选中导出Excel", httpMethod = "GET")
    @RequestMapping(value = "service/downloadExcelBySelect")
    @ResponseBody
    public ModelAndView downloadExcelBySelect(
            @ApiParam(value = "采购订单号", defaultValue = "") @RequestParam(defaultValue = "") String orderNOList,
            HttpServletResponse response) {
        try {
            if (StringUtils.isEmpty(orderNOList)) {
                throw new CommonException("id为空，操作失败", "id_is_blank");
            }
            // TODO Auto-generated catch block
            response.reset();
            StringBuffer header = new StringBuffer("attachment;");
            header.append("filename=\"" + URLEncoder.encode("供应商简称PO明细.xls", "UTF-8") + "\";");
            response.setHeader("Content-Disposition", header.toString());
            response.setCharacterEncoding("UTF-8");
            response.setContentType("application/vnd.ms-excel");
            OutputStream outputStream = response.getOutputStream();

            PurchaseOrderExample orderExample = new PurchaseOrderExample();
            PurchaseOrderExample.Criteria createCriteria = orderExample.createCriteria();
            createCriteria.andCompanyIdEqualTo(CompanyUtils.currentCompanyId()).andProjectIdEqualTo(
                    ProjectUtils.getProjectId());

            if (StringUtils.isNotBlank(orderNOList)) {
                createCriteria.andOrderNoIn(Arrays.asList(orderNOList.split(",")));
            }
            List<PurchaseOrderItem> listOrderItem = this.purchaseOrderItemService
                    .selectForExcel(orderExample);
            for (int i = 0; CollectionUtils.isNotEmpty(listOrderItem) && i < listOrderItem.size(); i++) {
                PurchaseOrderItem purchaseOrderItem = listOrderItem.get(i);
                BigDecimal quantity = purchaseOrderItem.getQuantity();// 订单数量
                BigDecimal receivedQuantity = purchaseOrderItem.getReceivedQuantity();// 已经收货数量
                BigDecimal noReceivedQuantity = BigDecimal.ZERO;
                if (purchaseOrderItem.getQuantity() != null) {
                    if (purchaseOrderItem.getReceivedQuantity() != null) {
                        noReceivedQuantity = quantity.subtract(receivedQuantity);
                    } else {
                        noReceivedQuantity = purchaseOrderItem.getQuantity();
                    }
                }
                purchaseOrderItem.setNoReceivedQuantity(noReceivedQuantity);
                if (purchaseOrderItem.getIsEnable() != null && purchaseOrderItem.getIsEnable() == 2) {
                    purchaseOrderItem.setFrozenFlag("已冻结");
                } else {
                    purchaseOrderItem.setFrozenFlag("未冻结");
                }
            }
            List<TitleAndModelKey> titleAndModelKeys = new ArrayList<TitleAndModelKey>();
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商编码", "companyCode"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商简称", "companyShortName"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购订单号", "orderNo"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("项目", "orderItemNo"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品号", "materialCode"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品名规格", "materialName"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("订单数量", "quantity"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("已交数量", "receivedQuantity"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("未交数量", "noReceivedQuantity"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("冻结标识", "frozenFlag"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("单位", "orderUnit"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("净价", "noTaxPrice"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("价格单位", "priceUnit"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("采购金额", "nonTaxAmount"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("币种", "currency"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("税率", "saleTaxCode"));
            // titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("凭证日期", "orderDate"));
            TitleAndModelKey titleAndModelKey = ExcelUtils.createTitleAndModelKey("凭证日期", "orderDate");
            DateConverter dateConverter = new DateConverter().setDateFormat("yyyy-MM-dd");
            titleAndModelKey.setToObjConverter(dateConverter);
            titleAndModelKey.setToStrConverter(dateConverter);
            titleAndModelKeys.add(titleAndModelKey);
            // titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("交货日期", "deliveredDate"));
            TitleAndModelKey titleAndModelKey2 = ExcelUtils.createTitleAndModelKey("交货日期", "deliveredDate");
            DateConverter dateConverter2 = new DateConverter().setDateFormat("yyyy-MM-dd");
            titleAndModelKey2.setToObjConverter(dateConverter2);
            titleAndModelKey2.setToStrConverter(dateConverter2);
            titleAndModelKeys.add(titleAndModelKey2);
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("供应商物料号", "supMaterialCode"));
            titleAndModelKeys.add(ExcelUtils.createTitleAndModelKey("品牌", "brand"));
            WritableWorkbook writableWorkbook = ExcelUtils.exportDataToExcel(outputStream, titleAndModelKeys,
                    listOrderItem, "采购订单", null, 0);
            writableWorkbook.write();
            outputStream.flush();
            writableWorkbook.close();
            outputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
            response.setStatus(HttpServletResponse.SC_EXPECTATION_FAILED);
        }
        return null;
    }

    /**
     * 采购订单PDF导出
     */
    @ApiOperation(httpMethod = "POST", value = "采购订单PDF导出")
    @RequestMapping("service/exportPDF")
    @ResponseBody
    public ResponseResult<List<FileData>> exportPDF(@ApiParam("采购订单头id") @RequestBody List<String> orderNoList) {
    	// 检验参数
    	if (orderNoList.isEmpty()) {
    		throw new CommonException("导出失败，未选择导出数据！");
    	}
    	
    	List<FileData> fileDatas = purchaseOrderHeaderService.getPdfFiles(orderNoList);
    	// 判断是否查询出文件信息集合
    	if (fileDatas.isEmpty()) {
    		throw new CommonException("未查询出数据，导出失败！");
    	}
    	
    	return ResponseResult.success(fileDatas);
    }
}
