package com.els.base.voucher.service.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.delivery.entity.DeliveryOrder;
import com.els.base.delivery.entity.DeliveryOrderItem;
import com.els.base.delivery.service.DeliveryOrderItemService;
import com.els.base.delivery.service.DeliveryOrderService;
import com.els.base.delivery.utils.DeliveryStatusEnum;
import com.els.base.purchase.entity.PurchaseOrder;
import com.els.base.purchase.entity.PurchaseOrderExample;
import com.els.base.purchase.entity.PurchaseOrderItem;
import com.els.base.purchase.entity.PurchaseOrderItemExample;
import com.els.base.purchase.service.PurchaseOrderItemService;
import com.els.base.purchase.service.PurchaseOrderService;
import com.els.base.supperorder.entity.SupplierOrder;
import com.els.base.supperorder.entity.SupplierOrderItem;
import com.els.base.supperorder.entity.SupplierOrderItemExample;
import com.els.base.supperorder.service.SupplierOrderItemService;
import com.els.base.supperorder.service.SupplierOrderService;
import com.els.base.voucher.dao.VoucherMapper;
import com.els.base.voucher.entity.Voucher;
import com.els.base.voucher.entity.VoucherDelivery;
import com.els.base.voucher.entity.VoucherExample;
import com.els.base.voucher.entity.VoucherItem;
import com.els.base.voucher.service.VoucherDeliveryService;
import com.els.base.voucher.service.VoucherItemService;
import com.els.base.voucher.service.VoucherService;

@Service("voucherService2")
public class VoucherServiceImpl2 implements VoucherService {

    private static Logger logger = LoggerFactory.getLogger(VoucherServiceImpl2.class);

    @Resource
    protected SupplierOrderService supplierOrderService;

    @Resource
    protected SupplierOrderItemService supplierOrderItemService;

    @Resource
    protected PurchaseOrderService purchaseOrderHeaderService;

    @Resource
    protected PurchaseOrderItemService purchaseOrderItemService;

    @Resource
    protected DeliveryOrderService deliveryOrderService;

    @Resource
    protected DeliveryOrderItemService deliveryOrderItemService;

    @Resource
    protected VoucherDeliveryService voucherDeliveryService;

    @Resource
    protected VoucherItemService voucherItemService;

    @Resource
    protected VoucherMapper voucherMapper;

    @CacheEvict(value = { "voucher" }, allEntries = true)
    @Override
    public void addObj(Voucher t) {
        this.voucherMapper.insertSelective(t);
    }

    @CacheEvict(value = { "voucher" }, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.voucherMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = { "voucher" }, allEntries = true)
    @Override
    public void modifyObj(Voucher t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.voucherMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "voucher", keyGenerator = "redisKeyGenerator")
    @Override
    public Voucher queryObjById(String id) {
        return this.voucherMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "voucher", keyGenerator = "redisKeyGenerator")
    @Override
    public List<Voucher> queryAllObjByExample(VoucherExample example) {
        return this.voucherMapper.selectByExample(example);
    }

    @Cacheable(value = "voucher", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<Voucher> queryObjByPage(VoucherExample example) {
        PageView<Voucher> pageView = example.getPageView();
        pageView.setQueryResult(this.voucherMapper.selectByExampleByPage(example));
        return pageView;
    }

    @CacheEvict(value = { "voucher", "voucherItem" }, allEntries = true)
    @Transactional
    @Override
    public void importVoucher(List<Voucher> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }

        // 1 根据凭证，更新采购订单的收货数量，可发货数量，是否完成
        logger.info("根据凭证，更新采购订单的收货数量，可发货数量，是否完成");
        this.handleForVoucher(list);

        // 2 保存物料凭证的数据
        this.importIntoDb(list);
    }

    /**
     * 把凭证头和行，还有送货单关系，插入到数据库
     * 
     * @param list
     */
    private void importIntoDb(List<Voucher> list) {
        for (Voucher voucher : list) {
        	
        	VoucherExample voucherExample = new VoucherExample();
        	voucherExample.createCriteria().andVoicherNoEqualTo(voucher.getVoicherNo());
        	
        	List<Voucher> voucherList = this.voucherMapper.selectByExample(voucherExample);
        	if (CollectionUtils.isEmpty(voucherList)) {
        		this.voucherMapper.insertSelective(voucher);
			}else {
				voucher.setId(voucherList.get(0).getId());
				voucher.setLastUpdateTime(new Date());
				this.voucherMapper.updateByPrimaryKeySelective(voucher);
			}

            List<VoucherItem> voucherItems = voucher.getItems();
            for (int i = 0; CollectionUtils.isNotEmpty(voucherItems) && i < voucherItems.size(); i++) {
                this.voucherItemService.addObj(voucherItems.get(i));

                List<VoucherDelivery> voucherDeliveryList = voucherItems.get(0).getVoucherDeliveryList();

                for (int j = 0; CollectionUtils.isNotEmpty(voucherDeliveryList)
                        && j < voucherDeliveryList.size(); j++) {
                    this.voucherDeliveryService.addObj(voucherDeliveryList.get(j));
                }
            }

        }
    }

    private void handleForDeliveryOrder(List<VoucherDelivery> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }

        Map<String, DeliveryOrder> deliveryOrderMap = new HashMap<>();
        for (VoucherDelivery voucherDelivery : itemList) {
        	logger.info("处理物料凭证过账信息，送货单头[{}],送货单行[{}]", voucherDelivery.getDeliveryOrderNo(), voucherDelivery.getDeliveryOrderItemNo());
        	
            DeliveryOrderItem deliveryOrderItem = voucherDelivery.getDeliveryOrderItem();
            if (deliveryOrderItem == null) {
                logger.warn("系统没找到凭证号下的该送货单，不做处理。送货单号[{}], 送货单行[{}]", voucherDelivery.getDeliveryOrderNo(),
                        voucherDelivery.getDeliveryOrderItemNo());
            }

            // 1 获取送货单信息
            DeliveryOrder deliveryOrder = this.deliveryOrderService.queryObjById(deliveryOrderItem
                    .getDeliveryOrderId());

            // 2 检查送货单是否已经发货，如果状态是未发货，但是已经收到了收货凭证，那么就先做发货操作
            if (DeliveryStatusEnum.UN_RECEIVED.getValue().equals(deliveryOrder.getDeliveryStatus())) {
                deliveryOrder.setItems(this.deliveryOrderItemService.queryByDeliveryOrderId(deliveryOrder.getId()));
                logger.info("送货单[{}]状态未发货，但收到收货凭证，说明已经发货了，执行发货操作", deliveryOrder.getDeliveryOrderNo());
                this.deliveryOrderService.sendDelivery(deliveryOrder);
            }

            // 3 送货单行开始收货
            DeliveryOrderItem tmpDeliveryOrderItem = new DeliveryOrderItem();
            tmpDeliveryOrderItem.setId(deliveryOrderItem.getId());
            if (voucherDelivery.getVoucherDate() != null) {
                tmpDeliveryOrderItem.setWarehouseScan(DateFormatUtils.format(voucherDelivery.getVoucherDate(), "yyyy-MM-dd"));
            }

            // 4、收货量
            String receiveQuatity = deliveryOrderItem.getReceiptQuantity();
            if (StringUtils.isBlank(receiveQuatity)) {
                receiveQuatity = "0";
            }

            Integer mobileType = voucherDelivery.getMobileType();
            if (mobileType == 101 || mobileType == 105 || mobileType == 123 || mobileType == 161) { // 收货事件
                // 161 在sap上是退货事件，实际做的是收货的处理。162则相反。具体业务请询问项目经理。
                receiveQuatity = new BigDecimal(receiveQuatity).add(voucherDelivery.getQuantity()).toString();
            } else {
                // 退货事件
                continue;
            }
            tmpDeliveryOrderItem.setDeliveryStatus(DeliveryStatusEnum.ALL_RECEIVED.getValue());
            tmpDeliveryOrderItem.setReceiptQuantity(receiveQuatity);
            logger.info("送货单收货,头[{}]，行[{}],收货[{}]-->[{}]，完成状态[{}]",deliveryOrderItem.getDeliveryOrderNo(), deliveryOrderItem.getDeliveryOrderItemNo(), deliveryOrderItem.getReceiptQuantity(), receiveQuatity,tmpDeliveryOrderItem.getDeliveryStatus());
            this.deliveryOrderItemService.modifyObj(tmpDeliveryOrderItem);

            DeliveryOrder tmp = deliveryOrderMap.get(deliveryOrder.getId());
            if (tmp == null) {
                tmp = new DeliveryOrder();
                deliveryOrderMap.put(deliveryOrder.getId(), tmp);
            }
            tmp.setId(deliveryOrder.getId());
            tmp.setDeliveryStatus(DeliveryStatusEnum.ALL_RECEIVED.getValue());
            tmp.setReceiveDate(voucherDelivery.getPostTime());
            tmp.setDeliveryOrderNo(deliveryOrder.getDeliveryOrderNo());
        }

        Iterator<String> deliveryOrderIdIterator = deliveryOrderMap.keySet().iterator();
        while (deliveryOrderIdIterator.hasNext()) {
            // 3.3 更新送货单的状态，改为已经收货
            DeliveryOrder tmp = deliveryOrderMap.get(deliveryOrderIdIterator.next());
            // 修改送货头
            this.deliveryOrderService.modifyObj(tmp);
            logger.info("更新送货单头，[{}],改为全部收货", tmp.getDeliveryOrderNo());
        }

    }

    /**
     * 根据物料凭证，处理采购订单，更新收货数量、在途数量等信息
     * 
     * @param list
     */
    private void handleForVoucher(List<Voucher> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }

        // 1 根据订单号，把凭证进行分组。map中，key是订单号，value是凭证行列表
        Map<String, List<VoucherItem>> purchaseOrderNoAndVoucherItemMap = this.gourpByPurchaseOrder(list);

        // 2 根据不同的订单号，逐个处理
        Iterator<String> purchaseOrderNoIterator = purchaseOrderNoAndVoucherItemMap.keySet().iterator();
        while (purchaseOrderNoIterator.hasNext()) {

            String orderNo = purchaseOrderNoIterator.next();
            this.handleForVoucherItem(orderNo, purchaseOrderNoAndVoucherItemMap.get(orderNo));
        }
    }

    /**
     * 根据订单号把物料凭证做分组，相同订单号的物料凭证分到同一个组
     * 
     * @param list
     * @return
     */
    private Map<String, List<VoucherItem>> gourpByPurchaseOrder(List<Voucher> list) {
        Map<String, List<VoucherItem>> result = new HashMap<>();
        for (Voucher voucher : list) {
            List<VoucherItem> voucherItemList = voucher.getItems();
            if (CollectionUtils.isEmpty(voucherItemList)) {
                continue;
            }

            for (VoucherItem item : voucherItemList) {
                String orderNo = item.getPurOrderNo();
                List<VoucherItem> temp = result.get(orderNo);
                if (temp == null) {
                    temp = new ArrayList<>();
                    result.put(orderNo, temp);
                }
                temp.add(item);
            }
        }
        return result;
    }

    /**
     * 根据订单号，处理物料凭证
     * 
     * @param orderNo
     * @param itemList
     */
    public void handleForVoucherItem(String orderNo, List<VoucherItem> itemList) {
        // 1、找出凭证关联的订单
        PurchaseOrderExample purchaseOrderExample = new PurchaseOrderExample();
        purchaseOrderExample.createCriteria().andOrderNoEqualTo(orderNo);
        
        List<PurchaseOrder> orderList = this.purchaseOrderHeaderService.queryAllObjByExample(purchaseOrderExample);
        if (CollectionUtils.isEmpty(orderList)) {
            logger.warn("系统没有找到该采购订单，不做处理。订单编码[{}]", orderNo);
            return;
        }

        // 2、更新各个采购订单行。
        // 根据凭证，更新订单行在途数量与收货数量
        logger.info("开始 更新采购订单行 状态", orderNo);
        this.updatePurchaseItemInfo(itemList);

        // 3 更新订单头。
        // 根据凭证，检查采购订单的是否已完成标记。查看是否存在未完成的订单行，如果没有，也就是说所有行都完成，那么整个采购订单都完成了
        logger.info("开始 更新采购订单头送货状态[{}].", orderNo);
        this.updateOrderDeliveryStatus(orderList.get(0));
    }

    /**
     * 更新采购订单的完成状态
     * 
     * @param deliveryStatus
     * @param orderNo
     */
    private void updateOrderDeliveryStatus(PurchaseOrder order) {
    	PurchaseOrderItemExample purchaseOrderItemExample = new PurchaseOrderItemExample();
        purchaseOrderItemExample.createCriteria()
            .andOrderIdEqualTo(order.getId())
            .andIsEnableEqualTo(Constant.YES_INT)
            .andFinishFlagEqualTo(Constant.NO_STRING);
        
        //查询未完成的订单行
        List<PurchaseOrderItem> unFinishPurItemList = this.purchaseOrderItemService.queryAllObjByExample(purchaseOrderItemExample);

        PurchaseOrder tempPurOrder = new PurchaseOrder();
        tempPurOrder.setId(order.getId());
        
        Integer deliveryStatus = null;
        if (CollectionUtils.isEmpty(unFinishPurItemList)
                && !Integer.valueOf(2).equals(order.getDeliveryStatus())) {
        	//没有未完成的，而且状态不是已完成的
        	deliveryStatus = 2;
        	
        } else if (Integer.valueOf(2).equals(order.getDeliveryStatus())) {
        	deliveryStatus = 1;
        }
        
        if(deliveryStatus != null){
        	tempPurOrder.setDeliveryStatus(deliveryStatus);
        	this.purchaseOrderHeaderService.modifyObjByKey(tempPurOrder);
        	logger.info("更新采购订单[{}],订单完成状态[{}].", order.getOrderNo(), deliveryStatus);
        }
		
		SupplierOrderItemExample supplierOrderItemExample = new SupplierOrderItemExample();
		supplierOrderItemExample.createCriteria()
		.andOrderIdEqualTo(order.getId())
		.andIsEnableEqualTo(Constant.YES_INT)
		.andFinishFlagEqualTo(Constant.NO_STRING);
		
		//查询未完成的订单行
		List<SupplierOrderItem> unFinishSupItemList = this.supplierOrderItemService.queryAllObjByExample(supplierOrderItemExample);
		
		SupplierOrder tempSupOrder = new SupplierOrder();
		tempSupOrder.setId(order.getId());
		
		if (CollectionUtils.isEmpty(unFinishSupItemList)
				&& !Integer.valueOf(2).equals(order.getDeliveryStatus())) {
			//没有未完成的，而且状态不是已完成的
			deliveryStatus = 2;
			
		} else if (Integer.valueOf(2).equals(order.getDeliveryStatus())) {
			deliveryStatus = 1;
		}
		
		if (deliveryStatus != null) {
			tempSupOrder.setDeliveryStatus(deliveryStatus);
			this.supplierOrderService.modifyObj(tempSupOrder);
			logger.info("更新销售订单[{}],订单完成状态[{}].", order.getOrderNo(), deliveryStatus);
		}
			

    }

    /**
     * 根据收货数量，更新订单行的收货数量，可发货数量，是否完成
     * 
     * @param orderNo
     * @param itemList
     */
    private void updatePurchaseItemInfo(List<VoucherItem> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }

        List<VoucherItem> removeList = new ArrayList<>();
        
        for (VoucherItem item : itemList) {
        	logger.info("处理物料凭证行，头[{}],行[{}]", item.getVoucherNo(), item.getVoucherItemNo());
        	
            Integer mobileType = item.getMobileType();
            if (mobileType != 102 && mobileType != 122 && mobileType != 106 && mobileType != 161
                    && mobileType != 101 && mobileType != 105 && mobileType != 123 && mobileType != 162) {
            	
            	//如果是无法识别的移动类型，就不用处理
                logger.warn("凭证头[{}], 凭证行[{}],无法识别移动类型mobileType[{}]", new Object[] { item.getVoucherNo(), item.getVoucherItemNo(), mobileType });
                continue;
            }

            // 1、根据凭证行的数据，查询关联的采购订单行
            if (item.getPurchaseOrderItem() == null) {
            	continue;
            }

//          由于数据可以能会发生变化，所以每次执行更新前，需要取最新数据
            PurchaseOrderItem purchaseOrderItem = this.purchaseOrderItemService.queryObjById(item.getPurchaseOrderItem().getId());
            PurchaseOrderItem tempPurchaseOrderItem = new PurchaseOrderItem();
            tempPurchaseOrderItem.setId(purchaseOrderItem.getId());
            
//          由于数据可以能会发生变化，所以每次执行更新前，需要取最新数据
            SupplierOrderItem supplierOrderItem = item.getSupplierOrderItem();
            SupplierOrderItem tempSupplierOrderItem = null;
            if (supplierOrderItem != null) {
            	supplierOrderItem = this.supplierOrderItemService.queryObjById(item.getSupplierOrderItem().getId());
            	tempSupplierOrderItem = new SupplierOrderItem();
            	tempSupplierOrderItem.setId(supplierOrderItem.getId());
			}

            // 2、根据订单行的数据进行计算
            if (mobileType == 102 || mobileType == 122 || mobileType == 106 || mobileType == 162) {
            	// 2.1 退货事件，冲销事件。 移动类型 162 在sap上是冲销事件。具体业务请询问项目经理 张良。
            	
            	// 2.1.1 收货数量减少
            	tempPurchaseOrderItem.setReceivedQuantity(purchaseOrderItem.getReceivedQuantity().subtract(item.getQuantity()));
            	// 2.1.2 可发货数量增加
            	//tempPurchaseOrderItem.setDeliveryAmount(purchaseOrderItem.getDeliveryAmount() + item.getQuantity().longValue());
                tempPurchaseOrderItem.setDeliveryAmount(purchaseOrderItem.getDeliveryAmount().add(item.getQuantity()));
            	if (tempSupplierOrderItem != null) {
            		tempSupplierOrderItem.setReceivedQuantity(supplierOrderItem.getReceivedQuantity().subtract(item.getQuantity()));
            		//tempSupplierOrderItem.setDeliveryAmount(supplierOrderItem.getDeliveryAmount() + item.getQuantity().longValue());
                    tempSupplierOrderItem.setDeliveryAmount(supplierOrderItem.getDeliveryAmount().add( item.getQuantity()));
            	}
            	
            }else{
            	// 2.2 收货事件
            	
            	// 2.2.1 在途数量减少。如果有关联送货单，就根据送货单的送货数量来扣减。如果没有，就不扣减
            	BigDecimal deliveryQuantity = BigDecimal.ZERO;
            	if (CollectionUtils.isNotEmpty(item.getVoucherDeliveryList())) {
            		for (VoucherDelivery delivery : item.getVoucherDeliveryList()) {
            			if (delivery.getDeliveryOrderItem() != null) {
            				deliveryQuantity = deliveryQuantity.add(delivery.getDeliveryOrderItem().getDeliveryQuantity());
            			}
            		}
            	}
            	
            	// 2.2.2 在途数量减少
            	tempPurchaseOrderItem.setOnwayQuantity(purchaseOrderItem.getOnwayQuantity().subtract(deliveryQuantity));
            	
            	// 2.2.3 收货数量增加
            	tempPurchaseOrderItem.setReceivedQuantity(purchaseOrderItem.getReceivedQuantity().add(item.getQuantity()));
            	
            	// 2.2.4 可发货数量减少，可发货数量 = 订单数量 - 在途数量 - 收货数量
            	BigDecimal diff = item.getQuantity().subtract(deliveryQuantity);
            	//tempPurchaseOrderItem.setDeliveryAmount(purchaseOrderItem.getDeliveryAmount() - diff.longValue());
                tempPurchaseOrderItem.setDeliveryAmount(purchaseOrderItem.getDeliveryAmount().subtract(diff));
            	if (tempSupplierOrderItem !=null) {
            		tempSupplierOrderItem.setReceivedQuantity(supplierOrderItem.getReceivedQuantity().add(item.getQuantity()));
            		tempSupplierOrderItem.setOnwayQuantity(supplierOrderItem.getOnwayQuantity().subtract(deliveryQuantity));
            		//tempSupplierOrderItem.setDeliveryAmount(supplierOrderItem.getDeliveryAmount() - diff.longValue());
                    tempSupplierOrderItem.setDeliveryAmount(supplierOrderItem.getDeliveryAmount().subtract(diff));
            	}
            }
            
            tempPurchaseOrderItem.setFinishFlag(tempPurchaseOrderItem.getReceivedQuantity().compareTo(purchaseOrderItem.getQuantity()) >=0 ? Constant.YES_STRING : Constant.NO_STRING);
            logger.info("更新采购订单,头[{}]，行[{}]在途数量[{}], 收货数量[{}]，可发货数量[{}],订单数量[{}],是否已完成[{}].",new Object[] { 
            		purchaseOrderItem.getOrderNo(), 
            		purchaseOrderItem.getOrderItemNo(),
            		tempPurchaseOrderItem.getOnwayQuantity(), 
            		tempPurchaseOrderItem.getReceivedQuantity(), 
            		tempPurchaseOrderItem.getDeliveryAmount(), 
            		purchaseOrderItem.getQuantity(),
            		tempPurchaseOrderItem.getFinishFlag() });
            tempPurchaseOrderItem.setUpdateTime(new Date());
            this.purchaseOrderItemService.modifyObj(tempPurchaseOrderItem);
            
            if (tempSupplierOrderItem!=null) {
            	tempSupplierOrderItem.setFinishFlag(tempSupplierOrderItem.getReceivedQuantity().compareTo(supplierOrderItem.getQuantity()) >=0 ? Constant.YES_STRING : Constant.NO_STRING);
            	logger.info("更新销售订单,头[{}]，行[{}]在途数量[{}], 收货数量[{}]，可发货数量[{}],订单数量[{}],是否已完成[{}].",new Object[] { 
            			supplierOrderItem.getOrderNo(), 
            			supplierOrderItem.getOrderItemNo(),
            			tempSupplierOrderItem.getOnwayQuantity(), 
            			tempSupplierOrderItem.getReceivedQuantity(), 
            			tempSupplierOrderItem.getDeliveryAmount(), 
            			supplierOrderItem.getQuantity(),
            			tempSupplierOrderItem.getFinishFlag() });
            	tempSupplierOrderItem.setUpdateTime(new Date());
            	this.supplierOrderItemService.modifyObj(tempSupplierOrderItem);
				
			}else {
				logger.info("订单未发送给供应商，销售订单不更新");
			}

            this.handleForDeliveryOrder(item.getVoucherDeliveryList());
        }
        
        if (CollectionUtils.isNotEmpty(removeList)) {
        	//清除没有处理的凭证行
			itemList.removeAll(removeList);
		}
    }

    @CacheEvict(value = { "voucher", "voucherItem" }, allEntries = true)
    @Transactional
	@Override
	public void addAll(List<Voucher> arg0) {
		if (CollectionUtils.isEmpty(arg0)) {
			return;
		}
		for (Voucher record : arg0) {
			this.voucherMapper.insertSelective(record);
		}
	}

    @CacheEvict(value = { "voucher", "voucherItem" }, allEntries = true)
	@Override
	public void deleteByExample(VoucherExample arg0) {
		Assert.isNotEmpty(arg0.getOredCriteria(), "删除的条件不能为空");
		this.voucherMapper.deleteByExample(arg0);
	}

}