package com.els.base.delivery.utils;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.text.MessageFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;

import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Base64Utils;

import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Constant;
import com.els.base.delivery.entity.DeliveryOrderItem;
import com.els.base.delivery.entity.DeliveryOrderPrint;
import com.els.base.file.entity.FileData;
import com.els.base.file.service.FileManagerFactory;
import com.els.base.logistic.entity.LogisticOrder;
import com.els.base.purchase.entity.LabelPrintRecordExtend;
import com.els.base.purchase.vo.KnMaterialQrcodeVO;
import com.els.base.utils.pdf.HtmlToPdfUtils;
import com.els.base.utils.template.BeetlTemplateUtils;
import com.google.zxing.BarcodeFormat;
import com.google.zxing.EncodeHintType;
import com.google.zxing.MultiFormatWriter;
import com.google.zxing.WriterException;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.pdf.BarcodePDF417;

/**
 * 送货单 打印工具
 * @author hzy
 *
 */
public abstract class DeliveryOrderPrintUtils {

	private static Logger logger = LoggerFactory.getLogger(DeliveryOrderPrintUtils.class);
	
	private static int WIDTH = 150;
	private static int HEIGHT = 64;

	private static String LENGTH80_WIDTH80 = "80*80";
	private static String LENGTH70_WIDTH45 = "70*45";


	public static FileData generateLogisticOrderPdf(LogisticOrder logisticOrder) throws Exception {
		String barCodeContent = MessageFormat.format("{0}#{1}#{2}#{3}", logisticOrder.getLogisticOrderNo(), logisticOrder.getDriverName(), logisticOrder.getIdCard(),logisticOrder.getCar());
		String qrCodeBase64 = getBase64QRCode(barCodeContent);
		
		Map<String, Object> params = new HashMap<>();
		params.put("logisticOrder", logisticOrder);
		params.put("qrCodeBase64", qrCodeBase64);
		String html = BeetlTemplateUtils.renderFromFile("logisticOrderA4.html", params);
		String filename = MessageFormat.format("物流单_A4{0}", DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm"));
		FileData pdfFile = createEmptyPDF(logisticOrder.getProjectId(), logisticOrder.getPurCompanyId(), filename);
		HtmlToPdfUtils.generatePdf(html, pdfFile.toFile(), HtmlToPdfUtils.PAGE_SIZE_A4, true, filename);  
		return pdfFile;
	}
	
	public static FileData generateDeliveryOrderPdf(DeliveryOrderPrint deliveryOrderPrint, String printType) throws WriterException, IOException, InterruptedException{
		//根据订单号，生成条形码，并设置到 打印实体里面
//		String barcodeBase64 = generateBarcodeBase64(deliveryOrderPrint.getDeliveryOrderNo());
		//根据订单号，送货单二维码 [S]ASN_NO[M]2018012300001[E]，并设置到 打印实体里面
		String barcodeBase64 = createPdf417QRCode("[S]ASN_NO[M]"+deliveryOrderPrint.getDeliveryOrderNo()+"[E]");
		deliveryOrderPrint.setScanPicPath(barcodeBase64);
		
		
		List<DeliveryOrderItem> items = deliveryOrderPrint.getItems();
		if(items != null) {
			for(DeliveryOrderItem item : items) {
				//临时用于送货单打印模版的工厂
				deliveryOrderPrint.setAttachment(item.getFactoryCode());
				break;
			}
		}

		Map<String, Object> params = new HashMap<>();
		params.put("deliveryOrder", deliveryOrderPrint);

		String filename = null;  //pdf文件名
		FileData pdfFile = null; //pdf文件
		String html = null;      //pdf生成需要的html
		
		switch (printType) {
		case "LetterA5":
			html = BeetlTemplateUtils.renderFromFile("deliveryOrderLetterA5.html", params);                                            //1 根据模板生成html
			filename =  "送货单_Letter_A5" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");                            //2 生成文件名
			pdfFile = createEmptyPDF(deliveryOrderPrint.getProjectId(), deliveryOrderPrint.getPurCompanyId(), filename); //3 生成空的pdf文件
			HtmlToPdfUtils.generatePdf(html, pdfFile.toFile(), 216f, 139.5f, filename);             //4根据html 生成pdf，并写入pdf文件里面
			break;
		case "Letter":
			html = BeetlTemplateUtils.renderFromFile("deliveryOrderLetter.html", params);                                            //1 根据模板生成html
			filename =  "送货单_Letter_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");                            //2 生成文件名
			pdfFile = createEmptyPDF(deliveryOrderPrint.getProjectId(), deliveryOrderPrint.getPurCompanyId(), filename); //3 生成空的pdf文件
			HtmlToPdfUtils.generatePdf(html, pdfFile.toFile(), "Letter", true, filename);             //4根据html 生成pdf，并写入pdf文件里面
			break;
		case "A4":
			html = BeetlTemplateUtils.renderFromFile("deliveryOrderA4.html", params);                                            //1 根据模板生成html
			filename =  "送货单_A4_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");                            //2 生成文件名
			pdfFile = createEmptyPDF(deliveryOrderPrint.getProjectId(), deliveryOrderPrint.getPurCompanyId(), filename); //3 生成空的pdf文件
			HtmlToPdfUtils.generatePdf(html, pdfFile.toFile(), HtmlToPdfUtils.PAGE_SIZE_A4, true, filename);             //4根据html 生成pdf，并写入pdf文件里面
			break;
			
		case "A5":
			html = BeetlTemplateUtils.renderFromFile("deliveryOrderA5.html", params);                                           //1 根据模板生成html
			filename =  "送货单_A5_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");                            //2 生成文件名
			pdfFile = createEmptyPDF(deliveryOrderPrint.getProjectId(), deliveryOrderPrint.getPurCompanyId(), filename); //3 生成空的pdf文件
			HtmlToPdfUtils.generatePdf(html, pdfFile.toFile(), HtmlToPdfUtils.PAGE_SIZE_A5, false, filename);             //4根据html 生成pdf，并写入pdf文件里面
			break;
			
		case "INVOICE":
			html = BeetlTemplateUtils.renderFromFile("InvoiceA4.html", params);                                           //1 根据模板生成html
			filename =  "INVOICE_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");                            //2 生成文件名
			pdfFile = createEmptyPDF(deliveryOrderPrint.getProjectId(), deliveryOrderPrint.getPurCompanyId(), filename); //3 生成空的pdf文件
			HtmlToPdfUtils.generatePdf(html, pdfFile.toFile(), HtmlToPdfUtils.PAGE_SIZE_A4, false, filename);             //4根据html 生成pdf，并写入pdf文件里面
			break;
			
		case "PACKINGLIST":
			html = BeetlTemplateUtils.renderFromFile("packinglistA4.html", params);                                           //1 根据模板生成html
			filename =  "PACKINGLIST_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");                            //2 生成文件名
			pdfFile = createEmptyPDF(deliveryOrderPrint.getProjectId(), deliveryOrderPrint.getPurCompanyId(), filename); //3 生成空的pdf文件
			HtmlToPdfUtils.generatePdf(html, pdfFile.toFile(), HtmlToPdfUtils.PAGE_SIZE_A4, false, filename);             //4根据html 生成pdf，并写入pdf文件里面
			break;

		default:
			throw new CommonException("不支持当前打印类型["+deliveryOrderPrint.getPrintSizeType()+"]");
		}

		return pdfFile;
	}
	
	public static FileData generateKnMaterialQrcodePdf(List<KnMaterialQrcodeVO> voList) throws WriterException, IOException, DocumentException, InterruptedException{
		
		Map<String, Object> params = new HashMap<>();
		params.put("printList", voList);

		String filename =  "物料二维码_MI_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");//2 生成文件名
		String html = BeetlTemplateUtils.renderFromFile("knMaterialQrcode72x38.html", params);//1 根据模板生成html
		FileData pdfFile = createEmptyPDF(voList.get(0).getProjectId(), voList.get(0).getPurCompanyId(), filename);//3 生成空的pdf文件
		//4根据html 生成pdf，并写入pdf文件里面
		HtmlToPdfUtils.generatePdf(html, pdfFile.toFile(), 72, 38, filename);

		return pdfFile;
	}

	/**
	 * 打印外箱二维码
	 * @param printList
	 * @throws WriterException
	 * @throws IOException
	 * @throws DocumentException
	 * @throws InterruptedException
	 */
	public static FileData generateDeliveryOrderOuterBoxPdf(List<LabelPrintRecordExtend> printList)
			throws WriterException, IOException, DocumentException, InterruptedException{
		logger.info("开始打印外箱");
		Map<String, Object> params = new HashMap<>();
		params.put("printList", printList);
		params.put("fontPath", DeliveryOrderPrint.DEFAULT_FONT_PATH);
		FileData pdfFile = null;
		if (printList.get(0).getPrintSize().equals(LENGTH80_WIDTH80)) {
			String html = BeetlTemplateUtils.renderFromFile("deliveryOrderOuterBox80x80.html", params);
//			logger.info("=================deliveryOrderOuterBox80x80.html================\n"+html+"\n=================html================\n");
			String filename = "外箱二维码_80x80_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");
			pdfFile = createEmptyPDF(printList.get(0).getProjectId(), printList.get(0).getPurCompanyId(),filename);
			File file =  pdfFile.toFile();
			HtmlToPdfUtils.generatePdf(html, file, 80, 80, filename);
		}else if (printList.get(0).getPrintSize().equals(LENGTH70_WIDTH45)) {
			String html = BeetlTemplateUtils.renderFromFile("deliveryOrderOuterBox70x45.html", params);
//			logger.info("=================deliveryOrderOuterBox70x45.html================\n"+html+"\n=================html================\n");
			String filename = "外箱二维码_70x45_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");
			pdfFile = createEmptyPDF(printList.get(0).getProjectId(), printList.get(0).getPurCompanyId(), filename);
			File file =  pdfFile.toFile();
			HtmlToPdfUtils.generatePdf(html, file, 70, 45, filename);
		}else{
			throw new CommonException("不支持当前打印尺寸["+printList.get(0).getPrintSize()+"]");
		}
		
		return pdfFile;
	}


	/**
	 * 打印内箱二维码
	 * @param printList
	 * @throws WriterException
	 * @throws IOException
	 * @throws DocumentException
	 * @throws InterruptedException
	 */
	public static FileData generateDeliveryOrderInnerBoxPdf(List<LabelPrintRecordExtend> printList)
			throws WriterException, IOException, InterruptedException{
		Map<String, Object> params = new HashMap<>();
		params.put("printList", printList);
		params.put("fontPath", DeliveryOrderPrint.DEFAULT_FONT_PATH);
		FileData pdfFile = null;
		if (printList.get(0).getPrintSize().equals(LENGTH80_WIDTH80)) {
			String html = BeetlTemplateUtils.renderFromFile("deliveryOrderInerBox80x80.html", params);
//			logger.info("=================deliveryOrderInerBox80x80.html================\n"+html+"\n=================html================\n");
			String filename = "内箱二维码_80x80_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");
			pdfFile = createEmptyPDF(printList.get(0).getProjectId(), printList.get(0).getPurCompanyId(), filename);
			File file =  pdfFile.toFile();
			HtmlToPdfUtils.generatePdf(html, file, 80, 80, filename);
		}else if (printList.get(0).getPrintSize().equals(LENGTH70_WIDTH45)) {
			String html = BeetlTemplateUtils.renderFromFile("deliveryOrderInerBox70x45.html", params);
//			logger.info("=================deliveryOrderInerBox70x45.html================\n"+html+"\n=================html================\n");
			String filename = "内箱二维码_70x45_" + DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm");
			pdfFile = createEmptyPDF(printList.get(0).getProjectId(), printList.get(0).getPurCompanyId(),filename);
			File file =  pdfFile.toFile();
			HtmlToPdfUtils.generatePdf(html, file, 70, 45, filename);
		}else{
			throw new CommonException("不支持当前打印尺寸["+printList.get(0).getPrintSize()+"]");
		}

		return pdfFile;
	}
	
	public static FileData createEmptyPDF(String projectId, String companyId, String filename) throws UnsupportedEncodingException, IOException{
		FileData pdfFile = new FileData();
		pdfFile.setProjectId(projectId);
		pdfFile.setCompanyId(companyId);
		
		pdfFile.setFileName(filename+".pdf");
		pdfFile.setFileSuffix("pdf");
		pdfFile.setIsEncrypt(String.valueOf(Constant.NO_INT));
		pdfFile.setExpiryDay(DateUtils.addMinutes(new Date(), 30));
		FileManagerFactory.getFileManager().createEmptyFile(pdfFile);
		return pdfFile;
	}

	/**
	 * 根据送货单号生成二维码
	 * @param deliveryOrderNo
	 * @return
	 * @throws WriterException
	 * @throws IOException
	 */
	public static String generateBarcodeBase64(String deliveryOrderNo) throws WriterException, IOException{
		BitMatrix bitMatrix = new MultiFormatWriter().encode(deliveryOrderNo, BarcodeFormat.CODE_128, WIDTH, HEIGHT, null);
		
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		MatrixToImageWriter.writeToStream(bitMatrix, "png", outputStream);
		outputStream.flush();
		outputStream.close();
		outputStream.toByteArray();
		byte[] base64Encode = Base64Utils.encode(outputStream.toByteArray());
		
		return new String(base64Encode);
	}

	/**
	 * 生成二维码并使用Base64编码
	 * @param content
	 * @throws WriterException 
	 * @throws IOException 
	 */

	public static String getBase64QRCode(String content) throws WriterException, IOException {
		MultiFormatWriter multiFormatWriter = new MultiFormatWriter();
		// 用于设置QR二维码参数
		Map<EncodeHintType, Object> hints = new HashMap<EncodeHintType, Object>();
		//设置二维码四周白色区域的大小
		hints.put(EncodeHintType.MARGIN,1);
		//设置二维码的容错性
		hints.put(EncodeHintType.ERROR_CORRECTION, ErrorCorrectionLevel.L);
		// 设置编码方式
		hints.put(EncodeHintType.CHARACTER_SET, "GBK");
		//画二维码
		BitMatrix bitMatrix = multiFormatWriter.encode(content, BarcodeFormat.QR_CODE, 400, 400, hints);
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		MatrixToImageWriter.writeToStream(bitMatrix, "png", outputStream);
		outputStream.flush();
		outputStream.close();
		outputStream.toByteArray();
		byte[] base64Encode = Base64Utils.encode(outputStream.toByteArray());
		return new String(base64Encode);
		
	}
	
	/**
	 * 生成二维条形码
	 * @param content
	 * @return
	 * @throws IOException
	 */
	public static String createPdf417QRCode(String content) throws IOException{
		BarcodePDF417 pdf = new BarcodePDF417();
		pdf.setText(content.getBytes("GBK"));
		Image pdfImage = pdf.createAwtImage(Color.BLACK, Color.white);
		BufferedImage img = new BufferedImage((int)pdfImage.getWidth(null),(int)pdfImage.getHeight(null),BufferedImage.TYPE_INT_RGB);
		Graphics g = img.getGraphics();
		g.drawImage(pdfImage, 0, 0, Color.white,null);
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		ImageIO.write(img, "png", outputStream);
		outputStream.flush();
		outputStream.close();
		byte[] base64Encode = Base64Utils.encode(outputStream.toByteArray());
		return new String(base64Encode);
	}
}
