package com.els.base.logistic.service.impl;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.delivery.entity.DeliveryOrder;
import com.els.base.delivery.entity.DeliveryOrderExample;
import com.els.base.delivery.entity.DeliveryOrderItem;
import com.els.base.delivery.entity.DeliveryOrderItemExample;
import com.els.base.delivery.service.DeliveryOrderItemService;
import com.els.base.delivery.service.DeliveryOrderService;
import com.els.base.delivery.utils.DeliveryStatusEnum;
import com.els.base.logistic.dao.LogisticOrderMapper;
import com.els.base.logistic.entity.LogisticOrder;
import com.els.base.logistic.entity.LogisticOrderExample;
import com.els.base.logistic.service.LogisticOrderService;
import com.els.base.purchase.entity.PurchaseOrder;
import com.els.base.purchase.service.PurchaseOrderService;

@Service("logisticOrderService")
public class LogisticOrderServiceImpl implements LogisticOrderService {
        private static Logger logger= LoggerFactory.getLogger(LogisticOrderServiceImpl.class);
    
	@Resource
	protected LogisticOrderMapper logisticOrderMapper;

	@Resource
	protected DeliveryOrderService deliveryOrderService;
	
	@Resource
	protected DeliveryOrderItemService deliveryOrderItemService;

	@Resource
	protected PurchaseOrderService purchaseOrderService;

	@CacheEvict(value = { "logisticOrder" }, allEntries = true)
	@Override
	public void addObj(LogisticOrder t) {
		this.logisticOrderMapper.insertSelective(t);
	}

	@CacheEvict(value = { "logisticOrder" }, allEntries = true)
	@Override
	public void deleteObjById(String id) {
		this.logisticOrderMapper.deleteByPrimaryKey(id);
	}

	@CacheEvict(value = { "logisticOrder" }, allEntries = true)
	@Override
	public void modifyObj(LogisticOrder t) {
		if (StringUtils.isBlank(t.getId())) {
			throw new NullPointerException("id 为空，无法更新");
		}
		this.logisticOrderMapper.updateByPrimaryKeySelective(t);
	}

	@Cacheable(value = "logisticOrder", keyGenerator="redisKeyGenerator")
	@Override
	public LogisticOrder queryObjById(String id) {
		return this.logisticOrderMapper.selectByPrimaryKey(id);
	}

    @Cacheable(value = "logisticOrder", keyGenerator="redisKeyGenerator")
	@Override
	public List<LogisticOrder> queryAllObjByExample(LogisticOrderExample example) {
		return this.logisticOrderMapper.selectByExample(example);
	}

	@Override
	public PageView<LogisticOrder> queryObjByPage(LogisticOrderExample example) {
		PageView<LogisticOrder> pageView = example.getPageView();
		pageView.setQueryResult(this.logisticOrderMapper.selectByExampleByPage(example));
		return pageView;
	}


	@Transactional
    @CacheEvict(value = { "logisticOrder" }, allEntries = true)
	@Override
	public void deleteLogisticOrder(List<String> logisticOrderIdList) {
		if (CollectionUtils.isEmpty(logisticOrderIdList)) {
			return;
		}

		for (String id : logisticOrderIdList) {

			//判断一下
			LogisticOrder logisticOrder = this.logisticOrderMapper.selectByPrimaryKey(id);
			if (logisticOrder==null) {
				continue;
			}

			//根据物流单号查询送货单状态
			DeliveryOrderExample example = new DeliveryOrderExample();
			DeliveryOrderExample.Criteria criteria = example.createCriteria();
			criteria.andLogisticOrderNoEqualTo(logisticOrder.getLogisticOrderNo());
			List<DeliveryOrder> list = deliveryOrderService.queryAllObjByExample(example);

			if(CollectionUtils.isEmpty(list)){
				continue;
			}
			

			for (DeliveryOrder deliveryOrder : list) {
				//任何一个送货单中的任何一个行项目已经收货，则不能进行删除
				if(DeliveryStatusEnum.ALL_RECEIVED.getValue().equals(deliveryOrder.getDeliveryStatus())
						|| DeliveryStatusEnum.HK_RECEIVED.getValue().equals(deliveryOrder.getDeliveryStatus())){
					throw new CommonException("该物流单中的送货单"+deliveryOrder.getDeliveryOrderNo()+"已经收货，不可删除，谢谢！", "goods_have_arrived",deliveryOrder.getDeliveryOrderNo());
				}

				//删除物流单后，送货单被撤销发货，状态变为未发货
				deliveryOrder.setDeliveryStatus(DeliveryStatusEnum.UN_RECEIVED.getValue());
				deliveryOrder.setLogisticOrderNo("");
				deliveryOrderService.modifyObj(deliveryOrder);
				
				DeliveryOrderItemExample deliveryOrderItemExample = new DeliveryOrderItemExample();
				deliveryOrderItemExample.createCriteria().andDeliveryOrderIdEqualTo(deliveryOrder.getId());
				DeliveryOrderItem tempItem = new DeliveryOrderItem();
				tempItem.setDeliveryStatus(DeliveryStatusEnum.UN_RECEIVED.getValue());
				this.deliveryOrderItemService.updateDeliveryOrderItem(tempItem, deliveryOrderItemExample);
			}
		
		}

		//删除该物流单
		LogisticOrderExample example = new LogisticOrderExample();
		LogisticOrderExample.Criteria criteria = example.createCriteria();
		criteria.andIdIn(logisticOrderIdList);
		logisticOrderMapper.deleteByExample(example);
	}

	@Override
	public LogisticOrder createLogisticOrderData(List<String> deliveryOrderIdList) {
		LogisticOrder logisticOrder = new LogisticOrder();
		DeliveryOrderExample example = new DeliveryOrderExample();
		example.createCriteria().andIdIn(deliveryOrderIdList);
		List<DeliveryOrder> items = this.deliveryOrderService.queryAllObjByExample(example);
		if(CollectionUtils.isEmpty(items)){
			throw new CommonException("送货单数据不能为空", "base_canot_be_null", "送货单数据");
		}
		String purCompanyCode=items.get(0).getPurCompanyCode();
		String purCompanyName=items.get(0).getPurCompanyName();
		logisticOrder.setPurCompanyCode(purCompanyCode);
		logisticOrder.setPurCompanyName(purCompanyName);
		logisticOrder.setItems(items);
		return logisticOrder;
	}
	@CacheEvict(value = { "logisticOrder" }, allEntries = true)
	@Transactional
	@Override
	public void saveLogisticOrder(LogisticOrder logisticOrder) {
		List<DeliveryOrder> items = logisticOrder.getItems();
		if (CollectionUtils.isEmpty(items)) {
			throw new CommonException("送货单数据不能为空", "base_canot_be_null", "送货单数据");
		}
		for (DeliveryOrder deliveryOrder : items) {
			deliveryOrder = this.deliveryOrderService.queryObjById(deliveryOrder.getId());
			deliveryOrder.setLogisticOrderNo(logisticOrder.getLogisticOrderNo());
			logger.info(deliveryOrder.getLogisticOrderNo()+"-----------logisticOrder.getLogisticOrderNo()--------");
			if (deliveryOrder == null) {
				throw new CommonException("没有找到该发货单", "do_not_exists", "发货单");
			}
			//已经全部发货
			if ((deliveryOrder.getDeliveryStatus() != null)
	                && (!DeliveryStatusEnum.UN_RECEIVED.getValue().equals(deliveryOrder.getDeliveryStatus()))) {
				this.deliveryOrderService.modifyObj(deliveryOrder);
	        }else{
	        	this.deliveryOrderService.sendDelivery(deliveryOrder);
	        }
			
		}
		String purCompanyCode=items.get(0).getPurCompanyCode();
		String purCompanyName=items.get(0).getPurCompanyName();
		logisticOrder.setPurCompanyCode(purCompanyCode);
		logisticOrder.setPurCompanyName(purCompanyName);
		this.logisticOrderMapper.insert(logisticOrder);
	}
	@Cacheable(value = "logisticOrder", keyGenerator="redisKeyGenerator")
	@Override
	public LogisticOrder queryLogisticOrder(String queryNo, String type) {
		if (type.equals("E")) {// 物流
			LogisticOrderExample example = new LogisticOrderExample();
			example.createCriteria().andLogisticOrderNoEqualTo(queryNo);
			List<LogisticOrder> logisticOrderList = this.logisticOrderMapper.selectByExample(example);
			if (CollectionUtils.isEmpty(logisticOrderList)) {
				return null;
			} else {
				return logisticOrderList.get(0);
			}
		} else if (type.equals("D")) {// 发货
			DeliveryOrderExample example = new DeliveryOrderExample();
			example.createCriteria().andDeliveryOrderNoEqualTo(queryNo);
			List<DeliveryOrder> deliveryOrderList = this.deliveryOrderService.queryAllObjByExample(example);
			if (CollectionUtils.isEmpty(deliveryOrderList)) {
				return null;
			} else {
				DeliveryOrder deliveryOrder = deliveryOrderList.get(0);
				LogisticOrder LogisticOrder = new LogisticOrder();
				LogisticOrder.setCar(deliveryOrder.getCar());
				LogisticOrder.setDriverName(deliveryOrder.getDriverName());
				LogisticOrder.setIdCard(deliveryOrder.getIdCard());
				LogisticOrder.setExpectDeliveryTime(deliveryOrder.getExpectDeliveryTime());
				LogisticOrder.setExpectArriveTime(deliveryOrder.getExpectArriveTime());
				return LogisticOrder;
			}
		}
		return null;
	}
	@Cacheable(value = "logisticOrder", keyGenerator="redisKeyGenerator")
	@Override
	public List<PurchaseOrder> queryPurchaseOrder(String queryNo, String type) {
		List<PurchaseOrder> purchaseOrderList = new ArrayList<PurchaseOrder>();
		if (type.equals("E")) {// 物流
			LogisticOrderExample example = new LogisticOrderExample();
			example.createCriteria().andLogisticOrderNoEqualTo(queryNo);
			List<LogisticOrder> logisticOrderList = this.logisticOrderMapper.selectByExample(example);
			if (CollectionUtils.isEmpty(logisticOrderList)) {
				return purchaseOrderList;
			} else {
				DeliveryOrderExample example2 = new DeliveryOrderExample();
				example2.createCriteria().andLogisticOrderNoEqualTo(logisticOrderList.get(0).getLogisticOrderNo());
				List<DeliveryOrder> deliveryOrderList = this.deliveryOrderService.queryAllObjByExample(example2);
				if (CollectionUtils.isEmpty(logisticOrderList)) {
					return purchaseOrderList;
				} else {
					for (DeliveryOrder deliveryOrder : deliveryOrderList) {
						PurchaseOrder purchaseOrder = purchaseOrderService
								.queryObjById(deliveryOrder.getPurchaseOrderId());
						if (purchaseOrder != null) {
							purchaseOrderList.add(purchaseOrder);
						}
					}
				}
			}
		} else if (type.equals("D")) {// 发货
			DeliveryOrderExample example = new DeliveryOrderExample();
			example.createCriteria().andDeliveryOrderNoEqualTo(queryNo);
			List<DeliveryOrder> deliveryOrderList = this.deliveryOrderService.queryAllObjByExample(example);
			if (CollectionUtils.isEmpty(deliveryOrderList)) {
				return purchaseOrderList;
			} else {
				DeliveryOrder deliveryOrder = deliveryOrderList.get(0);
				PurchaseOrder purchaseOrder = purchaseOrderService.queryObjById(deliveryOrder.getPurchaseOrderId());
				if (purchaseOrder != null) {
					purchaseOrderList.add(purchaseOrder);
				}
			}
		}
		return purchaseOrderList;
	}

	@CacheEvict(value = { "logisticOrder" }, allEntries = true)
	@Transactional
	@Override
	public void addAll(List<LogisticOrder> arg0) {
		if (CollectionUtils.isEmpty(arg0)) {
			return;
		}
		for (LogisticOrder record : arg0) {
			this.logisticOrderMapper.insertSelective(record);
		}
	}

	@CacheEvict(value = { "logisticOrder" }, allEntries = true)
	@Override
	public void deleteByExample(LogisticOrderExample arg0) {
		Assert.isNotEmpty(arg0.getOredCriteria(), "删除的条件不能为空");
		this.logisticOrderMapper.deleteByExample(arg0);
	}

}