package com.els.base.logistic.web.controller;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.file.entity.FileData;
import com.els.base.delivery.entity.DeliveryOrder;
import com.els.base.delivery.entity.DeliveryOrderExample;
import com.els.base.logistic.entity.LogisticOrder;
import com.els.base.logistic.entity.LogisticOrderExample;
import com.els.base.delivery.service.DeliveryOrderService;
import com.els.base.logistic.service.LogisticOrderService;
import com.els.base.delivery.utils.DeliveryOrderPrintUtils;
import com.els.base.purchase.utils.SysUtil;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@Api(value = "物流单管理")
@Controller
@RequestMapping("logisticOrder")
public class LogisticOrderController {
    @Resource
    protected LogisticOrderService logisticOrderService;

    @Resource
    protected DeliveryOrderService deliveryOrderService;

    @ApiOperation(httpMethod = "POST", value = "创建物流表：入参（整个实体类）")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody LogisticOrder logisticOrder) {
        logisticOrder.setProjectId(ProjectUtils.getProjectId());
        logisticOrder.setCompanyId(CompanyUtils.currentCompanyId());
        logisticOrder.setUserId(SpringSecurityUtils.getLoginUserId());
        logisticOrder.setUserName(SpringSecurityUtils.getLoginUserName());
        this.logisticOrderService.saveLogisticOrder(logisticOrder);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "编辑物流表：入参（整个实体类）")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody LogisticOrder logisticOrder) {
        if (StringUtils.isBlank(logisticOrder.getId())) {
            throw new CommonException("id为空，操作失败", "id_is_blank");
        }
        this.logisticOrderService.modifyObj(logisticOrder);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "物流单管理-删除物流表。入参（物流id数组）")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(@RequestBody List<String> logisticOrderIdList) {
        if (CollectionUtils.isEmpty(logisticOrderIdList)) {
            throw new CommonException("物流ID列表不能为空", "base_canot_be_null","物流ID列表");
        }
        this.logisticOrderService.deleteLogisticOrder(logisticOrderIdList);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "物流单管理，物流单列表。查询物流表:查询条件（物流单号）")
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<LogisticOrder>> findByPage(
            @ApiParam(value = "所在页", defaultValue = "0") @RequestParam(defaultValue = "0") int pageNo,
            @ApiParam(value = "每页数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize,
            @ApiParam(value = "查询条件,属性名请参考 LogisticOrder") @RequestBody(required = false) QueryParamWapper wapper) {
        LogisticOrderExample example = new LogisticOrderExample();
        example.setPageView(new PageView<LogisticOrder>(pageNo, pageSize));
        String companyId = CompanyUtils.currentCompanyId();
        String projectId = ProjectUtils.getProjectId();
        LogisticOrderExample.Criteria criteria = example.createCriteria();
        if (wapper != null) {
            CriteriaUtils.addCriterion(criteria, wapper);
        }
        criteria.andProjectIdEqualTo(projectId);
        criteria.andCompanyIdEqualTo(companyId);
        example.setOrderByClause("CREATE_TIME DESC");
        PageView<LogisticOrder> pageData = this.logisticOrderService.queryObjByPage(example);
        return ResponseResult.success(pageData);
    }

    @ApiOperation(httpMethod = "POST", value = "生成物流单数据：入参（发货单头id数组）")
    @RequestMapping("service/createLogisticOrderData")
    @ResponseBody
    public ResponseResult<LogisticOrder> createLogisticOrderData(@RequestBody List<String> deliveryOrderIdList) {
        if (CollectionUtils.isEmpty(deliveryOrderIdList)) {
            throw new CommonException("没有查询到数据", "not_query_to_data");
        }
        LogisticOrder logisticOrder = this.logisticOrderService.createLogisticOrderData(deliveryOrderIdList);
        logisticOrder.setProjectId(ProjectUtils.getProjectId());
        logisticOrder.setCompanyId(CompanyUtils.currentCompany().getId());
        logisticOrder.setCompanyName(CompanyUtils.currentCompany().getCompanyFullName());
        logisticOrder.setUserId(SpringSecurityUtils.getLoginUserId());
        logisticOrder.setUserName(SpringSecurityUtils.getLoginUserName());
        logisticOrder.setLogisticOrderNo(SysUtil.getNextLogisticsNumber());
        logisticOrder.setCreateTime(new Date());
        return ResponseResult.success(logisticOrder);
    }

    @ApiOperation(httpMethod = "POST", value = "通过物流单号查询送货单信息：入参（物流单号）")
    @RequestMapping("service/findDeliveryOrderByLogisticOrderNo")
    @ResponseBody
    public ResponseResult<List<DeliveryOrder>> findDeliveryOrderByLogisticOrderNo(
            @ApiParam(value = "物流单号", defaultValue = "") @RequestParam(defaultValue = "") String logisticOrderNo) {
        if (StringUtils.isBlank(logisticOrderNo)) {
            throw new CommonException("物流单号不能为空", "base_canot_be_null","物流单号");
        }
        List<DeliveryOrder> items = new ArrayList<DeliveryOrder>();
        DeliveryOrderExample example = new DeliveryOrderExample();
        example.createCriteria().andLogisticOrderNoEqualTo(logisticOrderNo);
        items = this.deliveryOrderService.queryAllObjByExample(example);
        return ResponseResult.success(items);
    }
    
    @ApiOperation(httpMethod = "POST", value = "物流打印:入参:LogisticOrder对象")
    @RequestMapping("service/print")
    @ResponseBody
    public ResponseResult<FileData> print(@RequestBody LogisticOrder logisticOrder) throws Exception {
    	logisticOrder.setProjectId(ProjectUtils.getProjectId());
		logisticOrder.setCompanyId(CompanyUtils.currentCompanyId());
 		FileData fileData = DeliveryOrderPrintUtils.generateLogisticOrderPdf(logisticOrder);
        return ResponseResult.success(fileData);
    }
}