package com.els.base.orderchange.service.impl;

import java.math.BigDecimal;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.core.entity.PageView;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.orderchange.dao.PurchaseOrderChangeMapper;
import com.els.base.orderchange.entity.PurchaseOrderChange;
import com.els.base.orderchange.entity.PurchaseOrderChangeExample;
import com.els.base.orderchange.service.PurchaseOrderChangeService;
import com.els.base.purchase.entity.PurchaseOrder;
import com.els.base.purchase.entity.PurchaseOrderExample;
import com.els.base.purchase.entity.PurchaseOrderItem;
import com.els.base.purchase.entity.PurchaseOrderItemExample;
import com.els.base.purchase.service.PurchaseOrderItemService;
import com.els.base.purchase.service.PurchaseOrderService;
import com.els.base.purchase.utils.SysUtil;
import com.els.base.supperorder.entity.SupplierOrder;
import com.els.base.supperorder.entity.SupplierOrderExample;
import com.els.base.supperorder.entity.SupplierOrderItem;
import com.els.base.supperorder.entity.SupplierOrderItemExample;
import com.els.base.supperorder.service.SupplierOrderItemService;
import com.els.base.supperorder.service.SupplierOrderService;

@Service("defaultPurchaseOrderChangeService")
public class PurchaseOrderChangeServiceImpl implements PurchaseOrderChangeService {

    Logger logger = LoggerFactory.getLogger(PurchaseOrderChangeServiceImpl.class);
    @Resource
    protected PurchaseOrderChangeMapper purchaseOrderChangeMapper;
    @Resource
    protected PurchaseOrderItemService purchaseOrderItemService;
    @Resource
    protected SupplierOrderItemService supplierOrderItemService;
    @Resource
    protected PurchaseOrderService purchaseOrderService;
    @Resource
    protected SupplierOrderService supplierOrderService;

    @Transactional
    @CacheEvict(value = { "purchaseOrderChange" }, allEntries = true)
    @Override
    public void deleteByExample(PurchaseOrderChangeExample example) {
        Assert.isNotEmpty(example.getOredCriteria(),"不能进行全表删除！");
        this.purchaseOrderChangeMapper.deleteByExample(example);
    }

    @Transactional
    @CacheEvict(value = { "purchaseOrderChange" }, allEntries = true)
    @Override
    public int updateByExampleSelective(PurchaseOrderChange record, PurchaseOrderChangeExample example) {
        Assert.isNotNull(record,"更新信息不能为空！");
        List<PurchaseOrderChangeExample.Criteria> oredCriteria = example.getOredCriteria();
        Assert.isNotEmpty(oredCriteria,"不能进行全表更新！");
        return this.purchaseOrderChangeMapper.updateByExampleSelective(record,example);
    }

    @CacheEvict(value = { "purchaseOrderChange" }, allEntries = true)
    @Override
    public void addObj(PurchaseOrderChange t) {
        this.purchaseOrderChangeMapper.insertSelective(t);
    }

    @CacheEvict(value = { "purchaseOrderChange" }, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.purchaseOrderChangeMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = { "purchaseOrderChange" }, allEntries = true)
    @Override
    public void modifyObj(PurchaseOrderChange t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.purchaseOrderChangeMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "purchaseOrderChange", keyGenerator = "redisKeyGenerator")
    @Override
    public PurchaseOrderChange queryObjById(String id) {
        return this.purchaseOrderChangeMapper.selectByPrimaryKey(id);
    }

    //@Cacheable(value = "purchaseOrderChange", key = "'PurchaseOrderChangeService_' + #root.methodName + '_'+ T(com.els.base.utils.encryption.Md5Utils).md5Object(#example)")
    @Cacheable(value = "purchaseOrderChange", keyGenerator = "redisKeyGenerator")
    @Override
    public List<PurchaseOrderChange> queryAllObjByExample(PurchaseOrderChangeExample example) {
        return this.purchaseOrderChangeMapper.selectByExample(example);
    }

    //@Cacheable(value = "purchaseOrderChange", key = "'PurchaseOrderChangeService_' + #root.methodName + '_'+ T(com.els.base.utils.encryption.Md5Utils).md5Object(#example)")
    @Cacheable(value = "purchaseOrderChange", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<PurchaseOrderChange> queryObjByPage(PurchaseOrderChangeExample example) {
        PageView<PurchaseOrderChange> pageView = example.getPageView();
        List<PurchaseOrderChange> selectByExampleByPage = this.purchaseOrderChangeMapper.selectByExampleByPage(example);
        pageView.setQueryResult(selectByExampleByPage);
        return pageView;
    }

    /**
     * 根据SAP的变更接口，修改采购订单
     */
    @CacheEvict(value = { "purchaseOrderChange" }, allEntries = true)
    @Override
    public void importFromSap(List<PurchaseOrderChange> purchaseOrderChangeList) {
        if (CollectionUtils.isEmpty(purchaseOrderChangeList)) {
            return;
        }
        for (PurchaseOrderChange purchaseOrderChange : purchaseOrderChangeList) {

            BigDecimal cancellApplyNum1 = purchaseOrderChange.getCancellApplyNum();
            BigDecimal quantity1 = purchaseOrderChange.getQuantity();
            BigDecimal cancellApplyNum = cancellApplyNum1 == null ? BigDecimal.ZERO : cancellApplyNum1;
            BigDecimal quantity = quantity1 == null ? BigDecimal.ZERO : quantity1;
            String orderNo = purchaseOrderChange.getOrderNo();
            String orderItemNo = purchaseOrderChange.getOrderItemNo();
            logger.info("取消申请数量为：{}",cancellApplyNum);
            logger.info("变更数量为：{}",quantity);
            logger.info("采购订单号为：{}",orderNo);
            logger.info("采购订单行号为：{}",orderItemNo);

            // 如果存在相同的变更申请，则改变已经存在的数据的状态
            PurchaseOrderChangeExample purchaseOrderChangeExample = new PurchaseOrderChangeExample();
            purchaseOrderChangeExample.createCriteria()
                    .andOrderNoEqualTo(orderNo)
                    .andOrderItemNoEqualTo(orderItemNo);
            List<PurchaseOrderChange> oldChangeList = this.purchaseOrderChangeMapper.selectByExample(purchaseOrderChangeExample);
            int changeTime = 0;
            if (CollectionUtils.isNotEmpty(oldChangeList)) {
            	PurchaseOrderChange temp = new PurchaseOrderChange();
            	temp.setConfirmStatus("2");
            	temp.setStatus("CLOSE");
            	this.purchaseOrderChangeMapper.updateByExampleSelective(temp, purchaseOrderChangeExample);
            	
            	for( PurchaseOrderChange oldChange: oldChangeList){
            		changeTime = oldChange.getChangeTimes() != null && oldChange.getChangeTimes() > changeTime ? oldChange.getChangeTimes() : changeTime;
            	}
            }
            
            logger.info("beforeChange :采购订单行号setChangeTimes：{}",changeTime);
            if (this.isChangeDateOrQuantity(purchaseOrderChange)) {
            	logger.info("changeing :采购订单行号setChangeTimes：{}",changeTime);
            	purchaseOrderChange.setChangeTimes(++changeTime);
            }else{
            	purchaseOrderChange.setChangeTimes(changeTime);
            }
            logger.info("采购订单行号setChangeTimes：{}",purchaseOrderChange.getChangeTimes());


            // 代表新的变更申请
            if (purchaseOrderChange.getStatus().equalsIgnoreCase("NEW"))
                this.newPurchaseOrder(purchaseOrderChange, cancellApplyNum, orderNo, orderItemNo);
            // 代表是根据原变更申请做的更改
            else if (purchaseOrderChange.getStatus().equalsIgnoreCase("CLOSE")) {
                this.updatePurchaseChangeOrder(purchaseOrderChange, cancellApplyNum, quantity, orderNo, orderItemNo);
            }
        }
    }

    /**
     * 看看是否有变更交期或者数量
     * @param purchaseOrderChange
     * @return
     */
    private boolean isChangeDateOrQuantity(PurchaseOrderChange purchaseOrderChange) {
    	logger.info("交期：before:[{}],now[{}], 数量:[{}],[{}]", purchaseOrderChange.getBeforeDeliveryDate(), purchaseOrderChange.getDeliveryDate(), purchaseOrderChange.getBeforeDeliveryNum(), purchaseOrderChange.getQuantity());
    	
    	if (purchaseOrderChange.getBeforeDeliveryDate() != null
    			&& purchaseOrderChange.getDeliveryDate() != null
    			&& purchaseOrderChange.getBeforeDeliveryDate().getTime() != purchaseOrderChange.getDeliveryDate().getTime()) {
    		logger.info("交期：before:[{}],now[{}]", purchaseOrderChange.getBeforeDeliveryDate(), purchaseOrderChange.getDeliveryDate());
    		return true;
		}
    	
    	if (purchaseOrderChange.getBeforeDeliveryNum() != null
    			&& purchaseOrderChange.getQuantity() != null
    			&& purchaseOrderChange.getBeforeDeliveryNum().compareTo(purchaseOrderChange.getQuantity()) !=0) {
    		logger.info("交期 数量:[{}],[{}]", purchaseOrderChange.getBeforeDeliveryNum(), purchaseOrderChange.getQuantity());
			return true;
		}
    	
		return false;
	}

	/**
     * 根据原变更申请做的更改
     * @param purchaseOrderChange
     * @param cancellApplyNum
     * @param quantity
     * @param orderNo
     * @param orderItemNo
     */
    private void updatePurchaseChangeOrder(PurchaseOrderChange purchaseOrderChange, BigDecimal cancellApplyNum, BigDecimal quantity, String orderNo, String orderItemNo) {

        SupplierOrderItem supplierOrderItem = querySupplierData(orderNo, orderItemNo);
        // 修改销售订单行的值
        if (supplierOrderItem != null) {
            BigDecimal canDeliveryAmount = SysUtil.getCanDeliveryAmount(quantity,
                    supplierOrderItem.getOnwayQuantity(), supplierOrderItem.getReceivedQuantity(),
                    cancellApplyNum);// 获取可发货数量
            SupplierOrderItem temp = new SupplierOrderItem();
            temp.setId(supplierOrderItem.getId());
            temp.setFreezeQuantity(cancellApplyNum);
            temp.setQuantity(quantity);
            temp.setDeliveryAmount(canDeliveryAmount);
            //
            temp.setOrderValue(purchaseOrderChange.getAmount());
            temp.setPriceTaxTotal(purchaseOrderChange.getItemPriceTaxTotal().toString());

            //自动设置为已完成
            temp.setFinishFlag(quantity.compareTo(supplierOrderItem.getReceivedQuantity()) > 0 ? Constant.NO_STRING : Constant.YES_STRING);

            // 更新供应商界面
            temp.setDeliveredDate(purchaseOrderChange.getDeliveryDate());
            temp.setQuantity(purchaseOrderChange.getQuantity());

            this.supplierOrderItemService.modifyObj(temp);
        }

        SupplierOrderExample supplierOrderExample = new SupplierOrderExample();
        supplierOrderExample.createCriteria()
                .andOrderNoEqualTo(orderNo);
        // 修改销售订单头的值，如果所有的行都已经交货已完成，则头也标识交货已完成
        List<SupplierOrder> supplierOrderList = this.supplierOrderService.queryAllObjByExample(supplierOrderExample);
        if (CollectionUtils.isNotEmpty(supplierOrderList)) {
            SupplierOrder supplierOrder = supplierOrderList.get(0);
            SupplierOrder temp = new SupplierOrder();
            boolean boo = queryOrderItemIsFinish(orderNo, "sup");
            if (boo == false) {
                temp.setDeliveryStatus(2);
            }
            temp.setId(supplierOrder.getId());
            temp.setTotalAmount(new BigDecimal(purchaseOrderChange.getOrderHeadAmount()));
            temp.setPriceTaxTotal(purchaseOrderChange.getHeadPriceTaxTotal().toString());
            this.supplierOrderService.modifyObj(temp);
        }

        // 修改采购订单行的值
        PurchaseOrderItem purchaseOrderItem = queryPurchaseData(orderNo, orderItemNo);
        if (purchaseOrderItem != null) {
            BigDecimal canDeliveryAmount = SysUtil.getCanDeliveryAmount(quantity,
                    purchaseOrderItem.getOnwayQuantity(), purchaseOrderItem.getReceivedQuantity(),
                    cancellApplyNum);// 获取可发货数量
            PurchaseOrderItem temp = new PurchaseOrderItem();
            temp.setFinishFlag(quantity.compareTo(purchaseOrderItem.getReceivedQuantity()) > 0 ? Constant.NO_STRING : Constant.YES_STRING);
            temp.setId(purchaseOrderItem.getId());
            temp.setFreezeQuantity(cancellApplyNum);
            temp.setQuantity(quantity);
            temp.setDeliveryAmount(canDeliveryAmount);
            temp.setOrderValue(purchaseOrderChange.getAmount());
            temp.setPriceTaxTotal(purchaseOrderChange.getItemPriceTaxTotal().toString());
            this.purchaseOrderItemService.modifyObj(temp);
        }

        // 修改采购订单头的值，如果所有的行都已经交货已完成，则头也标识交货已完成
        PurchaseOrderExample purchaseOrderExample = new PurchaseOrderExample();
        purchaseOrderExample.createCriteria().andOrderNoEqualTo(orderNo);
        List<PurchaseOrder> purchaseOrderList = this.purchaseOrderService.queryAllObjByExample(purchaseOrderExample);
        if (CollectionUtils.isNotEmpty(purchaseOrderList)) {
            PurchaseOrder purchaseOrder = purchaseOrderList.get(0);
            PurchaseOrder temp = new PurchaseOrder();
            boolean boo = queryOrderItemIsFinish(orderNo, "pur");
            if (boo == false) {
                temp.setDeliveryStatus(2);
            }
            temp.setId(purchaseOrder.getId());
            temp.setTaxAmount(new BigDecimal(purchaseOrderChange.getOrderHeadAmount()));
            temp.setPriceTaxTotal(purchaseOrderChange.getHeadPriceTaxTotal().toString());
            this.purchaseOrderService.modifyObj(temp);
        }
    }

    /**
     * 新的变更申请
     * @param purchaseOrderChange
     * @param cancellApplyNum
     * @param orderNo
     * @param orderItemNo
     */
    private void newPurchaseOrder(PurchaseOrderChange purchaseOrderChange, BigDecimal cancellApplyNum, String orderNo, String orderItemNo) {
        //if (purchaseOrderChange != null) {

        if ( StringUtils.isNotBlank(purchaseOrderChange.getId())) {
            this.purchaseOrderChangeMapper.updateByPrimaryKey(purchaseOrderChange);
        }else{
            this.purchaseOrderChangeMapper.insertSelective(purchaseOrderChange);
        }
        //}

        //变更数量会影响可发货数量
        //变更供应商行
        SupplierOrderItem supplierOrderItem = querySupplierData(orderNo, orderItemNo);
        if (supplierOrderItem != null) {
            BigDecimal canDeliveryAmount = SysUtil.getCanDeliveryAmount(supplierOrderItem.getQuantity(),
                    supplierOrderItem.getOnwayQuantity(), supplierOrderItem.getReceivedQuantity(),
                    cancellApplyNum);// 获取可发货数量
            SupplierOrderItem temp = new SupplierOrderItem();
            temp.setId(supplierOrderItem.getId());
            temp.setFreezeQuantity(cancellApplyNum);
            temp.setDeliveryAmount(canDeliveryAmount);

            //交期
            temp.setDeliveredDate(purchaseOrderChange.getDeliveryDate());
            temp.setQuantity(purchaseOrderChange.getQuantity());

            this.supplierOrderItemService.modifyObj(temp);
        }

        //变更采购行
        PurchaseOrderItem purchaseOrderItem = queryPurchaseData(orderNo, orderItemNo);
        if (purchaseOrderItem != null) {
            BigDecimal canDeliveryAmount = SysUtil.getCanDeliveryAmount(purchaseOrderItem.getQuantity(),
                    purchaseOrderItem.getOnwayQuantity(), purchaseOrderItem.getReceivedQuantity(),
                    cancellApplyNum);// 获取可发货数量
            PurchaseOrderItem temp = new PurchaseOrderItem();
            temp.setId(purchaseOrderItem.getId());
            temp.setFreezeQuantity(cancellApplyNum);
            temp.setDeliveryAmount(canDeliveryAmount);
            this.purchaseOrderItemService.modifyObj(temp);
        }
    }

    private boolean queryOrderItemIsFinish(String orderNo, String str) {
        if (str.equals("sup")) {
            SupplierOrderItemExample supplierOrderItemExample = new SupplierOrderItemExample();
            supplierOrderItemExample.createCriteria().andOrderNoEqualTo(orderNo).andFinishFlagEqualTo("N");
            List<SupplierOrderItem> queryAllObjByExample = this.supplierOrderItemService
                    .queryAllObjByExample(supplierOrderItemExample);
            if (CollectionUtils.isNotEmpty(queryAllObjByExample)) {
                return queryAllObjByExample.size() > 0;
            }
        } else if (str.equals("pur")) {
            PurchaseOrderItemExample purchaseOrderItemExample = new PurchaseOrderItemExample();
            purchaseOrderItemExample.createCriteria().andOrderNoEqualTo(orderNo).andFinishFlagEqualTo("N");
            List<PurchaseOrderItem> purchaseOrderItemList = this.purchaseOrderItemService
                    .queryAllObjByExample(purchaseOrderItemExample);
            if (CollectionUtils.isNotEmpty(purchaseOrderItemList)) {
                return purchaseOrderItemList.size() > 0;
            }
        }
        return false;
    }

    // private long getCanDeliveryAmount(BigDecimal quantity, BigDecimal onwayQuantity,
    // BigDecimal receivedQuantity, BigDecimal cancellApplyNum) {
    // long canDeliveryAmount = 0;
    // quantity = quantity == null ? BigDecimal.ZERO : quantity;// 采购订单数量
    // onwayQuantity = onwayQuantity == null ? BigDecimal.ZERO : onwayQuantity;// 在途数量
    // receivedQuantity = receivedQuantity == null ? BigDecimal.ZERO : receivedQuantity;// 收货数量
    // cancellApplyNum = cancellApplyNum == null ? BigDecimal.ZERO : cancellApplyNum;// 冻结数量
    // canDeliveryAmount = quantity.subtract(onwayQuantity).subtract(receivedQuantity)
    // .subtract(cancellApplyNum).longValue();
    //
    // return canDeliveryAmount;
    // }

    private SupplierOrderItem querySupplierData(String orderNo, String orderItemNo) {
        SupplierOrderItemExample supplierOrderItemExample = new SupplierOrderItemExample();
        supplierOrderItemExample.createCriteria()
                .andOrderNoEqualTo(orderNo)
                .andOrderItemNoEqualTo(orderItemNo);
        List<SupplierOrderItem> supplierOrderItemList = this.supplierOrderItemService.queryAllObjByExample(supplierOrderItemExample);
        if (CollectionUtils.isNotEmpty(supplierOrderItemList)) {
            SupplierOrderItem supplierOrderItem = supplierOrderItemList.get(0);
            return supplierOrderItem;
        } else {
            return null;
        }
    }

    private PurchaseOrderItem queryPurchaseData(String orderNo, String orderItemNo) {
        PurchaseOrderItemExample purchaseOrderItemExample = new PurchaseOrderItemExample();
        purchaseOrderItemExample.createCriteria().andOrderNoEqualTo(orderNo)
                .andOrderItemNoEqualTo(orderItemNo);
        List<PurchaseOrderItem> purchaseOrderItemList = this.purchaseOrderItemService
                .queryAllObjByExample(purchaseOrderItemExample);
        if (CollectionUtils.isNotEmpty(purchaseOrderItemList)) {
            PurchaseOrderItem purchaseOrderItem = purchaseOrderItemList.get(0);
            return purchaseOrderItem;
        } else {
            return null;
        }
    }
    @Cacheable(value = "purchaseOrderChange", keyGenerator = "redisKeyGenerator")
	@Override
	public int countByExample(
			PurchaseOrderChangeExample purchaseOrderChangeExample) {
		int countByExample = this.purchaseOrderChangeMapper.countByExample(purchaseOrderChangeExample);
		return countByExample;
	}


    @Transactional
    @CacheEvict(value = { "purchaseOrderChange" }, allEntries = true)
	@Override
	public void addAll(List<PurchaseOrderChange> arg0) {
		if (CollectionUtils.isEmpty(arg0)) {
			return;
		}
		for (PurchaseOrderChange record : arg0) {
			this.purchaseOrderChangeMapper.insertSelective(record);
		}
	}

}