package com.els.base.supperorder.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.company.dao.CompanyMapper;
import com.els.base.company.entity.Company;
import com.els.base.core.dao.user.UserMapper;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.user.User;
import com.els.base.core.entity.user.UserExample;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.file.entity.FileData;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;
import com.els.base.purchase.entity.PurchaseOrder;
import com.els.base.purchase.entity.PurchaseOrderExample;
import com.els.base.purchase.entity.PurchaseOrderExample.Criteria;
import com.els.base.purchase.entity.PurchaseOrderItem;
import com.els.base.purchase.entity.PurchaseOrderItemExample;
import com.els.base.purchase.service.PurchaseOrderItemService;
import com.els.base.purchase.service.PurchaseOrderService;
import com.els.base.purchase.utils.OrderBusinessTypeEnum;
import com.els.base.purchase.utils.PurchaseOrderSendStatusEnum;
import com.els.base.purchase.utils.PurchaseOrderStatusEnum;
import com.els.base.purchase.utils.SupplierOrderPdfPrintUtil;
import com.els.base.purchase.utils.SysUtil;
import com.els.base.purchase.vo.KnSupOrderPdfItemVO;
import com.els.base.purchase.vo.KnSupOrderPdfVO;
import com.els.base.supperorder.dao.SupplierOrderItemMapper;
import com.els.base.supperorder.dao.SupplierOrderMapper;
import com.els.base.supperorder.entity.SupplierOrder;
import com.els.base.supperorder.entity.SupplierOrderExample;
import com.els.base.supperorder.entity.SupplierOrderItem;
import com.els.base.supperorder.entity.SupplierOrderItemExample;
import com.els.base.supperorder.service.SupplierOrderService;
import com.els.base.utils.uuid.UUIDGenerator;

@Service("supplierOrderService")
public class SupplierOrderServiceImpl implements SupplierOrderService {

    Logger logger = LoggerFactory.getLogger(SupplierOrderServiceImpl.class);
    @Resource
    protected SupplierOrderMapper supplierOrderMapper;
    @Resource
    protected CompanyMapper companyMapper;
    @Resource
    protected UserMapper userMapper;
    @Resource
    protected SupplierOrderItemMapper supplierOrderItemMapper;
    @Resource
    protected PurchaseOrderService purchaseOrderService;
    @Resource
    protected PurchaseOrderItemService purchaseOrderItemService;

    @CacheEvict(value = {"supplierOrder"}, allEntries = true)
    @Override
    public void addObj(SupplierOrder t) {
        this.supplierOrderMapper.insertSelective(t);
    }

    @CacheEvict(value = {"supplierOrder"}, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.supplierOrderMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = {"supplierOrder"}, allEntries = true)
    @Override
    public void modifyObj(SupplierOrder t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.supplierOrderMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "supplierOrder", keyGenerator = "redisKeyGenerator")
    @Override
    public SupplierOrder queryObjById(String id) {
        return this.supplierOrderMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "supplierOrder", keyGenerator = "redisKeyGenerator")
    @Override
    public List<SupplierOrder> queryAllObjByExample(SupplierOrderExample example) {
        return this.supplierOrderMapper.selectByExample(example);
    }

    @Cacheable(value = "supplierOrder", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<SupplierOrder> queryObjByPage(SupplierOrderExample example) {
        PageView<SupplierOrder> pageView = example.getPageView();
        pageView.setQueryResult(this.supplierOrderMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Transactional
    @CacheEvict(value = {"supplierOrder", "supplierOrderItem"}, allEntries = true)
    @Override
    public void saveSupplierOrder(SupplierOrder supplierOrder) {
        supplierOrder.setLastUpdateTime(new Date());
        List<SupplierOrderItem> supplierOrderItems = supplierOrder.getItems();
        SupplierOrder supplierOrderTemp = this.supplierOrderMapper.selectByPrimaryKey(supplierOrder.getId());
        if (supplierOrderTemp == null) {
            this.supplierOrderMapper.insertSelective(supplierOrder);
            if (CollectionUtils.isNotEmpty(supplierOrderItems)) {
                for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
                    SupplierOrderItem supplierOrderItemTemp = this.supplierOrderItemMapper.selectByPrimaryKey(supplierOrderItem.getId());
                    supplierOrderItem.setUpdateTime(new Date());
                    if (supplierOrderItemTemp == null) {
//						supplierOrderItem.setUpdateTime(new Date());
                        this.supplierOrderItemMapper.insertSelective(supplierOrderItem);
                    } else {
//						supplierOrderItem.setUpdateTime(new Date());
//						supplierOrderItem.setHaveChange(supplierOrderItemTemp.getHaveChange());
                        this.supplierOrderItemMapper.updateByPrimaryKey(supplierOrderItem);
                    }
                }
            }
        } else {
            this.supplierOrderMapper.updateByPrimaryKey(supplierOrder);
            if (CollectionUtils.isNotEmpty(supplierOrderItems)) {
                for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
                    SupplierOrderItem supplierOrderItemTemp = this.supplierOrderItemMapper.selectByPrimaryKey(supplierOrderItem.getId());
                    supplierOrderItem.setUpdateTime(new Date());
                    if (supplierOrderItemTemp == null) {
//						supplierOrderItem.setUpdateTime(new Date());
                        this.supplierOrderItemMapper.insertSelective(supplierOrderItem);
                    } else {
//						supplierOrderItem.setUpdateTime(new Date());
//						supplierOrderItem.setHaveChange(supplierOrderItemTemp.getHaveChange());
                        this.supplierOrderItemMapper.updateByPrimaryKey(supplierOrderItem);
                    }
                }
            }
        }

    }

    /**
     * 判断退回的订单行数是否和订单行数相同
     *
     * @param supplierOrder
     * @return true 是
     */
    public boolean iSAllRefuse(SupplierOrder supplierOrder) {
        boolean flag = true;
        //查询已经被退回的订单行数是否
        SupplierOrderItemExample supplierOrderItemExample = new SupplierOrderItemExample();
        supplierOrderItemExample.createCriteria().andOrderNoEqualTo(supplierOrder.getOrderNo()).andCompanyIdEqualTo(supplierOrder.getCompanyId());
        List<SupplierOrderItem> supplierOrderItemItems = this.supplierOrderItemMapper.selectByExample(supplierOrderItemExample);
        for (SupplierOrderItem supplierOrderItem : supplierOrderItemItems) {
            Integer orderStatus = supplierOrderItem.getOrderStatus();
            //如果存在不是拒绝的订单行,同时不是删除的，删除的要过滤
            if (orderStatus != PurchaseOrderStatusEnum.BACK.getValue() && supplierOrderItem.getIsEnable() != 0) {
                flag = false;
                break;
            }
        }
        return flag;
    }

    /**
     * 判断确认的订单行数是否和订单行数相同
     *
     * @param supplierOrder
     * @return
     */
    public boolean iSAllConfirm(SupplierOrder supplierOrder) {
        boolean flag = true;
        //查询已经被确认的订单行数是否
        SupplierOrderItemExample supplierOrderItemExample = new SupplierOrderItemExample();
        supplierOrderItemExample.createCriteria().andOrderNoEqualTo(supplierOrder.getOrderNo()).andCompanyIdEqualTo(supplierOrder.getCompanyId());
        List<SupplierOrderItem> supplierOrderItemItems = this.supplierOrderItemMapper.selectByExample(supplierOrderItemExample);
        for (SupplierOrderItem supplierOrderItem : supplierOrderItemItems) {
            Integer orderStatus = supplierOrderItem.getOrderStatus();
            //如果存在不是拒绝的订单行,同时不是删除的，删除的要过滤
            if (orderStatus != PurchaseOrderStatusEnum.CONFIRM.getValue() && supplierOrderItem.getIsEnable() != 0) {
                flag = false;
                break;
            }
        }
        return flag;
    }

    /**
     * 订单行拒绝或者确认
     */
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(value = {"supplierOrder", "supplierOrderItem", "purchaseOrder", "purchaseOrderItem"}, allEntries = true)
    @Override
    public void supplierOrderItemRefuse(SupplierOrder supplierOrder) {

        logger.info("采购订单行拒绝开始{}", supplierOrder.getOrderNo());
        logger.info("采购订单状态为：{}", supplierOrder.getOrderStatus());


        List<SupplierOrderItem> supplierOrderItems = supplierOrder.getItems();
        int refuseNum = 0;// 拒绝的订单行数量
        int batchNum = 0; // 分批的订单行数量
        int confirmNum = 0; //确认的数量
        int noConfirmNum = 0;//未确认条数
        int totalOrderItem = supplierOrderItems.size();

        logger.info("1. 订单行数为：{}  ", totalOrderItem);

        // 修改销售订单行记录
        for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
            boolean isChange = false;
            logger.info("订单号:" + supplierOrderItem.getOrderNo() + "订单行号:" + supplierOrderItem.getOrderItemNo() + "订单状态:" + supplierOrderItem.getOrderStatus());
            if (supplierOrderItem.getOrderStatus() == PurchaseOrderStatusEnum.BACK.getValue()) {
                refuseNum++;
                isChange = true;
            }
            if (supplierOrderItem.getOrderStatus() == PurchaseOrderStatusEnum.ITEM_BATCH.getValue()) {
                batchNum++;
                isChange = true;
            }
            if (supplierOrderItem.getOrderStatus() == PurchaseOrderStatusEnum.CONFIRM.getValue()) {
                confirmNum++;
                isChange = true;
            }
            //如果改变了
            if (isChange) {
                supplierOrderItem.setUpdateTime(new Date());
                supplierOrderItem.setUserId(supplierOrder.getUserId());
                supplierOrderItem.setUserName(supplierOrder.getUserName());
                this.supplierOrderItemMapper.updateByPrimaryKey(supplierOrderItem);
            } else {
                logger.info("订单号:" + supplierOrderItem.getOrderNo() + "订单行号:" + supplierOrderItem.getOrderItemNo() + "没有修改状态");
                noConfirmNum++;
            }
        }
        if (noConfirmNum == supplierOrderItems.size()) {
            throw new CommonException("不存在确认、拒绝、分批的订单行，无法提交！");
        }

        logger.info("2. totalOrderItem = " + totalOrderItem);
        logger.info("batchNum = " + batchNum);
        logger.info("batchNum == totalOrderItem");

        // 判断是否是全部拒绝
        boolean flag = this.iSAllRefuse(supplierOrder);
        if (flag) {// 如果是全部拒绝
            logger.info("如果是全部拒绝---{}", supplierOrder.getOrderNo());
            supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.BACK.getValue());
            //如果进行进行最新记录还原
            if (SysUtil.isReturnNewRecord()) {
                //还原最新的头确认的记录
                if (!returnNewRecordHeader(supplierOrder)) {
                    //如果没有找到最新记录
                    //修改销售订单头状态
                    this.supplierOrderMapper.updateByPrimaryKey(supplierOrder);
                }
                for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
                    // 还原最新的行确认记录
                    returnNewRecordItem(supplierOrderItem);
                }
            } else {
                //修改销售订单头状态
                this.supplierOrderMapper.updateByPrimaryKey(supplierOrder);
            }
        } else {// 如果不是全部拒绝
            logger.info("如果不是全部拒绝---{}", supplierOrder.getOrderNo());
            if (confirmNum == supplierOrderItems.size()) {
                // 如果全部确认
                logger.info("如果全部确认---{}", supplierOrder.getOrderNo());
                supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.CONFIRM.getValue());
                supplierOrder.setLastUpdateTime(new Date());
                logger.info("修改销售订单头状态---{}", supplierOrder.getOrderNo());
                //修改销售订单头状态
                this.supplierOrderMapper.updateByPrimaryKey(supplierOrder);
                String hisId = UUIDGenerator.generateUUID();
                this.insertHis(hisId, supplierOrder.getId(), "订单头确认");
                for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
                    //如果是确认，则插入历史记录
                    logger.info("如果是确认，则插入历史记录---{}", supplierOrderItem.getOrderItemNo());
                    String hisId2 = UUIDGenerator.generateUUID();
                    this.supplierOrderItemMapper.insertHis(hisId2, supplierOrderItem.getId(), "订单行确认");
                }

            } else if (noConfirmNum > 0 && (batchNum == 0 && refuseNum == 0)) {
                //存在没有确认的订单行
                logger.info("存在没有确认的订单行---{}", supplierOrder.getOrderNo());
                supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.PART_CONFIRM.getValue());
                supplierOrder.setLastUpdateTime(new Date());
                //修改销售订单头状态
                logger.info("修改销售订单头状态---{}", supplierOrder.getOrderNo());
                this.supplierOrderMapper.updateByPrimaryKey(supplierOrder);
                String hisId = UUIDGenerator.generateUUID();
                this.insertHis(hisId, supplierOrder.getId(), "订单头部分确认");
                for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
                    //如果是确认，则插入历史记录
                    logger.info("如果是确认，则插入历史记录---{}", supplierOrderItem.getOrderItemNo());
                    String hisId2 = UUIDGenerator.generateUUID();
                    this.supplierOrderItemMapper.insertHis(hisId2, supplierOrderItem.getId(), "订单行部分确认");
                }
            } else if (batchNum == supplierOrderItems.size()
                    // 如果只有确认和分批状态并存的时候，属于分批状态
                    || (refuseNum == 0 && batchNum > 0 && confirmNum > 0)) {
                //如果只有确认和分批状态并存的时候 且不存在未确认订单行 属于分批状态
                logger.info("如果只有确认和分批状态并存的时候 且不存在未确认订单行 属于分批状态---{}", supplierOrder.getOrderNo());
                //如果全部分批
                supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.BATCH.getValue());
                supplierOrder.setLastUpdateTime(new Date());
                //修改销售订单头状态
                logger.info("修改销售订单头状态---{}", supplierOrder.getOrderNo());
                this.supplierOrderMapper.updateByPrimaryKey(supplierOrder);
                String hisId = UUIDGenerator.generateUUID();
                this.insertHis(hisId, supplierOrder.getId(), "订单头分批");
                for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
                    //如果是确认，则插入历史记录
                    logger.info("如果是确认，则插入历史记录---{}", supplierOrderItem.getOrderItemNo());
                    String hisId2 = UUIDGenerator.generateUUID();
                    this.supplierOrderItemMapper.insertHis(hisId2, supplierOrderItem.getId(), "订单行分批");
                }

            } else {

                //如果是部分拒绝
                logger.info("如果是部分拒绝---{}", supplierOrder.getOrderNo());
                supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.PART_BACK.getValue());
                for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
                    // 如果是拒绝的
                    if (supplierOrderItem.getOrderStatus() == PurchaseOrderStatusEnum.BACK.getValue()) {
                        //如果进行进行最新记录还原
                        if (SysUtil.isReturnNewRecord()) {
                            // 还原最新的行确认记录
                            returnNewRecordItem(supplierOrderItem);
                        }
                    } else {
                        //如果是确认，则插入历史记录
                        logger.info("如果是确认，则插入历史记录---{}", supplierOrderItem.getOrderItemNo());
                        String hisId = UUIDGenerator.generateUUID();
                        this.supplierOrderItemMapper.insertHis(hisId, supplierOrderItem.getId(), "订单行确认");
                    }
                }
                //修改销售订单头状态
                supplierOrder.setLastUpdateTime(new Date());
                this.supplierOrderMapper.updateByPrimaryKey(supplierOrder);
                String hisId = UUIDGenerator.generateUUID();
                this.insertHis(hisId, supplierOrder.getId(), "订单头部分拒绝");
            }
        }

        // 修改采购订单的状态
        sendToPurchase(supplierOrder);
    }

    /**
     * 查询最新的一条订单头记录
     *
     * @return
     */
    public SupplierOrder getNewestOrder(String id) {
        SupplierOrder supplierOrderTemp = new SupplierOrder();
        SupplierOrderExample example = new SupplierOrderExample();
        List<Integer> values = new ArrayList<Integer>();
        values.add(PurchaseOrderStatusEnum.CONFIRM.getValue());
        values.add(PurchaseOrderStatusEnum.PART_BACK.getValue());
        example.createCriteria().andIdEqualTo(id).andOrderStatusIn(values);
        example.setOrderByClause("LAST_UPDATE_TIME DESC");
        List<SupplierOrder> supplierOrderList = this.supplierOrderMapper.selectHisByExampleByPage(example);
        if (CollectionUtils.isNotEmpty(supplierOrderList)) {
            supplierOrderTemp = supplierOrderList.get(0);
            return supplierOrderTemp;
        } else {
            return null;
        }
    }

    /**
     * 查询最新的一条订单行记录
     *
     * @return
     */
    public SupplierOrderItem getNewestOrderItem(String id) {
        SupplierOrderItem supplierOrderItemTemp = new SupplierOrderItem();
        SupplierOrderItemExample example = new SupplierOrderItemExample();
        example.createCriteria().andIdEqualTo(id).andOrderStatusEqualTo(PurchaseOrderStatusEnum.CONFIRM.getValue());
        example.setOrderByClause("UPDATE_TIME DESC");
        List<SupplierOrderItem> supplierOrderItemList = this.supplierOrderItemMapper.selectHisByExampleByPage(example);
        if (CollectionUtils.isNotEmpty(supplierOrderItemList)) {
            supplierOrderItemTemp = supplierOrderItemList.get(0);
            return supplierOrderItemTemp;
        } else {
            return null;
        }
    }

    @Override
    public int insertHis(String hisId, String orderId, String cause) {
        return this.supplierOrderMapper.insertHis(hisId, orderId, cause);
    }


    /**
     * 订单头拒绝
     */
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(value = {"supplierOrder", "supplierOrderItem", "purchaseOrder", "purchaseOrderItem"}, allEntries = true)
    @Override
    public void supplierOrderRefuse(SupplierOrder supplierOrder) {
        int comfirmNum = 0;//之前确认的最新记录的订单行个数
        SupplierOrderItemExample exampleItem = new SupplierOrderItemExample();
        exampleItem.createCriteria()
                .andCompanyIdEqualTo(supplierOrder.getCompanyId())
                .andOrderNoEqualTo(supplierOrder.getOrderNo())
                .andFirstFlagEqualTo(Constant.NO_INT);
        // 过滤首批订单
        exampleItem.or()
                .andCompanyIdEqualTo(supplierOrder.getCompanyId())
                .andOrderNoEqualTo(supplierOrder.getOrderNo())
                .andFirstFlagIsNull();
        List<SupplierOrderItem> supplierOrderItems = this.supplierOrderItemMapper.selectByExample(exampleItem);
        String supOrderHeadRemark = supplierOrder.getSupRemark();
        int noRefuseCount = 0;
        logger.info("==============================================");
        logger.info("supOrderHeadRemark = " + supOrderHeadRemark);

        logger.info("=====================修改采购订单行=========================");
        for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
            //修改销售订单行状态
            //未确认订单行 才可以
            if (supplierOrderItem.getOrderStatus().equals(PurchaseOrderStatusEnum.UNCONFIRM.getValue())) {
                supplierOrderItem.setUserId(supplierOrder.getUserId());
                supplierOrderItem.setUserName(supplierOrder.getUserName());
                supplierOrderItem.setOrderStatus(PurchaseOrderStatusEnum.BACK.getValue());
                supplierOrderItem.setRemark(supOrderHeadRemark);
                supplierOrderItem.setSupRemark(supOrderHeadRemark);
                //如果进行进行最新记录还原
                if (SysUtil.isReturnNewRecord()) {
                    // 还原最新的行确认记录
                    if (returnNewRecordItem(supplierOrderItem)) {
                        comfirmNum++;
                    } else {
                        supplierOrderItem.setUpdateTime(new Date());
                        this.supplierOrderItemMapper.updateByPrimaryKey(supplierOrderItem);
                    }
                } else {
                    supplierOrderItem.setUpdateTime(new Date());
                    this.supplierOrderItemMapper.updateByPrimaryKey(supplierOrderItem);
                }
            } else {
                noRefuseCount++;
            }
        }

        logger.info("=====================修改采购订单头=========================");
        SupplierOrder record = new SupplierOrder();
        supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.BACK.getValue());//全部拒绝
        // 修改销售订单头备注
        record.setSupRemark(supplierOrder.getSupRemark());
        // 修改销售订单头状态
        record.setOrderStatus(supplierOrder.getOrderStatus());//全部拒绝
        record.setId(supplierOrder.getId());
        record.setUserId(supplierOrder.getUserId());
        record.setUserName(supplierOrder.getUserName());
        record.setLastUpdateTime(new Date());

        //如果进行进行最新记录还原
        if (SysUtil.isReturnNewRecord()) {
            // 还原最新的头确认记录
            if (!returnNewRecordHeader(supplierOrder)) {
                if (noRefuseCount > 0) {
                    supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.PART_BACK.getValue());//部分拒绝
                    record.setOrderStatus(supplierOrder.getOrderStatus());//部分拒绝
                }
				/*if(comfirmNum!=0){
					supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.PART_BACK.getValue());//部分拒绝
					record.setOrderStatus(supplierOrder.getOrderStatus());//部分拒绝
				}*/
                this.supplierOrderMapper.updateByPrimaryKeySelective(record);
            }
        }
        supplierOrder.setItems(supplierOrderItems);
        //修改采购订单
        this.sendToPurchase(supplierOrder);

    }

    /**
     * 订单头确认
     */
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(value = {"supplierOrder", "supplierOrderItem", "purchaseOrder", "purchaseOrderItem"}, allEntries = true)
    @Override
    public void supplierOrderComfirm(SupplierOrder supplierOrder) {

        // 修改销售单头状态
        supplierOrder.setOrderStatus(PurchaseOrderStatusEnum.CONFIRM.getValue());
        SupplierOrder record = new SupplierOrder();
        record.setOrderStatus(supplierOrder.getOrderStatus());
        record.setId(supplierOrder.getId());
        record.setUserId(supplierOrder.getUserId());
        record.setUserName(supplierOrder.getUserName());
        record.setLastUpdateTime(new Date());
        record.setSupRemark(supplierOrder.getSupRemark());
        this.supplierOrderMapper.updateByPrimaryKeySelective(record);

        //插入头确认历史
        String hisId = UUIDGenerator.generateUUID();
        this.insertHis(hisId, supplierOrder.getId(), "订单头确认");

        //修改销售单行状态
        SupplierOrderItem recordItem = new SupplierOrderItem();
        recordItem.setOrderStatus(PurchaseOrderStatusEnum.CONFIRM.getValue());
        recordItem.setUserId(supplierOrder.getUserId());
        recordItem.setUserName(supplierOrder.getUserName());
        recordItem.setUpdateTime(new Date());
        SupplierOrderItemExample exampleItem = new SupplierOrderItemExample();
        List<Integer> isEnableValues = new ArrayList<Integer>();
        isEnableValues.add(1);
        isEnableValues.add(2);
        exampleItem.createCriteria()
                .andIsEnableIn(isEnableValues)
                .andCompanyIdEqualTo(supplierOrder.getCompanyId())
                .andOrderNoEqualTo(supplierOrder.getOrderNo())
                .andOrderStatusEqualTo(PurchaseOrderStatusEnum.UNCONFIRM.getValue());
        // 过滤首批订单
        exampleItem.or()
                .andIsEnableIn(isEnableValues)
                .andCompanyIdEqualTo(supplierOrder.getCompanyId())
                .andOrderNoEqualTo(supplierOrder.getOrderNo());
        this.supplierOrderItemMapper.updateByExampleSelective(recordItem, exampleItem);

        //插入行确认历史
        List<SupplierOrderItem> supplierOrderItems = this.supplierOrderItemMapper.selectByExample(exampleItem);
        for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
            String hisId2 = UUIDGenerator.generateUUID();
            this.supplierOrderItemMapper.insertHis(hisId2, supplierOrderItem.getId(), "订单行确认");
        }

        supplierOrder.setItems(supplierOrderItems);
        //修改采购订单
        sendToPurchase(supplierOrder);
    }

    /**
     * 供应商拒绝或者确认订单后发送给采购方
     *
     * @param supplierOrder
     */
    public void sendToPurchase(SupplierOrder supplierOrder) {
        PurchaseOrder record = new PurchaseOrder();
        record = this.purchaseOrderService.queryObjById(supplierOrder.getId());
        if (record != null) {
            //修改备注
            record.setSupRemark(supplierOrder.getSupRemark());
            //修改状态
            logger.info("supplierOrder.getOrderStatus() = " + supplierOrder.getOrderStatus());
            record.setOrderStatus(supplierOrder.getOrderStatus());
            record.setLastUpdateTime(new Date());
            PurchaseOrderExample example = new PurchaseOrderExample();
            example.createCriteria()
                    .andOrderNoEqualTo(supplierOrder.getOrderNo())
                    .andSupCompanySapCodeEqualTo(supplierOrder.getCompanySapCode());
            this.purchaseOrderService.updateByPurcahseOrder(record, example);
        }
        List<SupplierOrderItem> supplierOrderItems = supplierOrder.getItems();
        for (SupplierOrderItem supplierOrderItem : supplierOrderItems) {
            PurchaseOrderItem recordItem = new PurchaseOrderItem();
            recordItem = this.purchaseOrderItemService.queryObjById(supplierOrderItem.getId());
            if (recordItem != null) {
                //修改备注
                recordItem.setSupRemark(supplierOrderItem.getSupRemark());
                //修改状态
                recordItem.setOrderStatus(supplierOrderItem.getOrderStatus());
                PurchaseOrderItemExample exampleItem = new PurchaseOrderItemExample();
                List<Integer> isEnableValues = new ArrayList<Integer>();
                isEnableValues.add(1);
                isEnableValues.add(2);
                exampleItem.createCriteria()
                        .andIsEnableIn(isEnableValues)
                        .andSupCompanyIdEqualTo(supplierOrderItem.getCompanyId())
                        .andOrderNoEqualTo(supplierOrder.getOrderNo())
                        .andOrderItemNoEqualTo(supplierOrderItem.getOrderItemNo());
                this.purchaseOrderItemService.updateByPurcahseOrder(recordItem, exampleItem);
            }
        }

        Message<SupplierOrder> message = Message.init(supplierOrder)
                .setCompanyCode(supplierOrder.getCompanySapCode())
                .setSenderId(supplierOrder.getUserId())
                .addReceiverId(supplierOrder.getPurUserId())
                .setMsgLevel(MessageLevelEnum.HIGH);

        if (supplierOrder.getOrderStatus() == PurchaseOrderStatusEnum.BACK.getValue()) {
            //全部拒绝
            message.setBusinessTypeCode(OrderBusinessTypeEnum.PO_ALL_BACK.getCode());

        } else if (supplierOrder.getOrderStatus() == PurchaseOrderStatusEnum.PART_BACK.getValue()) {
            //部分拒绝
            message.setBusinessTypeCode(OrderBusinessTypeEnum.PO_ALL_BACK.getCode());

        } else {
            message.setBusinessTypeCode(OrderBusinessTypeEnum.PO_CONFIRM.getCode());
        }
        MessageSendUtils.sendMessage(message);
    }

    /**
     * 发送消息提醒给采购方
     *
     * @param title
     * @param content
     * @param toUserId
     * @param companyCode
     */
	/*public void sendMessage(String title,String content,String toUserId,String toCompanyCode){
 		try {
			MessageUtils.pushSystemMessage(title, content, toUserId,
					toCompanyCode, MessageLevelEnum.HIGH);
		} catch (ConnectException e) {
			e.printStackTrace();
		}
	}*/
    @CacheEvict(value = {"supplierOrder"}, allEntries = true)
    @Override
    public void updateBySupplierOrder(SupplierOrder record, SupplierOrderExample example) {
        this.supplierOrderMapper.updateByExampleSelective(record, example);

    }

    /**
     * 供应商确认或者拒绝之前判断采购方是否修改了订单
     *
     * @param supplierOrder
     */
    @Override
    public void isHaveNewOrder(SupplierOrder supplierOrder) {
        PurchaseOrderExample example = new PurchaseOrderExample();
        Criteria criteria = example.createCriteria();
        criteria.andOrderNoEqualTo(supplierOrder.getOrderNo());
        if (StringUtils.isNotBlank(supplierOrder.getCompanySapCode())) {
            criteria.andSupCompanySapCodeEqualTo(supplierOrder.getCompanySapCode());
        }
        if (StringUtils.isNotBlank(supplierOrder.getCompanyId())) {
            criteria.andSupCompanyIdEqualTo(supplierOrder.getCompanyId());
        }
        List<PurchaseOrder> purchaseOrders = this.purchaseOrderService.queryAllObjByExample(example);
        if (CollectionUtils.isNotEmpty(purchaseOrders)) {
            PurchaseOrder purchaseOrder = purchaseOrders.get(0);
            // 判断是否采购方订单发生变更
            if (purchaseOrder != null && purchaseOrder.getOrderSendStatus() == PurchaseOrderSendStatusEnum.UPDATE_NOTSEND.getValue()) {
                throw new CommonException("此订单客户已变更，不可确认！");
            }
        }
    }

    @Override
    public PageView<SupplierOrder> findByPageForAvaliableOrder(SupplierOrderExample example) {
        PageView<SupplierOrder> pageView = example.getPageView();
        pageView.setQueryResult(this.supplierOrderMapper.selectVisableOrderByExampleByPage(example));
        return pageView;
    }

    /**
     * 还原头最新的记录
     *
     * @param supplierOrder
     */
    public boolean returnNewRecordHeader(SupplierOrder supplierOrder) {
        boolean flag = false;
        SupplierOrder supplierOrderTemp = getNewestOrder(supplierOrder.getId());
        if (supplierOrderTemp != null) {// 如果有最新的确认记录（非第一次拒绝或者确认）
            flag = true;
            //还原最新的确认的记录
            this.supplierOrderMapper.updateByPrimaryKeySelective(supplierOrderTemp);
        }
        return flag;
    }

    /**
     * 还原行最新的记录
     *
     * @param supplierOrder
     */
    public boolean returnNewRecordItem(SupplierOrderItem supplierOrderItem) {
        boolean flag = false;
        SupplierOrderItem supplierOrderItemTemp = getNewestOrderItem(supplierOrderItem.getId());
        if (supplierOrderItemTemp != null) {// 如果有最新的确认记录（非第一次拒绝或者确认）
            flag = true;
            // 还原最新的确认记录
            this.supplierOrderItemMapper.updateByPrimaryKey(supplierOrderItemTemp);
        }
        return flag;
    }

    @Override
    public boolean isSameType(Set<String> orderIdList) {
        // 根据订单行查询出订单头信息
        boolean boo = true;

        List<String> idList = new ArrayList<String>();
        CollectionUtils.addAll(idList, orderIdList.iterator());

        SupplierOrderExample example = new SupplierOrderExample();
        example.createCriteria().andIdIn(idList);
        List<SupplierOrder> list = this.queryAllObjByExample(example);
        String type = null;
        for (SupplierOrder order : list) {
            if (type == null) {
                type = order.getVoucherType();
            } else if (!type.equals(order.getVoucherType())) {
                return false;
            }
        }
        return boo;
    }

    @Override
    public List<FileData> getPdfFiles(List<String> orderNoList) {
        List<FileData> fileDatas = new ArrayList<>();

        for (String orderNo : orderNoList) {
            KnSupOrderPdfVO supVo = new KnSupOrderPdfVO();
            List<KnSupOrderPdfItemVO> orderItemVOs = new ArrayList<>();
            // PDF头信息
            SupplierOrderExample example = new SupplierOrderExample();
            example.createCriteria().andOrderNoEqualTo(orderNo);
            List<SupplierOrder> orderList = supplierOrderMapper.selectByExample(example);
            if (!orderList.isEmpty()) {
                BeanUtils.copyProperties(orderList.get(0), supVo);
                // 公司信息
                Company company = companyMapper.selectByPrimaryKey(orderList.get(0).getCompanyId());
                BeanUtils.copyProperties(company, supVo);
                supVo.setSupCompanyName(orderList.get(0).getCompanyName());
                supVo.setSupEmail(company.getEmail());
                supVo.setSupplierPhone(company.getMobilephone());
                // 需方联系人信息
                UserExample userExample = new UserExample();
                userExample.createCriteria().andLoginNameEqualTo(orderList.get(0).getCreateUser());
                List<User> userList = userMapper.selectByExample(userExample);
                if (!userList.isEmpty()) {
                    User user = userList.get(0);
                    BeanUtils.copyProperties(user, supVo);
                    supVo.setNeddEmail(user.getEmail());
                }
            }

            // 物料清单信息
            SupplierOrderItemExample itemExample = new SupplierOrderItemExample();
            itemExample.createCriteria().andOrderNoEqualTo(orderNo);
            List<SupplierOrderItem> orderItemList = supplierOrderItemMapper.selectByExample(itemExample);
            for (SupplierOrderItem supplierOrderItem : orderItemList) {
                KnSupOrderPdfItemVO orderItemVO = new KnSupOrderPdfItemVO();
                BeanUtils.copyProperties(supplierOrderItem, orderItemVO);
                orderItemVOs.add(orderItemVO);
            }

            // =============================物料清单排序 start===========================
            // 新建集合存放排序字段
            List<String> itemNoList = new ArrayList<>();
            for (KnSupOrderPdfItemVO itemVO : orderItemVOs) {
                // 增加排序字段
                itemNoList.add(itemVO.getOrderItemNo());
            }
            // 排序
            Collections.sort(itemNoList);
            // 存放排序完成的物料清单集合
            List<KnSupOrderPdfItemVO> sortOrderItemVOs = new ArrayList<>();
            for (String itemNo : itemNoList) {
                for (KnSupOrderPdfItemVO itemVO : orderItemVOs) {
                    if (itemNo.equals(itemVO.getOrderItemNo())) {
                        sortOrderItemVOs.add(itemVO);
                    }
                }
            }

            // =============================物料清单排序 end============================

            supVo.setOrderItemVOs(sortOrderItemVOs);

            FileData pdfFile = SupplierOrderPdfPrintUtil.getPdfFile(supVo);
            fileDatas.add(pdfFile);
        }

        return fileDatas;
    }

    @Cacheable(value = "supplierOrder", keyGenerator = "redisKeyGenerator")
    @Override
    public int countByExample(SupplierOrderExample supplierOrderExample) {
        int countByExample = this.supplierOrderMapper.countByExample(supplierOrderExample);
        return countByExample;
    }

    @CacheEvict(value = {"supplierOrder"}, allEntries = true)
    @Transactional
	@Override
	public void addAll(List<SupplierOrder> arg0) {
		if (CollectionUtils.isEmpty(arg0)) {
			return;
		}
		for (SupplierOrder record : arg0) {
			this.supplierOrderMapper.insertSelective(record);
		}
	}

    @CacheEvict(value = {"supplierOrder"}, allEntries = true)
	@Override
	public void deleteByExample(SupplierOrderExample arg0) {
		Assert.isNotEmpty(arg0.getOredCriteria(), "删除的条件不能为空");
		this.supplierOrderMapper.deleteByExample(arg0);
	}
}
