package com.els.base.quality.annex.web.controller;

import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.company.utils.CompanyUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParam;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.quality.annex.entity.Annex;
import com.els.base.quality.annex.entity.AnnexExample;
import com.els.base.quality.annex.entity.AnnexItem;
import com.els.base.quality.annex.entity.AnnexItemExample;
import com.els.base.quality.annex.service.AnnexItemService;
import com.els.base.quality.annex.service.AnnexService;
import com.els.base.quality.utils.BillStatusEnum;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

import java.util.List;

import javax.annotation.Resource;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

@Api(value = "质量管理—附件往来表")
@Controller
@RequestMapping("annex")
public class AnnexController {
	@Resource
	protected AnnexService annexService;
	
	@Resource
	protected AnnexItemService annexItemService;

	@ApiOperation(httpMethod = "POST", value = "采购商创建质量管理—附件往来表")
	@RequestMapping("service/create")
	@ResponseBody
	public ResponseResult<String> create(@RequestBody Annex annex) {
		Assert.isNotNull(annex, "对象不能为空");
		Assert.isNotBlank(annex.getSupCompanySrmCode(), "供应商编码不能为空");
		if(StringUtils.isNotBlank(annex.getPurRemarks())){
			if(annex.getPurRemarks().length()>500){
				throw new CommonException("你输入的备注超出规定长度，操作失败");
			}
		}
		List<AnnexItem> annexItems = annex.getAnnexItemList();
		for(AnnexItem annexItem : annexItems){
			if(StringUtils.isNotBlank(annexItem.getItemPurRemarks())){
				if(annexItem.getItemPurRemarks().length()>500){
					throw new CommonException("你输入的采购方备注超出规定长度");
				}
			}
			if(StringUtils.isNotBlank(annexItem.getPurUploadName())){
				if(annexItem.getPurUploadName().length()>3500){
					throw new CommonException("你上传的附件数超标，操作失败");
				}
			}
		}
		this.annexService.insert(ProjectUtils.getProjectId(), SpringSecurityUtils.getLoginUser(), CompanyUtils.currentCompany(), annex);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod = "POST", value = "采购商批量删除质量管理—附件往来表")
	@RequestMapping("service/deleteByIds")
	@ResponseBody
	public ResponseResult<String> deleteByIds(@RequestBody(required = true) List<Annex> annexList) {
		Assert.isNotEmpty(annexList, "数据不能为空");
		this.annexService.deleteObjByIds(ProjectUtils.getProjectId(),annexList);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod = "POST", value = "采购商批量作废质量管理—附件往来表")
	@RequestMapping("service/invalidByIds")
	@ResponseBody
	public ResponseResult<String> invalidByIds(@RequestBody(required = true) List<Annex> annexList) {
		Assert.isNotEmpty(annexList, "数据不能为空");
		this.annexService.invalidObjByIds(ProjectUtils.getProjectId(),SpringSecurityUtils.getLoginUser(),annexList);
		return ResponseResult.success();
	}
	
	@ApiOperation(httpMethod = "POST", value = "采购商批量审核质量管理—附件往来表")
	@RequestMapping("service/checkByIds")
	@ResponseBody
	public ResponseResult<String> checkByIds(@RequestBody(required = true) List<Annex> annexList){
		Assert.isNotEmpty(annexList, "数据不能为空");
		this.annexService.checkObjByIds(ProjectUtils.getProjectId(),SpringSecurityUtils.getLoginUser(),annexList);
		return ResponseResult.success();
	}
	
	@ApiOperation(httpMethod = "POST", value = "采购商批量驳回质量管理—附件往来表")
	@RequestMapping("service/rejectByIds")
	@ResponseBody
	public ResponseResult<String> rejectByIds(@RequestBody(required = true) List<Annex> annexList){
		Assert.isNotEmpty(annexList, "数据不能为空");
		this.annexService.rejectObjByIds(ProjectUtils.getProjectId(),SpringSecurityUtils.getLoginUser(),annexList);
		return ResponseResult.success();
	}
	
	@ApiOperation(httpMethod = "POST", value = "采购商批量发送质量管理—附件往来表")
	@RequestMapping("service/pursendByIds")
	@ResponseBody
	public ResponseResult<String> pursendByIds(@RequestBody(required = true) List<Annex> annexList){
		Assert.isNotEmpty(annexList, "数据不能为空");
		this.annexService.purSendToSup(ProjectUtils.getProjectId(),SpringSecurityUtils.getLoginUser(),annexList);
		return ResponseResult.success();
	}
	
	@ApiOperation(httpMethod = "POST", value = "供应商批量发送质量管理—附件往来表")
	@RequestMapping("service/supsendByIds")
	@ResponseBody
	public ResponseResult<String> supsendByIds(@RequestBody(required = true) List<Annex> annexList){
		Assert.isNotEmpty(annexList, "数据不能为空");
		this.annexService.supSendToPur(ProjectUtils.getProjectId(),SpringSecurityUtils.getLoginUser(),annexList);
		return ResponseResult.success();
	}
	

	@ApiOperation(httpMethod = "POST", value = "采购商查询质量管理—附件往来表")
	@ApiImplicitParams({@ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
						@ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"), 
						@ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 Annex", paramType = "body", dataType = "QueryParamWapper") })
	@RequestMapping("service/pur/findByPage")
	@ResponseBody
	public ResponseResult<PageView<Annex>> purFindByPage(
				@RequestParam(defaultValue = "0") int pageNo, 
				@RequestParam(defaultValue = "10") int pageSize, 
				@RequestBody(required = false) QueryParamWapper wapper) {
		
		AnnexExample example = new AnnexExample();
		example.setPageView(new PageView<Annex>(pageNo, pageSize));
		example.setOrderByClause("CREATE_BILL_TIME DESC");

		AnnexExample.Criteria criteria = example.createCriteria();
		criteria.andIsEnableEqualTo(Constant.YES_INT)
				.andProjectIdEqualTo(ProjectUtils.getProjectId())
				.andPurCompanyIdEqualTo(CompanyUtils.currentCompanyId());
		if (wapper != null) {
			CriteriaUtils.addCriterion(criteria, wapper);
		}

		PageView<Annex> pageData = this.annexService.queryObjByPage(example);
		return ResponseResult.success(pageData);
	}
	
	@ApiOperation(httpMethod = "POST", value = "供应商查询质量管理—附件往来表")
	@ApiImplicitParams({ @ApiImplicitParam(name = "pageNo", required = false, value = "所在页", paramType = "query", dataType = "String", defaultValue = "0"),
						@ApiImplicitParam(name = "pageSize", required = false, value = "每页数量", paramType = "query", dataType = "String", defaultValue = "10"), 
						@ApiImplicitParam(name = "wapper", required = false, value = "查询条件,属性名请参考 Annex", paramType = "body", dataType = "QueryParamWapper") })
	@RequestMapping("service/sup/findByPage")
	@ResponseBody
	public ResponseResult<PageView<Annex>> supFindByPage(@RequestParam(defaultValue = "0") int pageNo, 
				@RequestParam(defaultValue = "10") int pageSize, 
				@RequestBody(required = false) QueryParamWapper wapper) {
		AnnexExample example = new AnnexExample();
		example.setPageView(new PageView<Annex>(pageNo, pageSize));
		example.setOrderByClause("CREATE_BILL_TIME DESC");
		
		//添加查询条件
		AnnexExample.Criteria criteria = example.createCriteria();
		criteria.andSendStatusEqualTo(Constant.YES_INT)
				.andProjectIdEqualTo(ProjectUtils.getProjectId())
				.andIsEnableEqualTo(Constant.YES_INT)
				.andSupCompanyIdEqualTo(CompanyUtils.currentCompanyId());
				
		if (wapper != null) {
			CriteriaUtils.addCriterion(criteria, wapper);
		}

		PageView<Annex> pageData = this.annexService.queryObjByPage(example);
		return ResponseResult.success(pageData);
	}
	
	@ApiOperation(httpMethod="POST", value="采购商查看单据详情")
	@RequestMapping("service/pur/findAnnexById")
	@ResponseBody
	public ResponseResult<Annex> purFindAnnexById(@RequestParam(required=true) String id){
		Assert.isNotBlank(id, "单据ID不能为空");
		Annex annex=this.annexService.queryObjById(id);
		Assert.isNotNull(annex, "根据ID查询的数据为空");
		// 查询附件往来行项数据
		AnnexItemExample annexItemExample = new AnnexItemExample();
		annexItemExample.setOrderByClause("SORT_NUMBER ASC");
		annexItemExample.createCriteria().andAnnexIdEqualTo(id);
		List<AnnexItem> annexItemList = this.annexItemService.queryAllObjByExample(annexItemExample);
		//如果单据未回签，采购商不能看到供应商上传的附件资料，回签了才能看到
		if (annex.getBillStatus().equals(BillStatusEnum.STATUS_UN_SIGN.getBillStatus())) {
			for (AnnexItem annexItem : annexItemList) {
				annexItem.setItemSupRemarks(null);
				annexItem.setSupReplyName(null);
			}
		}
		annex.setAnnexItemList(annexItemList);

		return ResponseResult.success(annex);
	}
	
	@ApiOperation(httpMethod="POST", value="供应商查看单据详情")
	@RequestMapping("service/sup/findAnnexById")
	@ResponseBody
	public ResponseResult<Annex> supFindAnnexById(@RequestParam(required=true) String id){
		Assert.isNotBlank(id, "单据ID不能为空");
		Annex annex=this.annexService.queryObjById(id);
		Assert.isNotNull(annex, "根据ID查询的数据为空");
		//查询附件往来行项数据
		AnnexItemExample annexItemExample=new AnnexItemExample();
		annexItemExample.setOrderByClause("SORT_NUMBER ASC");
		annexItemExample.createCriteria().andAnnexIdEqualTo(id);
		List<AnnexItem> annexItemList=this.annexItemService.queryAllObjByExample(annexItemExample);
		annex.setAnnexItemList(annexItemList);
		return ResponseResult.success(annex);
	}
	
	@ApiOperation(httpMethod = "POST", value = "采购商编辑质量管理—附件往来表")
	@RequestMapping("service/pur/edit")
	@ResponseBody
	public ResponseResult<String> purEdit(@RequestBody Annex annex) {
		if (StringUtils.isBlank(annex.getId())) {
			throw new CommonException("id 为空，保存失败");
		}
		if(StringUtils.isBlank(annex.getSupCompanySrmCode())){
			throw new CommonException("供应商编码为空，操作失败");
		}
		if(StringUtils.isNotBlank(annex.getPurRemarks())){
			if(annex.getPurRemarks().length()>500){
				throw new CommonException("你输入的备注超出规定长度，操作失败");
			}
		}
		List<AnnexItem> annexItems = annex.getAnnexItemList();
		for(AnnexItem annexItem : annexItems){
			if(StringUtils.isNotBlank(annexItem.getItemPurRemarks())){
				if(annexItem.getItemPurRemarks().length()>500){
					throw new CommonException("你输入的采购方备注超出规定长度");
				}
			}
			if(StringUtils.isNotBlank(annexItem.getPurUploadName())){
				if(annexItem.getPurUploadName().length()>3500){
					throw new CommonException("你上传的附件数超标，操作失败");
				}
			}
		}
		if(annex.getSendStatus()==Constant.YES_INT){
			throw new CommonException("单据已发送，不能再修改");
		}
		this.annexService.modifyObj(annex);
		return ResponseResult.success();
	}
	
	@ApiOperation(httpMethod = "POST", value = "供应商编辑质量管理—附件往来表")
	@RequestMapping("service/sup/edit")
	@ResponseBody
	public ResponseResult<String> supEdit(@RequestBody Annex annex) {
		if (StringUtils.isBlank(annex.getId())) {
			throw new CommonException("id 为空，保存失败");
		}
		List<AnnexItem> annexItems=annex.getAnnexItemList();
		for(AnnexItem annexItem : annexItems){
			if(StringUtils.isNotBlank(annexItem.getItemSupRemarks())){
				if(annexItem.getItemSupRemarks().length()>500){
					throw new CommonException("你输入的供应方备注超出规定长度");
				}
			}
			if(StringUtils.isNotBlank(annexItem.getSupReplyName())){
				if(annexItem.getSupReplyName().length()>3500){
					throw new CommonException("你上传的附件数超标，操作失败");
				}
			}
		}
		if(annex.getBillStatus()==BillStatusEnum.STATUS_UN_SIGN.getBillStatus()){
			this.annexService.supUpdateInfo(annex);
		}else{
			throw new CommonException("单据不是未回签状态，不能再修改");
		}
		return ResponseResult.success();
	}
}