package com.els.base.quality.fracas.service.impl;

import com.els.base.company.entity.Company;
import com.els.base.company.entity.CompanyExample;
import com.els.base.company.entity.CompanyPurRef;
import com.els.base.company.entity.CompanyPurRefExample;
import com.els.base.company.service.CompanyPurRefService;
import com.els.base.company.service.CompanyService;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.project.Project;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.quality.fracas.dao.FraCasMapper;
import com.els.base.quality.fracas.entity.FraCas;
import com.els.base.quality.fracas.entity.FraCasExample;
import com.els.base.quality.fracas.service.FraCasService;

import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import com.els.base.quality.fracas.utils.FraCasStatusEnum;
import com.els.base.utils.uuid.UUIDGenerator;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service("defaultFraCasService")
public class FraCasServiceImpl implements FraCasService {
    @Resource
    protected FraCasMapper fraCasMapper;

    @Resource
    protected CompanyService companyService;

    @Resource
    protected CompanyPurRefService companyPurRefService;

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void addObj(FraCas t) {
        this.fraCasMapper.insertSelective(t);
    }

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.fraCasMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void modifyObj(FraCas t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.fraCasMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "fraCas", keyGenerator = "redisKeyGenerator")
    @Override
    public FraCas queryObjById(String id) {
        return this.fraCasMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "fraCas", keyGenerator = "redisKeyGenerator")
    @Override
    public List<FraCas> queryAllObjByExample(FraCasExample example) {
        return this.fraCasMapper.selectByExample(example);
    }

    @Cacheable(value = "fraCas", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<FraCas> queryObjByPage(FraCasExample example) {
        PageView<FraCas> pageView = example.getPageView();
        pageView.setQueryResult(this.fraCasMapper.selectByExampleByPage(example));
        return pageView;
    }

    /**
     * Fracas数据导入SRM
     **/
    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void importFraCas(String projectId, String companyId, User user, List<FraCas> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        Date now = new Date();
        for (FraCas item : list) {
            try {
                item.setStatus(FraCasStatusEnum.STATUS_IMPORT.getValue());
                item.setProjectId(projectId);
                item.setCompanyId(companyId);
                item.setCreateUserId(user.getId());
                item.setUpdateUserId(user.getId());
                item.setCreateUserName(user.getNickName());
                item.setUpdateUserName(user.getNickName());
                item.setCreateTime(now);
                item.setUpdateTime(now);
                this.addObj(item);
            } catch (CommonException ee) {
                throw new CommonException("数据导入失败：" + ee.getMessage());
            }
        }
    }

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
	public void addFraCas(String projectId, String companyId,
			com.els.base.auth.entity.User user, FraCas fraCas) {
        Date now = new Date();
        fraCas.setStatus(FraCasStatusEnum.STATUS_NEW.getValue());
        fraCas.setProjectId(projectId);
        fraCas.setCompanyId(companyId);
        fraCas.setCreateUserId(user.getId());
        fraCas.setUpdateUserId(user.getId());
        fraCas.setCreateUserName(user.getNickName());
        fraCas.setUpdateUserName(user.getNickName());
        fraCas.setCreateTime(now);
        fraCas.setUpdateTime(now);
        this.addObj(fraCas);
		
	}
    /**
     * 将FraCas推送给采购员
     */
    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void pushedToBuyer(String projectId, String companyId, User user, List<FraCas> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        Date now = new Date();
        for (FraCas item : list) {
            try {
                if (StringUtils.isEmpty(item.getSupCompanySapCode())) {
                    throw new CommonException("尚未设置FRACAS单据责任供应商，无法分配给对应的采购");
                }
                FraCas fracas = this.queryObjById(item.getId());
                if (!FraCasStatusEnum.STATUS_IMPORT.getValue().equals(fracas.getStatus())) {
                    throw new CommonException("只有导入状态的单据才可发送采购员！");
                }
                //查找供应商数据
                Company supCompany = null;
                CompanyExample supompanyExample = new CompanyExample();
                supompanyExample.createCriteria().andCompanySapCodeEqualTo(item.getSupCompanySapCode());
                List<Company> supCompanyList = companyService.queryAllObjByExample(supompanyExample);
                if (CollectionUtils.isNotEmpty(supCompanyList)) {
                    supCompany = supCompanyList.get(0);
                }
                //找到供应商对应的采购员
                CompanyPurRefExample companyPurRefExample = new CompanyPurRefExample();
                CompanyPurRefExample.Criteria companyPurRefCriteria = companyPurRefExample.createCriteria();
                companyPurRefCriteria.andProjectIdEqualTo(projectId);
                companyPurRefCriteria.andSupCompanyIdEqualTo(supCompany.getId());
                companyPurRefCriteria.andPurCompanyIdEqualTo(companyId);
                List<CompanyPurRef> Clist = companyPurRefService.queryAllObjByExample(companyPurRefExample);
                if (Clist.isEmpty()) {
                    throw new CommonException("供应商" + supCompany.getCompanyName() + "尚未维护对应的采购员，无法分配给对应的采购员!");
                }
                item.setStatus(FraCasStatusEnum.STATUS_NEW.getValue());
                item.setUpdateUserId(user.getId());
                item.setUpdateUserName(user.getNickName());
                item.setUpdateTime(now);
                this.modifyObj(item);
                //

            } catch (CommonException ee) {
                throw new CommonException("推送失败：" + ee.getMessage());
            }
        }
    }


    /**
     * 将FraCas发送给供应商
     */
    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void pushedToSuppler(String projectId, String companyId, User user, List<FraCas> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        Date now = new Date();
        for (FraCas item : list) {
            try {
                item.setSendStatus(Constant.YES_INT);
                item.setStatus(FraCasStatusEnum.STATUS_PUBLIC.getValue());
                item.setUpdateUserId(user.getId());
                item.setUpdateUserName(user.getNickName());
                item.setUpdateTime(now);
                this.modifyObj(item);
                //

            } catch (CommonException ee) {
                throw new CommonException("推送失败：" + ee.getMessage());
            }
        }
    }

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void supplerResponse(User user, List<FraCas> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        Date now = new Date();
        for (FraCas item : list) {
            try {
                item.setStatus(FraCasStatusEnum.STATUS_CONFIRM.getValue());
                item.setUpdateUserId(user.getId());
                item.setUpdateUserName(user.getNickName());
                item.setUpdateTime(now);
                this.modifyObj(item);
                //
            } catch (CommonException ee) {
                throw new CommonException("推送失败：" + ee.getMessage());
            }
        }
    }

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void acceptToSuppler(String projectId, String companyId, User user, List<FraCas> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        Date now = new Date();
        for (FraCas item : list) {
            try {
                item.setStatus(FraCasStatusEnum.STATUS_APPROVE.getValue());
                item.setUpdateUserId(user.getId());
                item.setUpdateUserName(user.getNickName());
                item.setUpdateTime(now);
                this.modifyObj(item);
                //
            } catch (CommonException ee) {
                throw new CommonException("推送失败：" + ee.getMessage());
            }
        }
    }

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void refuseToSuppler(String projectId, String companyId, User user, List<FraCas> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        Date now = new Date();
        for (FraCas item : list) {
            try {
                item.setStatus(FraCasStatusEnum.STATUS_REFUSED.getValue());
                item.setUpdateUserId(user.getId());
                item.setUpdateUserName(user.getNickName());
                item.setUpdateTime(now);
                this.modifyObj(item);
                //
            } catch (CommonException ee) {
                throw new CommonException("推送失败：" + ee.getMessage());
            }
        }
    }

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void addAttachment(User user, FraCas fraCas) {
        Date now = new Date();
        fraCas.setUpdateUserId(user.getId());
        fraCas.setUpdateUserName(user.getNickName());
        fraCas.setUpdateTime(now);
        this.fraCasMapper.updateByPrimaryKey(fraCas);
    }
    
    @Cacheable(value = "fraCas", keyGenerator = "redisKeyGenerator")
	@Override
	public int countByExample(FraCasExample fraCasExample) {
		int countByExample = this.fraCasMapper.countByExample(fraCasExample);
		return countByExample;
	}

    @Transactional
    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void addAll(List<FraCas> arg0) {
        if (CollectionUtils.isEmpty(arg0)) {
            return;
        }
        arg0.stream().forEach(item->{
            if(StringUtils.isBlank(item.getId())){
                item.setId(UUIDGenerator.generateUUID());
            }
            this.fraCasMapper.insertSelective(item);
        });
        
    }

    @CacheEvict(value = {"fraCas"}, allEntries = true)
    @Override
    public void deleteByExample(FraCasExample arg0) {
        Assert.isNotNull(arg0, "参数不能为空");
        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
        this.fraCasMapper.deleteByExample(arg0);
        
    }

	
}