package com.els.base.quality.inspection.service.impl;

import com.els.base.company.entity.Company;
import com.els.base.company.entity.CompanyExample;
import com.els.base.company.service.CompanyService;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.quality.bill.service.UnQualifiedBillService;
import com.els.base.quality.inspection.dao.InspectionMapper;
import com.els.base.quality.inspection.entity.Inspection;
import com.els.base.quality.inspection.entity.InspectionExample;
import com.els.base.quality.inspection.service.InspectionService;

import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import com.els.base.utils.json.JsonUtils;
import com.els.base.utils.uuid.UUIDGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service("defaultInspectionService")
public class InspectionServiceImpl implements InspectionService {
	Logger LOGGER = LoggerFactory.getLogger(InspectionServiceImpl.class);
	
    @Resource
    protected InspectionMapper inspectionMapper;

    @Resource
    protected UnQualifiedBillService billService;

    @Resource
    protected CompanyService companyService;

    @CacheEvict(value = {"inspection"}, allEntries = true)
    @Override
    public void addObj(Inspection t) {
        this.inspectionMapper.insertSelective(t);
    }

    @CacheEvict(value = {"inspection"}, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.inspectionMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = {"inspection"}, allEntries = true)
    @Override
    public void modifyObj(Inspection t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.inspectionMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "inspection", keyGenerator = "redisKeyGenerator")
    @Override
    public Inspection queryObjById(String id) {
        return this.inspectionMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "inspection", keyGenerator = "redisKeyGenerator")
    @Override
    public List<Inspection> queryAllObjByExample(InspectionExample example) {
        return this.inspectionMapper.selectByExample(example);
    }

    @Cacheable(value = "inspection", keyGenerator = "redisKeyGenerator")
    @Override
    public PageView<Inspection> queryObjByPage(InspectionExample example) {
    	LOGGER.info("查询接口------------------");
        PageView<Inspection> pageView = example.getPageView();
        pageView.setQueryResult(this.inspectionMapper.selectByExampleByPage(example));
        return pageView;
    }

    /**
     * 从SAP导入不合格检验批次系信息
     **/
    @CacheEvict(value = {"inspection"}, allEntries = true)
    @Override
    public int importInspectionFromSAP(List<Inspection> list, StringBuilder builder) {
        int errorCount = 0;
        builder.append("开始处理数据").append("\r\n");
        if (list == null) {
            throw new IllegalArgumentException("parameter materials is null");
        }
        builder.append("接收到不合格检验批次数据" + list.size() + "条").append("\r\n");

        String projectId = "6";
        String companyId = "1";
        String purCompanyCode = "50000000";
        CompanyExample companyExample = new CompanyExample();
        companyExample.createCriteria().andCompanyCodeEqualTo(purCompanyCode);
        List<Company> companyList = companyService.queryAllObjByExample(companyExample);
        if (CollectionUtils.isNotEmpty(companyList)) {
            Company c = companyList.get(0);
            projectId = c.getProjectId();
            companyId = c.getId();
        }

        builder.append("开始处理数据").append("\r\n");
        for (Inspection item : list) {
            try {

                InspectionExample inspectionExample = new InspectionExample();
                InspectionExample.Criteria criteria = inspectionExample.createCriteria();
                criteria.andInspectionLotEqualTo(item.getInspectionLot());
                int count = inspectionMapper.countByExample(inspectionExample);
                if (count > 0) {
                    Inspection inspection = inspectionMapper.selectByExample(inspectionExample).get(0);
                    criteria.andIdEqualTo(inspection.getId());
                    item.setUpdateTime(new Date());
                    item.setUpdateUserId("Webservice");
                    item.setUpdateUserName("Webservice");
                    inspectionMapper.updateByExampleSelective(item, inspectionExample);
                } else {
                    item.setProjectId(projectId);
                    item.setCompanyId(companyId);
                    item.setCreateTime(new Date());
                    item.setCreateUserId(item.getInspectionUser());
                    item.setCreateUserName(item.getInspectionUser());
                    item.setUpdateTime(new Date());
                    item.setUpdateUserId(item.getInspectionUser());
                    item.setUpdateUserName(item.getInspectionUser());
                    inspectionMapper.insertSelective(item);
                    //根据不合格品检验单生成不合格品处置单
                    //if (item.getUdCodeShortTxt().equals("退货")) {
                        try {
                            billService.createUnQualifiedBill(item);
                        } catch (CommonException ee) {
                            builder.append("不合格检验批次生成不合格品处置单据失败,错误原因:" + ee.getMessage()).append("\r\n");
                        }
                        catch (Exception ee) {
                            builder.append("不合格检验批次生成不合格品处置单据失败,错误原因:" + ee.getMessage()).append("\r\n");
                        }
                    //}
                }

            } catch (Exception e) {
                errorCount++;
                builder.append("检验批次:" + item.getInspectionLot() + "数据保存失败")
                        .append("\r\n");
                try {
                    builder.append("检验批次信息").append("\r\n");
                    builder.append(JsonUtils.writeValueAsString(item)).append("\r\n");
                } catch (JsonProcessingException e1) {
                    e1.printStackTrace();
                }
                builder.append("错误原因:" + e.getMessage()).append("\r\n");
            }
        }
        builder.append("处理成功条数 " + (list.size() - errorCount) + "条").append("\r\n");
        builder.append("处理失败条数 " + errorCount + "条").append("\r\n");
        builder.append("数据处理完成").append("\r\n");
        return errorCount;
    }

    @Cacheable(value = "inspection", keyGenerator = "redisKeyGenerator")
    public Boolean isExists(Inspection entity) {

        InspectionExample inspectionExample = new InspectionExample();
        InspectionExample.Criteria criteria = inspectionExample.createCriteria();
        criteria.andInspectionLotEqualTo(entity.getInspectionLot());
        int count = inspectionMapper.countByExample(inspectionExample);
        if (count > 0) {
            return true;
        } else {
            return false;
        }
    }

    @Transactional
    @CacheEvict(value = {"inspection"}, allEntries = true)
    @Override
    public void addAll(List<Inspection> arg0) {
        if (CollectionUtils.isEmpty(arg0)) {
            return;
        }
        arg0.stream().forEach(item->{
            if(StringUtils.isBlank(item.getId())){
                item.setId(UUIDGenerator.generateUUID());
            }
            this.inspectionMapper.insertSelective(item);
        });
        
    }

    @CacheEvict(value = {"inspection"}, allEntries = true)
    @Override
    public void deleteByExample(InspectionExample arg0) {
        Assert.isNotNull(arg0, "参数不能为空");
        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
        this.inspectionMapper.deleteByExample(arg0);
        
    }
}