package com.els.base.quality.report.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.codegenerator.service.GenerateCodeService;
import com.els.base.company.entity.Company;
import com.els.base.company.service.CompanyService;
import com.els.base.company.service.CompanyUserRefService;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.project.Project;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;
import com.els.base.quality.harms.im.HarmMaterialImTemplet;
import com.els.base.quality.report.dao.BadReportMapper;
import com.els.base.quality.report.entity.BadReport;
import com.els.base.quality.report.entity.BadReportExample;
import com.els.base.quality.report.entity.BadReportItem;
import com.els.base.quality.report.entity.BadReportItemExample;
import com.els.base.quality.report.service.BadReportItemService;
import com.els.base.quality.report.service.BadReportService;
import com.els.base.quality.utils.BillStatusEnum;
import com.els.base.utils.uuid.UUIDGenerator;

@Service("defaultBadeReportService")
public class BadReportServiceImpl implements BadReportService {

    private static final String IM_CODE = "BAD_REPORT_MESSAGE";

    @Resource
    private BadReportMapper badReportMapper;
    @Resource
    private CompanyService companyService;
    @Resource
    private GenerateCodeService generateCodeService;
    @Resource
    private BadReportItemService badReportItemService;
    @Resource
    private CompanyUserRefService companyUserRefService;
    @Resource
    private ThreadPoolTaskExecutor defaultThreadPool;

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void editForSup(Project project, Company company, User loginUser, BadReport badReport) {
        //校验信息
        Assert.isNotBlank(badReport.getId(),"不良报告ID不能为空,操作失败!");
        BadReport badReportOld = badReportMapper.selectByPrimaryKey(badReport.getId());
        Assert.isNotNull(badReportOld,"不良报告不存在,操作失败!");

        //更新不良报告头信息
        this.updateBadReportInfoForSup(badReport,badReportOld);

        //更新不良报告行信息
        //this.updateBadReportItemInfoForSup(badReport,badReportOld);

    }

    /**
     * 只更新附件信息
     * @param badReport
     * @param badReportOld
     */
    private void updateBadReportInfoForSup(BadReport badReport, BadReport badReportOld) {
        Assert.isNotBlank(badReport.getSupBadReportFile(),"单据中供应商回复附件没有上传，请检查，谢谢！");
        if(badReport.getSupBadReportFile().equals("[]")||badReport.getSupBadReportFile().equals("[ ]")){
                throw  new CommonException("单据中供应商回复附件没有上传，请检查，谢谢！");
        }
        badReportOld.setSupBadReportFile(badReport.getSupBadReportFile());
        badReportOld.setSupCommitTime(new Date());
        this.badReportMapper.updateByPrimaryKeySelective(badReportOld);
    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void editForPur(Project project, Company company, User loginUser, BadReport badReport) {

        //校验信息
        Assert.isNotBlank(badReport.getId(),"不良报告ID不能为空,操作失败!");
        BadReport badReportOld = badReportMapper.selectByPrimaryKey(badReport.getId());
        Assert.isNotNull(badReportOld,"不良报告不存在,操作失败!");

        //更新不良报告头信息
        this.updateBadReportInfoForPur(badReport,badReportOld);

        //更新不良报告行信息
        this.updateBadReportItemInfoForPur(badReport,badReportOld);

    }

    /**
     * 更新行信息
     * @param badReport  前端传进来的信息
     * @param badReportOld  查询出来的信息
     */
    private void updateBadReportItemInfoForPur(BadReport badReport, BadReport badReportOld) {
        List<BadReportItem> badReportItemList = badReport.getBadReportItemList();
        Assert.isNotEmpty(badReportItemList,"不良报告行列表不能为空！");
        Assert.isNotBlank(badReport.getId(),"不良报告ID不能为空！");

        //删除之前的
        BadReportItemExample badReportItemExample = new BadReportItemExample();
        badReportItemExample.createCriteria().andBadReportIdEqualTo(badReport.getId());
        this.badReportItemService.deleteByExample(badReportItemExample);

        //重新插入
        for (BadReportItem badReportItem : badReportItemList) {
            badReportItem.setId(null);
            badReportItem.setBadReportId(badReportOld.getId());
            this.badReportItemService.addObj(badReportItem);
        }
    }

    /**
     * 质量管理-更新头部信息
     * @param badReport  前端传进来的信息
     * @param badReportOld  查询出来的信息
     */
    private void updateBadReportInfoForPur(BadReport badReport, BadReport badReportOld) {

        //供应商提交资料时间
        //badReportOld.setSupCommitTime(new Date());

        //机种名
        badReportOld.setModelName(badReport.getModelName());

        //零件名
        //零件编号
        badReportOld.setSupComponentNo(badReport.getSupComponentNo());
        badReportOld.setSupComponentName(badReport.getSupComponentName());

        //LOT NO
        //LOT数
        //LOT 描述
        badReportOld.setLotNo(badReport.getLotNo());
        badReportOld.setLotQuantity(badReport.getLotQuantity());
        badReportOld.setLotDesc(badReport.getLotDesc());

        //发生时间
        //发生场所
        badReportOld.setReportTime(badReport.getReportTime());
        badReportOld.setReportPlace(badReport.getReportPlace());

        //承认人
        //品管部门确认
        badReportOld.setRecognizedUserId(badReport.getRecognizedUserId());
        badReportOld.setRecognizedUserName(badReport.getRecognizedUserName());
        badReportOld.setQualityConfirmId(badReport.getQualityConfirmId());
        badReportOld.setQualityConfirmName(badReport.getQualityConfirmName());

        //发行
        badReportOld.setPublishName(badReport.getPublishName());

        //图片
        badReportOld.setBadReportImg(badReport.getBadReportImg());

        //附件
        badReportOld.setPurBadReportFile(badReport.getPurBadReportFile());
        badReportOld.setSupBadReportFile(badReport.getSupBadReportFile());
        badReportOld.setPurCommitTime(new Date());

        //采购备注
        badReportOld.setPurRemark(badReport.getPurRemark());

        this.badReportMapper.updateByPrimaryKeySelective(badReportOld);

    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void reject(Project project, Company company, final User loginUser, List<BadReport> badReportList) {

        //校验
        Assert.isNotEmpty(badReportList,"不良报告列表不能为空!");
        List<String> ids=new ArrayList<>();
        for (BadReport badReport : badReportList) {
            Assert.isNotBlank(badReport.getId(),"不良报告ID不能为空!");
            ids.add(badReport.getId());
        }
        BadReportExample badReportExample =new BadReportExample();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.NO_INT)
                .andIdIn(ids);

        //未发送的不能驳回
        int countByExample = badReportMapper.countByExample(badReportExample);
        if(countByExample>0){
            throw  new CommonException("单据中含有未发送数据，操作失败!");
        }

        //判断是否满足已回签的条件
        List<Integer> billStatus=new ArrayList<>();
        billStatus.add(BillStatusEnum.STATUS_UN_SIGN.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_SUCCESS.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_FAIL.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_ABOLISH.getBillStatus());

        badReportExample.clear();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andBillStatusIn(billStatus)
                .andIdIn(ids);
        int countByExample2 = badReportMapper.countByExample(badReportExample);
        if(countByExample2>0){
            throw  new CommonException("单据中含有已驳回，已作废，已审批，未回签等数据，操作失败!");
        }

        //已发送，已回签才可以驳回
        badReportExample.clear();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andBillStatusEqualTo(BillStatusEnum.STATUS_SIGNED.getBillStatus())
                .andIdIn(ids);

        BadReport  badReport=new BadReport();
        badReport.setPurSendTime(new Date());
        badReport.setBillStatus(BillStatusEnum.STATUS_APPROVE_FAIL.getBillStatus());

        this.badReportMapper.updateByExampleSelective(badReport,badReportExample);

        //消息推送
        badReportExample.clear();
        badReportExample.createCriteria().andIdIn(ids);
        final   List<BadReport> badReports = badReportMapper.selectByExample(badReportExample);
        if(CollectionUtils.isNotEmpty(badReports)){
            this.defaultThreadPool.execute(new Runnable() {
                @Override
                public void run() {
                	for(BadReport badReport : badReports){
                		final HarmMaterialImTemplet templet = constructImInfo(1, "驳回", badReport);
                		sendMessagesToSup(loginUser,templet,badReport);
                	}
                }
            });
        }

    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void audit(Project project, Company company, final User loginUser, List<BadReport> badReportList) {
        //校验
        Assert.isNotEmpty(badReportList,"不良报告列表不能为空!");
        List<String> ids=new ArrayList<>();
        for (BadReport badReport : badReportList) {
            Assert.isNotBlank(badReport.getId(),"不良报告ID不能为空!");
            ids.add(badReport.getId());
        }
        BadReportExample badReportExample =new BadReportExample();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.NO_INT)
                .andIdIn(ids);

        //未发送的不能审核
        int countByExample = badReportMapper.countByExample(badReportExample);
        if(countByExample>0){
            throw  new CommonException("单据中含有未发送数据，操作失败!");
        }

        //判断是否满足已回签的条件
        List<Integer> billStatus=new ArrayList<>();
        billStatus.add(BillStatusEnum.STATUS_UN_SIGN.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_SUCCESS.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_FAIL.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_ABOLISH.getBillStatus());

        badReportExample.clear();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andBillStatusIn(billStatus)
                .andIdIn(ids);
        int countByExample2 = badReportMapper.countByExample(badReportExample);
        if(countByExample2>0){
            throw  new CommonException("单据中含有已驳回，已作废，已审批，未回签等数据，操作失败!");
        }

        //已发送，已回签才可以审核通过
        badReportExample.clear();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andBillStatusEqualTo(BillStatusEnum.STATUS_SIGNED.getBillStatus())
                .andIdIn(ids);

        BadReport  badReport=new BadReport();
        badReport.setPurSendTime(new Date());
        badReport.setBillStatus(BillStatusEnum.STATUS_APPROVE_SUCCESS.getBillStatus());

        this.badReportMapper.updateByExampleSelective(badReport,badReportExample);

        //消息推送
        badReportExample.clear();
        badReportExample.createCriteria().andIdIn(ids);
        final   List<BadReport> badReports = badReportMapper.selectByExample(badReportExample);
        if(CollectionUtils.isNotEmpty(badReports)){
			this.defaultThreadPool.execute(new Runnable() {
				@Override
				public void run() {
					for (BadReport badReport : badReports) {
						final HarmMaterialImTemplet templet = constructImInfo(1, "审核通过", badReport);
						sendMessagesToSup(loginUser, templet, badReport);
					}
				}
			});
        }
    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void abolish(Project project, Company company, final User loginUser, List<BadReport> badReportList) {

        //校验
        Assert.isNotEmpty(badReportList,"不良报告列表不能为空!");
        List<String> ids=new ArrayList<>();
        for (BadReport badReport : badReportList) {
            Assert.isNotBlank(badReport.getId(),"不良报告ID不能为空!");
            ids.add(badReport.getId());
        }
        BadReportExample badReportExample =new BadReportExample();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.NO_INT)
                .andIdIn(ids);

        //未发送的不能驳回
        int countByExample = badReportMapper.countByExample(badReportExample);
        if(countByExample>0){
            throw  new CommonException("单据中含有未发送数据，操作失败!");
        }

        // 已发送，未回签的才可以作废
        // 已驳回，已作废，已审批，已回签
        List<Integer> billStatus=new ArrayList<>();
        billStatus.add(BillStatusEnum.STATUS_SIGNED.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_SUCCESS.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_FAIL.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_ABOLISH.getBillStatus());

        badReportExample.clear();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andBillStatusIn(billStatus)
                .andIdIn(ids);
        int countByExample2 = badReportMapper.countByExample(badReportExample);
        if(countByExample2>0){
            throw  new CommonException("单据中含有已驳回，已作废，已审批，已回签等数据，操作失败!");
        }

        BadReport  badReport=new BadReport();
        badReport.setPurSendTime(new Date());
        badReport.setBillStatus(BillStatusEnum.STATUS_ABOLISH.getBillStatus());

        //已发送，未回签的数据
        badReportExample.clear();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andBillStatusEqualTo(BillStatusEnum.STATUS_UN_SIGN.getBillStatus())
                .andIdIn(ids);
        this.badReportMapper.updateByExampleSelective(badReport,badReportExample);


        //消息推送
        badReportExample.clear();
        badReportExample.createCriteria().andIdIn(ids);
        final   List<BadReport> badReports = badReportMapper.selectByExample(badReportExample);
        if(CollectionUtils.isNotEmpty(badReports)){
            this.defaultThreadPool.execute(new Runnable() {
                @Override
                public void run() {
                	for(BadReport badReport : badReports){
                		final HarmMaterialImTemplet templet = constructImInfo(1, "作废", badReport);
                		sendMessagesToSup(loginUser,templet,badReport);
                	}
                }
            });
        }

    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void sendForSup(Project project, Company company, final User loginUser, List<BadReport> badReportList) {

        //校验
        Assert.isNotEmpty(badReportList,"不良报告列表不能为空!");
        List<String> ids=new ArrayList<>();
        for (BadReport badReport : badReportList) {
            Assert.isNotBlank(badReport.getId(),"不良报告ID不能为空!");
            ids.add(badReport.getId());
        }

        List<Integer> billStatus =new ArrayList<>();
        billStatus.add(BillStatusEnum.STATUS_SIGNED.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_SUCCESS.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_FAIL.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_ABOLISH.getBillStatus());


        BadReportExample badReportExample =new BadReportExample();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSupCompanyIdEqualTo(company.getId())
                .andBillStatusIn(billStatus)
                .andIdIn(ids);

        int countByExample = badReportMapper.countByExample(badReportExample);
        if(countByExample>0){
            throw  new CommonException("单据中含有已回签，已审批，已驳回，已作废等数据，操作失败!");
        }

        badReportExample.clear();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSupCompanyIdEqualTo(company.getId())
                .andBillStatusEqualTo(BillStatusEnum.STATUS_UN_SIGN.getBillStatus())
                .andIdIn(ids);

        //更新状态
        final List<BadReport> badReports = badReportMapper.selectByExample(badReportExample);
        if(CollectionUtils.isEmpty(badReports)){
            throw  new  CommonException("找不到符合条件的单据，操作失败！");
        }
        for (final BadReport report : badReports) {
            Assert.isNotBlank(report.getSupBadReportFile(),"单据中含有供应商回复附件还没有上传，请先上传后再回签，谢谢！");
            if(report.getSupBadReportFile().equals("[]")||report.getSupBadReportFile().equals("[ ]")){
                throw new CommonException("单据中含有供应商回复附件还没有上传，请先上传后再回签，谢谢！");
            }
            report.setSupSignTime(new Date());
            report.setBillStatus(BillStatusEnum.STATUS_SIGNED.getBillStatus());
            this.badReportMapper.updateByPrimaryKeySelective(report);
        }

        //发送消息
        this.defaultThreadPool.execute(new Runnable() {
            @Override
            public void run() {
            	for(BadReport badReport : badReports){
            		final HarmMaterialImTemplet templet = constructImInfo(2, "回签", badReport);
            		sendMessagesToPur( badReport,templet);
            	}
            }
        });
    }


    private HarmMaterialImTemplet constructImInfo(int role,String operate,BadReport list){
    	Assert.isNotNull(list, "传输数据为空");
        HarmMaterialImTemplet tem = new HarmMaterialImTemplet();
        //设置角色信息:1是为采购商,其他为供应商
        if(role == 1){
            tem.setRole("采购商");
            //设置采购商的名称
            if(StringUtils.isNotEmpty(list.getPurCompanyName())){
                tem.setCompanyName(list.getPurCompanyName());
            }else{
                tem.setCompanyName("");
            }
        }else{
            tem.setRole("供应商");
            //设置供应商的名称
            if(StringUtils.isNotEmpty(list.getSupCompanyName())){
                tem.setCompanyName(list.getSupCompanyName());
            }else{
                tem.setCompanyName("");
            }
        }
        //设置操作
        if(StringUtils.isEmpty(operate)){
            throw new CommonException("没有传入操作字段,无法生成消息模板!");
        }
        tem.setOperate(operate);
        //设置其他信息
        tem.setOtherInfo("了新的零件不良报告单据,单据号包括：");

        //单据号
        /*List<String > noList = new ArrayList<>();
        for (BadReport badReport : list) {
            noList.add(badReport.getBadReportNo());
        }
        String  billData= StringUtils.join(noList,",");*/
        tem.setBillList(list.getBadReportNo());
        return tem;
    }


    /**
     * 采购商发送消息给供应商
     */
    public void sendMessagesToSup(User loginUser,HarmMaterialImTemplet templet,BadReport data){
        User supUser = companyUserRefService.queryMainUserOfCompany(data.getSupCompanyId());
        Message<HarmMaterialImTemplet> message = Message.init(templet) // 传数据
                .setBusinessTypeCode(IM_CODE) // 业务编码
                .setCompanyCode(data.getPurCompanySrmCode()) // 发送消息企业SRM编码
                .setMsgLevel(MessageLevelEnum.HIGH) // 消息等级
                .setSenderId(loginUser.getId()) // 发送人userId
                .addReceiverId(supUser.getId()); // 接收人userId
        MessageSendUtils.sendMessage(message);
    }

    /**
     * 发送消息给采购商
     */
    public void sendMessagesToPur(BadReport data, HarmMaterialImTemplet templet){
        User supUser = companyUserRefService.queryMainUserOfCompany(data.getSupCompanyId());
        Message<HarmMaterialImTemplet>  message = Message.init(templet) // 传数据
                .setBusinessTypeCode(IM_CODE) // 业务编码
                .setCompanyCode(data.getSupCompanySrmCode())
                .setMsgLevel(MessageLevelEnum.HIGH)
                .setSenderId(supUser.getId())
                .addReceiverId(data.getPurUserId());
        MessageSendUtils.sendMessage(message);
    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void sendForPur(Project project, Company company, final User loginUser, List<BadReport> badReportList) {

        //校验
        Assert.isNotEmpty(badReportList,"不良报告列表不能为空!");
        List<String> ids=new ArrayList<>();
        for (BadReport badReport : badReportList) {
            Assert.isNotBlank(badReport.getId(),"不良报告ID不能为空!");
            ids.add(badReport.getId());
        }
        BadReportExample badReportExample =new BadReportExample();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andIdIn(ids);

        //已经发送的不能再发送了
        int countByExample = badReportMapper.countByExample(badReportExample);
        if(countByExample>0){
            throw  new CommonException("单据中含有已发送数据，操作失败!");
        }

        badReportExample.clear();
        badReportExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.NO_INT)
                .andIdIn(ids);

        BadReport  badReport=new BadReport();
        badReport.setPurSendTime(new Date());
        badReport.setSendStatus(Constant.YES_INT);

        this.badReportMapper.updateByExampleSelective(badReport,badReportExample);


        //消息推送
        badReportExample.clear();
        badReportExample.createCriteria().andIdIn(ids);
        final   List<BadReport> badReports = badReportMapper.selectByExample(badReportExample);
        if(CollectionUtils.isNotEmpty(badReports)){
            this.defaultThreadPool.execute(new Runnable() {
                @Override
                public void run() {
                	for(BadReport badReport : badReports){
                		final HarmMaterialImTemplet templet = constructImInfo(1, "发送", badReport);
                		sendMessagesToSup(loginUser,templet,badReport);
                	}
                }
            });
        }

    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void create(Project project, Company company, User loginUser, BadReport badReport) {
        Assert.isNotNull(badReport,"单据不能为空!");

        //设置采购商信息
        this.setPurCompanyInfo(project,company, badReport);

        //设置供应商信息
        this.setSupCompanyInfo(badReport);

        //设置不良单据信息
        this.setBadeReportInfo(loginUser, badReport);

    }


    /**
     * 设置不良报告基本信息
     * @param loginUser
     * @param badReport
     */
    private void setBadeReportInfo(User loginUser, BadReport badReport) {

        badReport.setId(null);
        badReport.setIsEnable(Constant.YES_INT);
        badReport.setBadReportNo(generateCodeService.getNextCode("BAD_REPORT_NO"));
        badReport.setPurUserId(loginUser.getId());
        badReport.setCreateTime(new Date());
        badReport.setUpdateTime(new Date());
        badReport.setPurCommitTime(new Date());

        this.badReportMapper.insertSelective(badReport);

        List<BadReportItem> badReportItemList = badReport.getBadReportItemList();
        Assert.isNotEmpty(badReportItemList,"不良报告行数据，不能为空!");

        for (BadReportItem badReportItem : badReportItemList) {
            Assert.isNotNull(badReportItem,"不良报告数据，不能为空!");
            badReportItem.setId(null);
            badReportItem.setBadReportId(badReport.getId());
            this.badReportItemService.addObj(badReportItem);
        }

    }

    /**
     * 设置供应商基本信息
     * @param badReport
     */
    private void setSupCompanyInfo(BadReport badReport) {

        //校验前端传来的供应商基本信息是否有误
        //Assert.isNotBlank(badReport.getSupCompanyId(),"供应商ID不能为空!");
        Assert.isNotBlank(badReport.getSupCompanySrmCode(),"供应商SRM编码不能为空!");

        //Company supCompany = companyService.queryObjById(badReport.getSupCompanyId());
        Company supCompany = companyService.queryCompanyByCode(badReport.getSupCompanySrmCode());
        Assert.isNotNull(supCompany,"公司不存在，操作失败");

        badReport.setSupCompanyId(supCompany.getId());
        badReport.setSupCompanyName(supCompany.getCompanyName());
        badReport.setSupCompanyFullName(supCompany.getCompanyFullName());
        badReport.setSupCompanySrmCode(supCompany.getCompanyCode());
        badReport.setPurCompanySapCode(supCompany.getCompanySapCode());
    }


    /**
     * 设置采购商信息
     * @param project
     * @param company
     * @param badReport
     */
    private void setPurCompanyInfo(Project project, Company company, BadReport badReport) {
        badReport.setProjectId(project.getId());
        badReport.setPurCompanyId(project.getCompanyId());
        badReport.setPurCompanyName(company.getCompanyName());
        badReport.setPurCompanyFullName(company.getCompanyFullName());
        badReport.setPurCompanySrmCode(company.getCompanyCode());
        badReport.setPurCompanySapCode(company.getCompanySapCode());

    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Transactional
    @Override
    public void deleteObjByIds(Project project, Company company, User loginUser, List<BadReport> badReportList) {

        Assert.isNotEmpty(badReportList,"单据不能为空!");
        List<String> ids=new ArrayList<>();
        for (BadReport badReport : badReportList) {
            Assert.isNotBlank(badReport.getId(),"单据ID不能为空!");
            ids.add(badReport.getId());
        }

        //已发送不能删除
        BadReportExample example = new BadReportExample();
        example.createCriteria()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andIdIn(ids);

        int countByExample = badReportMapper.countByExample(example);
        if(countByExample>0){
            throw new CommonException("单据中包含已发送数据，操作失败!");
        }

        //已回签、已审批、已驳回、已作废等数据不能删除
       /* List<Integer> billStatus =new ArrayList<>();
        billStatus.add(BillStatusEnum.STATUS_SIGNED.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_SUCCESS.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_APPROVE_FAIL.getBillStatus());
        billStatus.add(BillStatusEnum.STATUS_ABOLISH.getBillStatus());*/

        example.clear();
        example.createCriteria()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andBillStatusNotEqualTo(BillStatusEnum.STATUS_UN_SIGN.getBillStatus())
                .andIdIn(ids);
        int countByExample2 = badReportMapper.countByExample(example);
        if(countByExample2>0){
            throw new CommonException("单据中包含除了未回签等状态其他数据，操作失败!");
        }

        //未发送，未回签的数据才可以删除
        example.clear();
        example.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andIsEnableEqualTo(Constant.YES_INT)
                .andBillStatusEqualTo(BillStatusEnum.STATUS_UN_SIGN.getBillStatus())
                .andIdIn(ids);

        BadReport badReport =new BadReport();
        badReport.setIsEnable(Constant.NO_INT);
        this.badReportMapper.updateByExampleSelective(badReport,example);

    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Override
    public void addObj(BadReport t) {
        this.badReportMapper.insertSelective(t);
    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.badReportMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Override
    public void modifyObj(BadReport t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.badReportMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="badReport", keyGenerator="redisKeyGenerator")
    @Override
    public BadReport queryObjById(String id) {
        return this.badReportMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="badReport", keyGenerator="redisKeyGenerator")
    @Override
    public List<BadReport> queryAllObjByExample(BadReportExample example) {
        return this.badReportMapper.selectByExample(example);
    }

    @Cacheable(value="badReport", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<BadReport> queryObjByPage(BadReportExample example) {
        PageView<BadReport> pageView = example.getPageView();
        pageView.setQueryResult(this.badReportMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Transactional
    @CacheEvict(value={"badReport"}, allEntries=true)
    @Override
    public void addAll(List<BadReport> arg0) {
        if (CollectionUtils.isEmpty(arg0)) {
            return;
        }
        arg0.stream().forEach(item->{
            if(StringUtils.isBlank(item.getId())){
                item.setId(UUIDGenerator.generateUUID());
            }
            this.badReportMapper.insertSelective(item);
        });
        
    }

    @CacheEvict(value={"badReport"}, allEntries=true)
    @Override
    public void deleteByExample(BadReportExample arg0) {
        Assert.isNotNull(arg0, "参数不能为空");
        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
        this.badReportMapper.deleteByExample(arg0);
        
    }
}