package com.els.base.quality.result.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.codegenerator.service.GenerateCodeService;
import com.els.base.company.entity.Company;
import com.els.base.company.service.CompanyService;
import com.els.base.company.service.CompanyUserRefService;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.project.Project;
import com.els.base.core.entity.user.User;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;
import com.els.base.msg.Message;
import com.els.base.msg.MessageLevelEnum;
import com.els.base.msg.MessageSendUtils;
import com.els.base.quality.harms.im.HarmMaterialImTemplet;
import com.els.base.quality.result.dao.CheckResultMapper;
import com.els.base.quality.result.entity.CheckResult;
import com.els.base.quality.result.entity.CheckResultData;
import com.els.base.quality.result.entity.CheckResultDataExample;
import com.els.base.quality.result.entity.CheckResultExample;
import com.els.base.quality.result.entity.CheckResultItem;
import com.els.base.quality.result.entity.CheckResultItemExample;
import com.els.base.quality.result.service.CheckResultDataService;
import com.els.base.quality.result.service.CheckResultItemService;
import com.els.base.quality.result.service.CheckResultService;
import com.els.base.utils.uuid.UUIDGenerator;

@Service("defaultCheckResultService")
public class CheckResultServiceImpl implements CheckResultService {

    private static final String IM_CODE = "BAD_RESULT_REPORT_MESSAGE";

    @Resource
    protected CheckResultMapper checkResultMapper;
    @Resource
    private CompanyService companyService;
    @Resource
    private GenerateCodeService generateCodeService;
    @Resource
    private CheckResultItemService checkResultItemService;
    @Resource
    private CheckResultDataService checkResultDataService;
    @Resource
    private CompanyUserRefService companyUserRefService;
    @Resource
    private ThreadPoolTaskExecutor defaultThreadPool;

    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Transactional
    @Override
    public void deleteObjByIds(Project project, Company company, User loginUser, List<CheckResult> checkResultList) {

        Assert.isNotEmpty(checkResultList,"单据不能为空!");
        List<String> ids=new ArrayList<>();
        for (CheckResult checkResult : checkResultList) {
            Assert.isNotBlank(checkResult.getId(),"单据ID不能为空!");
            ids.add(checkResult.getId());
        }

        //已发送不能删除
        CheckResultExample example = new CheckResultExample();
        example.createCriteria()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andIdIn(ids);

        int countByExample = checkResultMapper.countByExample(example);
        if(countByExample>0){
            throw new CommonException("单据中包含已发送数据，操作失败!");
        }

        CheckResult checkResult =new CheckResult();
        checkResult.setIsEnable(Constant.NO_INT);

        example.clear();
        example.createCriteria()
                .andIsEnableEqualTo(Constant.YES_INT)
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andIdIn(ids);
        this.checkResultMapper.updateByExampleSelective(checkResult,example);

    }

    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Transactional
    @Override
    public void sendForPur(Project project, Company company, final User loginUser, List<CheckResult> checkResultList) {

        //校验
        Assert.isNotEmpty(checkResultList,"检查结果列表不能为空!");
        List<String> ids=new ArrayList<>();
        for (CheckResult checkResult : checkResultList) {
            Assert.isNotBlank(checkResult.getId(),"检查结果报告ID不能为空!");
            ids.add(checkResult.getId());
        }
        CheckResultExample checkResultExample =new CheckResultExample();
        checkResultExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.YES_INT)
                .andIdIn(ids);

        //已经发送的不能再发送了
        int countByExample = checkResultMapper.countByExample(checkResultExample);
        if(countByExample>0){
            throw  new CommonException("单据中含有已发送数据，操作失败!");
        }

        checkResultExample.clear();
        checkResultExample.createCriteria()
                .andProjectIdEqualTo(project.getId())
                .andPurCompanyIdEqualTo(project.getCompanyId())
                .andSendStatusEqualTo(Constant.NO_INT)
                .andIdIn(ids);

        CheckResult  checkResult=new CheckResult();
        checkResult.setPurSendTime(new Date());
        checkResult.setSendStatus(Constant.YES_INT);

        this.checkResultMapper.updateByExampleSelective(checkResult,checkResultExample);


        checkResultExample.clear();
        checkResultExample.createCriteria()
                .andIdIn(ids);
        final   List<CheckResult> checkResults = checkResultMapper.selectByExample(checkResultExample);
        if(CollectionUtils.isNotEmpty(checkResults)){
            this.defaultThreadPool.execute(new Runnable() {
                @Override
                public void run() {
                	for(CheckResult checkResult : checkResults){
                		final HarmMaterialImTemplet templet = constructImInfo(1, "发送", checkResult);
                		sendMessagesToSup(loginUser,templet,checkResult);
                	}
                }
            });
        }

    }

    private HarmMaterialImTemplet constructImInfo(int role,String operate,CheckResult list){
        Assert.isNotNull(list, "传输数据为空");
    	HarmMaterialImTemplet tem = new HarmMaterialImTemplet();
        //设置角色信息:1是为采购商,其他为供应商
        if(role == 1){
            tem.setRole("采购商");
            //设置采购商的名称
            if(StringUtils.isNotEmpty(list.getPurCompanyName())){
                tem.setCompanyName(list.getPurCompanyName());
            }else{
                tem.setCompanyName("");
            }
        }else{
            tem.setRole("供应商");
            //设置供应商的名称
            if(StringUtils.isNotEmpty(list.getSupCompanyName())){
                tem.setCompanyName(list.getSupCompanyName());
            }else{
                tem.setCompanyName("");
            }
        }
        //设置操作
        if(StringUtils.isEmpty(operate)){
            throw new CommonException("没有传入操作字段,无法生成消息模板!");
        }
        tem.setOperate(operate);
        //设置其他信息
        tem.setOtherInfo("了新的检查结果兼处理报告单据,单据号包括：");

        //单据号
        /*List<String > noList = new ArrayList<>();
        for (CheckResult checkResult : list) {
            noList.add(checkResult.getResultReportNo());
        }
        String  billData= StringUtils.join(noList,",");*/
        tem.setBillList(list.getResultReportNo());
        return tem;
    }


    /**
     * 采购商发送消息给供应商
     */
    public void sendMessagesToSup(com.els.base.core.entity.user.User loginUser, HarmMaterialImTemplet templet, CheckResult data){
        User supUser = companyUserRefService.queryMainUserOfCompany(data.getSupCompanyId());
        Message<HarmMaterialImTemplet> message = Message.init(templet) // 传数据
                .setBusinessTypeCode(IM_CODE) // 业务编码
                .setCompanyCode(data.getPurCompanySrmCode()) // 发送消息企业SRM编码
                .setMsgLevel(MessageLevelEnum.HIGH) // 消息等级
                .setSenderId(loginUser.getId()) // 发送人userId
                .addReceiverId(supUser.getId()); // 接收人userId
        MessageSendUtils.sendMessage(message);
    }

    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Transactional
    @Override
    public void edit(Project project, Company company, User loginUser, CheckResult checkResult) {

        //校验信息
        Assert.isNotBlank(checkResult.getId(),"检查结果兼处理报告ID不能为空,操作失败!");
        CheckResult checkResultOld = checkResultMapper.selectByPrimaryKey(checkResult.getId());
        Assert.isNotNull(checkResultOld,"检查结果兼处理报告不存在,操作失败!");

        //更新检查结果信息
        this.updateCheckResultForPur(checkResult,checkResultOld);

        //更新检查结果行信息
        this.updateCheckResultItemInfoForPur(checkResult,checkResultOld);

        //更新尺寸测量数据
        this.updateCheckResultDataForPur(checkResult,checkResultOld);
    }

    /**
     * 更新尺寸测量数据
     * @param checkResult
     * @param checkResultOld
     */
    private void updateCheckResultDataForPur(CheckResult checkResult, CheckResult checkResultOld) {
        List<CheckResultData> checkResultDataList = checkResult.getCheckResultDataList();
        Assert.isNotEmpty(checkResultDataList,"尺寸测量数据列表不能为空！");
        Assert.isNotBlank(checkResult.getId(),"检查结果ID不能为空！");

        CheckResultDataExample checkResultDataExample =new CheckResultDataExample();
        checkResultDataExample.createCriteria().andQualityResultIdEqualTo(checkResult.getId());
        this.checkResultDataService.deleteByExample(checkResultDataExample);

        for (CheckResultData checkResultData : checkResultDataList) {
            checkResultData.setId(null);
            checkResultData.setQualityResultId(checkResultOld.getId());
            this.checkResultDataService.addObj(checkResultData);
        }
    }

    /**
     * 更新检查结果行信息
     * @param checkResult
     * @param checkResultOld
     */
    private void updateCheckResultItemInfoForPur(CheckResult checkResult, CheckResult checkResultOld) {
        List<CheckResultItem> checkResultItemList = checkResult.getCheckResultItemList();
        Assert.isNotEmpty(checkResultItemList,"检查结果列表不能为空！");
        Assert.isNotBlank(checkResult.getId(),"检查结果ID不能为空！");

        //删除之前的
        CheckResultItemExample checkResultItemExample = new CheckResultItemExample();
        checkResultItemExample.createCriteria().andQualityResultIdEqualTo(checkResult.getId());
        this.checkResultItemService.deleteByExample(checkResultItemExample);

        //重新插入
        for (CheckResultItem checkResultItem : checkResultItemList) {
            checkResultItem.setId(null);
            checkResultItem.setQualityResultId(checkResultOld.getId());
            this.checkResultItemService.addObj(checkResultItem);
        }
    }

    /**
     * 更新检查结果信息
     * @param checkResult
     * @param checkResultOld
     */
    private void updateCheckResultForPur(CheckResult checkResult, CheckResult checkResultOld) {

        //物料编码
        //物料名称
        //物料描述
        checkResultOld.setPurMaterialCode(checkResult.getPurMaterialCode());
        checkResultOld.setPurMaterialName(checkResult.getPurMaterialName());
        checkResultOld.setPurMaterialDesc(checkResult.getPurMaterialDesc());

        //抽样方案
        checkResultOld.setSampling(checkResult.getSampling());

        //适用机种
        checkResultOld.setAvailableMachine(checkResult.getAvailableMachine());

        //来料日期
        checkResultOld.setIncomeMaterialTime(checkResult.getIncomeMaterialTime());

        //检验批量
        checkResultOld.setInspectionLot(checkResult.getInspectionLot());

        //LOT NO
        checkResultOld.setLotNo(checkResult.getLotNo());
        checkResultOld.setLotDesc(checkResult.getLotDesc());
        checkResultOld.setLotQuantity(checkResult.getLotQuantity());

        //检查日期
        checkResultOld.setCheckTime(checkResult.getCheckTime());

        //检查基准
        checkResultOld.setCheckStandard(checkResult.getCheckStandard());

        //检查目的
        checkResultOld.setCheckPurpose(checkResult.getCheckPurpose());

        //供应商确认时间

        //资料提交时间

        //创建时间

        //备注
        checkResultOld.setPurRemark(checkResult.getPurRemark());

        //不良内容描述DefectDescription(100字内)
        checkResultOld.setDefectDesc(checkResult.getDefectDesc());
        //处理及不合格评审Handling(100字内)
        checkResultOld.setHandleWay(checkResult.getHandleWay());

        //处理人

        //确认人
        checkResultOld.setConfirmUserId(checkResult.getConfirmUserId());
        checkResultOld.setConfirmUserName(checkResult.getConfirmUserName());
        //承认人
        checkResultOld.setRecognizeUserId(checkResult.getRecognizeUserId());
        checkResultOld.setRecognizeUserName(checkResult.getRecognizeUserName());
        //生技人
        checkResultOld.setBioTechUserId(checkResult.getBioTechUserId());
        checkResultOld.setBioTechUserName(checkResult.getBioTechUserName());
        //质量管理人
        checkResultOld.setQualityTechUserId(checkResult.getQualityTechUserId());
        checkResultOld.setQualityTechUserName(checkResult.getQualityTechUserName());

        checkResultMapper.updateByPrimaryKeySelective(checkResultOld);
    }

    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Transactional
    @Override
    public void create(Project project, Company company, User loginUser, CheckResult checkResult) {

        Assert.isNotNull(checkResult,"单据不能为空!");

        //设置采购商信息
        this.setPurCompanyInfo(project,company, checkResult);

        //设置供应商信息
        this.setSupCompanyInfo(checkResult);

        //设置检查结果单据信息
        this.setCheckResultInfo(loginUser, checkResult);

    }


    /**
     * 设置检查结果信息
     * @param loginUser
     * @param checkResult
     */
    private void setCheckResultInfo(User loginUser, CheckResult checkResult) {

        checkResult.setId(null);
        checkResult.setIsEnable(Constant.YES_INT);
        checkResult.setResultReportNo(generateCodeService.getNextCode("RESULT_REPORT_NO"));
        checkResult.setPurUserId(loginUser.getId());
        checkResult.setCreateTime(new Date());
        checkResult.setUpdateTime(new Date());
        checkResult.setPurUserId(loginUser.getId());
        checkResult.setPurUserName(loginUser.getNickName());
        this.checkResultMapper.insertSelective(checkResult);


        //设置行数据
        List<CheckResultItem> checkResultItemList = checkResult.getCheckResultItemList();
        Assert.isNotEmpty(checkResultItemList,"检查结果报告行数据，不能为空!");
        for (CheckResultItem checkResultItem : checkResultItemList) {
            Assert.isNotNull(checkResultItem,"检查结果报告数据，不能为空!");
            checkResultItem.setId(null);
            checkResultItem.setQualityResultId(checkResult.getId());
            this.checkResultItemService.addObj(checkResultItem);
        }


        //设置测量数据
        List<CheckResultData> checkResultDataList = checkResult.getCheckResultDataList();
        Assert.isNotEmpty(checkResultDataList,"尺寸测量数据列表，不能为空!");
        for (CheckResultData checkResultData : checkResultDataList) {
            Assert.isNotNull(checkResultData,"尺寸测量数据，不能为空!");
            checkResultData.setId(null);
            checkResultData.setQualityResultId(checkResult.getId());
            this.checkResultDataService.addObj(checkResultData);
        }


    }


    /**
     * 设置供应商信息
     * @param checkResult
     */
    private void setSupCompanyInfo(CheckResult checkResult) {

        //校验前端传来的供应商基本信息是否有误
        Assert.isNotBlank(checkResult.getSupCompanySrmCode(),"供应商SRM编码不能为空!");
        Company supCompany = companyService.queryCompanyByCode(checkResult.getSupCompanySrmCode());
        Assert.isNotNull(supCompany,"公司不存在，操作失败");

        checkResult.setSupCompanyId(supCompany.getId());
        checkResult.setSupCompanyName(supCompany.getCompanyName());
        checkResult.setSupCompanyFullName(supCompany.getCompanyFullName());
        checkResult.setPurCompanySapCode(supCompany.getCompanySapCode());
        checkResult.setSupCompanySrmCode(supCompany.getCompanyCode());
    }

    /**
     * 设置采购商信息
     * @param project
     * @param company
     * @param checkResult
     */
    private void setPurCompanyInfo(Project project, Company company, CheckResult checkResult) {
        checkResult.setProjectId(project.getId());
        checkResult.setPurCompanyId(project.getCompanyId());
        checkResult.setPurCompanyName(company.getCompanyName());
        checkResult.setPurCompanySrmCode(company.getCompanyCode());
        checkResult.setPurCompanyFullName(company.getCompanyFullName());
        checkResult.setPurCompanySapCode(company.getCompanySapCode());
    }

    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Override
    public void addObj(CheckResult t) {
        this.checkResultMapper.insertSelective(t);
    }

    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.checkResultMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Override
    public void modifyObj(CheckResult t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.checkResultMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="checkResult", keyGenerator="redisKeyGenerator")
    @Override
    public CheckResult queryObjById(String id) {
        return this.checkResultMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="checkResult", keyGenerator="redisKeyGenerator")
    @Override
    public List<CheckResult> queryAllObjByExample(CheckResultExample example) {
        return this.checkResultMapper.selectByExample(example);
    }

    @Cacheable(value="checkResult", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<CheckResult> queryObjByPage(CheckResultExample example) {
        PageView<CheckResult> pageView = example.getPageView();
        pageView.setQueryResult(this.checkResultMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Transactional
    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Override
    public void addAll(List<CheckResult> arg0) {
        if (CollectionUtils.isEmpty(arg0)) {
            return;
        }
        arg0.stream().forEach(item->{
            if(StringUtils.isBlank(item.getId())){
                item.setId(UUIDGenerator.generateUUID());
            }
            this.checkResultMapper.insertSelective(item);
        });
        
    }

    @CacheEvict(value={"checkResult"}, allEntries=true)
    @Override
    public void deleteByExample(CheckResultExample arg0) {
        Assert.isNotNull(arg0, "参数不能为空");
        Assert.isNotEmpty(arg0.getOredCriteria(), "批量删除不能全表删除");
        this.checkResultMapper.deleteByExample(arg0);
        
    }
}